/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2018, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

package ru.rutoken.samples.pkcs11Wrapper;

import com.sun.jna.Memory;
import com.sun.jna.NativeLong;
import com.sun.jna.Pointer;
import com.sun.jna.ptr.NativeLongByReference;
import ru.rutoken.pkcs11jna.CK_ATTRIBUTE;
import ru.rutoken.pkcs11jna.CK_MECHANISM;
import ru.rutoken.pkcs11jna.Pkcs11Constants;
import ru.rutoken.pkcs11jna.RtPkcs11Constants;
import ru.rutoken.samples.Constants;

import java.util.Random;

public class DeriveGOSTR3410_2012_512 {
    private final static int UKM_LENGTH = 8;

    private final static NativeLongByReference hPubKeyR = new NativeLongByReference();
    private final static NativeLongByReference hPrivKeyR = new NativeLongByReference();

    private final static CK_ATTRIBUTE[] publicKeyTmpl = (CK_ATTRIBUTE[]) (new CK_ATTRIBUTE()).toArray(8);
    private final static CK_ATTRIBUTE[] privateKeyTmpl = (CK_ATTRIBUTE[]) (new CK_ATTRIBUTE()).toArray(9);

    private static void initPubKeyTemplate() {
        publicKeyTmpl[0].setAttr(new NativeLong(Pkcs11Constants.CKA_CLASS), new NativeLong(Pkcs11Constants.CKO_PUBLIC_KEY));
        publicKeyTmpl[1].setAttr(new NativeLong(Pkcs11Constants.CKA_LABEL), "Sample GOST R 34.10-2012 512 Public Key (Aktiv Co.)");
        publicKeyTmpl[2].setAttr(new NativeLong(Pkcs11Constants.CKA_ID), "Sample GOST R 34.10-2012 512 Keypair (Aktiv Co.)");
        publicKeyTmpl[3].setAttr(new NativeLong(Pkcs11Constants.CKA_KEY_TYPE), new NativeLong(RtPkcs11Constants.CKK_GOSTR3410_512));
        publicKeyTmpl[4].setAttr(new NativeLong(Pkcs11Constants.CKA_TOKEN), true);
        publicKeyTmpl[5].setAttr(new NativeLong(Pkcs11Constants.CKA_PRIVATE), false);
        publicKeyTmpl[6].setAttr(new NativeLong(RtPkcs11Constants.CKA_GOSTR3410_PARAMS), Constants.ATTR_CRYPTO_PRO_A_512);
        publicKeyTmpl[7].setAttr(new NativeLong(RtPkcs11Constants.CKA_GOSTR3411_PARAMS), Constants.ATTR_HASH_2012_512);
    }

    private static void initPrivKeyTemplate() {
        privateKeyTmpl[0].setAttr(new NativeLong(Pkcs11Constants.CKA_CLASS), new NativeLong(Pkcs11Constants.CKO_PRIVATE_KEY));
        privateKeyTmpl[1].setAttr(new NativeLong(Pkcs11Constants.CKA_LABEL), "Sample GOST R 34.10-2012 512 Private Key (Aktiv Co.)");
        privateKeyTmpl[2].setAttr(new NativeLong(Pkcs11Constants.CKA_ID), "Sample GOST R 34.10-2012 512 Keypair (Aktiv Co.)");
        privateKeyTmpl[3].setAttr(new NativeLong(Pkcs11Constants.CKA_KEY_TYPE), new NativeLong(RtPkcs11Constants.CKK_GOSTR3410_512));
        privateKeyTmpl[4].setAttr(new NativeLong(Pkcs11Constants.CKA_TOKEN), true);
        privateKeyTmpl[5].setAttr(new NativeLong(Pkcs11Constants.CKA_PRIVATE), true);
        privateKeyTmpl[6].setAttr(new NativeLong(Pkcs11Constants.CKA_DERIVE), true);
        privateKeyTmpl[7].setAttr(new NativeLong(RtPkcs11Constants.CKA_GOSTR3410_PARAMS), Constants.ATTR_CRYPTO_PRO_A_512);
        privateKeyTmpl[8].setAttr(new NativeLong(RtPkcs11Constants.CKA_GOSTR3411_PARAMS), Constants.ATTR_HASH_2012_512);
    }

    public static void main(String[] args) {
        NativeLong hSession = new NativeLong(Pkcs11Constants.CK_INVALID_HANDLE);
        try {
            System.out.println("Example of C_DeriveKey using");

            System.out.println("Library initialization and acquiring of function list");
            NativeLong rv = RtPkcs11Library.getPkcs11Interface().C_Initialize(null);
            Pkcs11Exception.throwIfNotOk("C_Initialize failed", rv);

            initPubKeyTemplate();
            initPrivKeyTemplate();

            System.out.println("Acquiring list of slots with connected tokens");
            NativeLongByReference slotsCount = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11Interface().C_GetSlotList(Pkcs11Constants.CK_TRUE, null, slotsCount);
            Pkcs11Exception.throwIfNotOk("C_GetSlotList failed", rv);
            if (slotsCount.getValue().intValue() == 0) {
                throw new Exception("No Rutoken is available!");
            }

            NativeLong[] pSlotList = new NativeLong[slotsCount.getValue().intValue()];
            rv = RtPkcs11Library.getPkcs11Interface().C_GetSlotList(Pkcs11Constants.CK_TRUE, pSlotList, slotsCount);
            Pkcs11Exception.throwIfNotOk("C_GetSlotList failed", rv);

            System.out.println("Opening session");
            NativeLongByReference phSession = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11Interface().C_OpenSession(pSlotList[0],
                    new NativeLong(Pkcs11Constants.CKF_SERIAL_SESSION | Pkcs11Constants.CKF_RW_SESSION),
                    null, null, phSession);
            Pkcs11Exception.throwIfNotOk("C_OpenSession failed", rv);
            hSession = phSession.getValue();

            System.out.println("Logging in as user");
            rv = RtPkcs11Library.getPkcs11Interface().C_Login(hSession, new NativeLong(Pkcs11Constants.CKU_USER), Constants.DEFAULT_USER_PIN, new NativeLong(Constants.DEFAULT_USER_PIN.length));
            Pkcs11Exception.throwIfNotOk("C_Login failed", rv);

            CK_MECHANISM ckmGenMech = new CK_MECHANISM(new NativeLong(RtPkcs11Constants.CKM_GOSTR3410_512_KEY_PAIR_GEN), Pointer.NULL, new NativeLong(0));

            System.out.println("Generating key pair");
            rv = RtPkcs11Library.getPkcs11Interface().C_GenerateKeyPair(hSession, ckmGenMech, publicKeyTmpl, new NativeLong(publicKeyTmpl.length),
                    privateKeyTmpl, new NativeLong(privateKeyTmpl.length), hPubKeyR, hPrivKeyR);
            Pkcs11Exception.throwIfNotOk("C_GenerateKeyPair failed", rv);

            NativeLong hPrivKey = hPrivKeyR.getValue();

            Random r = new Random();
            byte[] deriveParamUkm = new byte[UKM_LENGTH];
            r.nextBytes(deriveParamUkm);

            Memory deriveParameters = Util.allocateDeriveParamsGOSTR3410_2012((int) RtPkcs11Constants.CKD_CPDIVERSIFY_KDF, Constants.RECIPIENT_KEY_512, deriveParamUkm);

            CK_MECHANISM ckmDeriveMech = new CK_MECHANISM(new NativeLong(RtPkcs11Constants.CKM_GOSTR3410_12_DERIVE), deriveParameters, new NativeLong(deriveParameters.size()));

            CK_ATTRIBUTE[] derivedKeyAttr = (CK_ATTRIBUTE[]) (new CK_ATTRIBUTE()).toArray(8);
            derivedKeyAttr[0].setAttr(new NativeLong(Pkcs11Constants.CKA_LABEL), "Derived GOST 28147-89 key");
            derivedKeyAttr[1].setAttr(new NativeLong(Pkcs11Constants.CKA_CLASS), new NativeLong(Pkcs11Constants.CKO_SECRET_KEY));
            derivedKeyAttr[2].setAttr(new NativeLong(Pkcs11Constants.CKA_KEY_TYPE), new NativeLong(RtPkcs11Constants.CKK_GOST28147));
            derivedKeyAttr[3].setAttr(new NativeLong(Pkcs11Constants.CKA_TOKEN), false);
            derivedKeyAttr[4].setAttr(new NativeLong(Pkcs11Constants.CKA_MODIFIABLE), true);
            derivedKeyAttr[5].setAttr(new NativeLong(Pkcs11Constants.CKA_PRIVATE), true);
            derivedKeyAttr[6].setAttr(new NativeLong(Pkcs11Constants.CKA_EXTRACTABLE), true);
            derivedKeyAttr[7].setAttr(new NativeLong(Pkcs11Constants.CKA_SENSITIVE), false);
            NativeLong derivedKeyAttrLen = new NativeLong(derivedKeyAttr.length);

            NativeLongByReference phKey = new NativeLongByReference();

            System.out.println("Deriving key");
            rv = RtPkcs11Library.getPkcs11Interface().C_DeriveKey(hSession, ckmDeriveMech, hPrivKey, derivedKeyAttr, derivedKeyAttrLen, phKey);
            Pkcs11Exception.throwIfNotOk("C_DeriveKey failed", rv);
        } catch (Exception e) {
            System.out.println(e.getMessage());
        } finally {
            System.out.println("Deleting public key");
            NativeLong rv = RtPkcs11Library.getPkcs11Interface().C_DestroyObject(hSession, hPubKeyR.getValue());
            Util.checkIfNotOk("C_DestroyObject failed", rv.longValue());

            System.out.println("Deleting private key");
            rv = RtPkcs11Library.getPkcs11Interface().C_DestroyObject(hSession, hPrivKeyR.getValue());
            Util.checkIfNotOk("C_DestroyObject failed", rv.longValue());

            System.out.println("Logging out");
            rv = RtPkcs11Library.getPkcs11Interface().C_Logout(hSession);
            Util.checkIfNotOk("C_Logout failed", rv.longValue());

            System.out.println("Closing session");
            rv = RtPkcs11Library.getPkcs11Interface().C_CloseSession(hSession);
            Util.checkIfNotOk("C_CloseSession failed", rv.longValue());

            System.out.println("Finalizing PKCS11 library");
            rv = RtPkcs11Library.getPkcs11Interface().C_Finalize(null);
            Util.checkIfNotOk("C_Finalize failed", rv.longValue());

            System.out.println("Test has been completed.");
        }
    }
}