/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2018, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

package ru.rutoken.samples.pkcs11Wrapper;

import com.sun.jna.Memory;
import com.sun.jna.NativeLong;
import com.sun.jna.Pointer;
import com.sun.jna.ptr.NativeLongByReference;
import ru.rutoken.pkcs11jna.CK_ATTRIBUTE;
import ru.rutoken.pkcs11jna.CK_MECHANISM;
import ru.rutoken.pkcs11jna.Pkcs11Constants;
import ru.rutoken.pkcs11jna.RtPkcs11Constants;
import ru.rutoken.samples.Constants;

import java.util.Base64;

public class PKIExtensionsGOSTR3410_2001 {
    // Шаблон для импорта открытого ключа ГОСТ Р 34.10-2001
    private final static CK_ATTRIBUTE[] publicKeyTmpl = (CK_ATTRIBUTE[]) (new CK_ATTRIBUTE()).toArray(7);
    // Шаблон для импорта закрытого ключа ГОСТ Р 34.10-2001
    private final static CK_ATTRIBUTE[] privateKeyTmpl = (CK_ATTRIBUTE[]) (new CK_ATTRIBUTE()).toArray(8);
    // Шаблон для импорта сертификата ключа подписи
    private final static CK_ATTRIBUTE[] CertTmpl = (CK_ATTRIBUTE[]) (new CK_ATTRIBUTE()).toArray(6);
    // Данные для подписи
    private final static byte[] pbtData = {'1'};
    // Хендлы объектов, которые будут созданы вызовами C_CreateObject
    private final static NativeLongByReference hPubKey = new NativeLongByReference();
    private final static NativeLongByReference hPrvKey = new NativeLongByReference();
    private final static NativeLongByReference hCert = new NativeLongByReference();
    // Указатели на объекты, которые будут созданы вызовами C_CreateObject
    private final static Pointer ppbtCsr = new Memory(Pointer.SIZE);
    private final static Pointer pptSignature = new Memory(Pointer.SIZE);
    private final static Pointer ppbtInfo = new Memory(Pointer.SIZE);

    private static void initPubKeyTemplate() {
        publicKeyTmpl[0].setAttr(new NativeLong(Pkcs11Constants.CKA_CLASS), new NativeLong(Pkcs11Constants.CKO_PUBLIC_KEY));
        publicKeyTmpl[1].setAttr(new NativeLong(Pkcs11Constants.CKA_LABEL), "Sample GOST R 34.10-2001 Public Key (Aktiv Co.)");
        publicKeyTmpl[2].setAttr(new NativeLong(Pkcs11Constants.CKA_ID), "Sample GOST R 34.10-2001 Keypair (Aktiv Co.)");
        publicKeyTmpl[3].setAttr(new NativeLong(Pkcs11Constants.CKA_KEY_TYPE), new NativeLong(RtPkcs11Constants.CKK_GOSTR3410));
        publicKeyTmpl[4].setAttr(new NativeLong(Pkcs11Constants.CKA_TOKEN), true);
        publicKeyTmpl[5].setAttr(new NativeLong(Pkcs11Constants.CKA_PRIVATE), false);
        publicKeyTmpl[6].setAttr(new NativeLong(RtPkcs11Constants.CKA_GOSTR3410_PARAMS), Constants.ATTR_CRYPTO_PRO_A);
    }

    private static void initPrivKeyTemplate() {
        privateKeyTmpl[0].setAttr(new NativeLong(Pkcs11Constants.CKA_CLASS), new NativeLong(Pkcs11Constants.CKO_PRIVATE_KEY));
        privateKeyTmpl[1].setAttr(new NativeLong(Pkcs11Constants.CKA_LABEL), "Sample GOST R 34.10-2001 Private Key (Aktiv Co.)");
        privateKeyTmpl[2].setAttr(new NativeLong(Pkcs11Constants.CKA_ID), "Sample GOST R 34.10-2001 Keypair (Aktiv Co.)");
        privateKeyTmpl[3].setAttr(new NativeLong(Pkcs11Constants.CKA_KEY_TYPE), new NativeLong(RtPkcs11Constants.CKK_GOSTR3410));
        privateKeyTmpl[4].setAttr(new NativeLong(Pkcs11Constants.CKA_TOKEN), true);
        privateKeyTmpl[5].setAttr(new NativeLong(Pkcs11Constants.CKA_PRIVATE), true);
        privateKeyTmpl[6].setAttr(new NativeLong(Pkcs11Constants.CKA_DERIVE), true);
        privateKeyTmpl[7].setAttr(new NativeLong(RtPkcs11Constants.CKA_GOSTR3410_PARAMS), Constants.ATTR_CRYPTO_PRO_A);
    }

    private static void initCertTemplate() {
        CertTmpl[0].setAttr(new NativeLong(Pkcs11Constants.CKA_CLASS), new NativeLong(Pkcs11Constants.CKO_CERTIFICATE));
        CertTmpl[1].setAttr(new NativeLong(Pkcs11Constants.CKA_CERTIFICATE_TYPE), new NativeLong(Pkcs11Constants.CKC_X_509));
        CertTmpl[2].setAttr(new NativeLong(Pkcs11Constants.CKA_ID), "Sample GOST R 34.10-2001 Keypair (Aktiv Co.)");
        CertTmpl[3].setAttr(new NativeLong(Pkcs11Constants.CKA_TOKEN), true);
        CertTmpl[4].setAttr(new NativeLong(Pkcs11Constants.CKA_PRIVATE), false);
    }

    public static void main(String[] args) {
        NativeLong hSession = new NativeLong(Pkcs11Constants.CK_INVALID_HANDLE);

        try {
            System.out.println("Example of rtpkcs11ecp PKI extensions usage via JNA");

            System.out.println("Library initialization and acquiring of function list");
            NativeLong rv = RtPkcs11Library.getPkcs11Interface().C_Initialize(null);
            Pkcs11Exception.throwIfNotOk("C_Initialize failed", rv);

            initPubKeyTemplate();
            initPrivKeyTemplate();
            initCertTemplate();

            ppbtCsr.setPointer(0, null);
            pptSignature.setPointer(0, null);
            ppbtInfo.setPointer(0, null);

            System.out.println("Acquiring list of slots with connected tokens");
            NativeLongByReference slotsCount = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11Interface().C_GetSlotList(Pkcs11Constants.CK_TRUE, null, slotsCount);
            Pkcs11Exception.throwIfNotOk("C_GetSlotList failed", rv);

            if (slotsCount.getValue().intValue() == 0)
                throw new Exception("No Rutoken is available!");

            NativeLong[] pSlotList = new NativeLong[slotsCount.getValue().intValue()];
            rv = RtPkcs11Library.getPkcs11Interface().C_GetSlotList(Pkcs11Constants.CK_TRUE, pSlotList, slotsCount);
            Pkcs11Exception.throwIfNotOk("C_GetSlotList failed", rv);

            System.out.println("Opening session");
            NativeLongByReference phSession = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11Interface().C_OpenSession(pSlotList[0],
                    new NativeLong(Pkcs11Constants.CKF_SERIAL_SESSION | Pkcs11Constants.CKF_RW_SESSION),
                    null, null, phSession);
            Pkcs11Exception.throwIfNotOk("C_OpenSession failed", rv);

            hSession = phSession.getValue();
            System.out.println("Logging in as user");
            rv = RtPkcs11Library.getPkcs11Interface().C_Login(hSession, new NativeLong(Pkcs11Constants.CKU_USER), Constants.DEFAULT_USER_PIN, new NativeLong(Constants.DEFAULT_USER_PIN.length));
            Pkcs11Exception.throwIfNotOk("C_Login failed", rv);

            System.out.println("Generate key pair");
            rv = RtPkcs11Library.getPkcs11Interface().C_GenerateKeyPair(hSession, new CK_MECHANISM(new NativeLong(RtPkcs11Constants.CKM_GOSTR3410_KEY_PAIR_GEN),
                            null, new NativeLong(0)), publicKeyTmpl, new NativeLong(publicKeyTmpl.length), privateKeyTmpl,
                    new NativeLong(privateKeyTmpl.length), hPubKey, hPrvKey);
            Pkcs11Exception.throwIfNotOk("C_GenerateKeyPair failed", rv);

            System.out.println("Creating certificate request");
            NativeLongByReference ulCsrSize = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11ExtendedInterface().C_EX_CreateCSR(hSession, hPubKey.getValue(), Constants.DN, new NativeLong(Constants.DN.length),
                    ppbtCsr, ulCsrSize, hPrvKey.getValue(), null, new NativeLong(0),
                    Constants.EXTS, new NativeLong(Constants.EXTS.length));
            Pkcs11Exception.throwIfNotOk("C_EX_CreateCSR failed", rv);

            System.out.println("Certificate request is: ");
            Pointer pbtCsr = ppbtCsr.getPointer(0);
            Util.printCsr(pbtCsr.getByteArray(0, ulCsrSize.getValue().intValue()));

            System.out.println("\nEnter certificate in base64 format:");
            String cert = Util.readCert();
            CertTmpl[5].setAttr(new NativeLong(Pkcs11Constants.CKA_VALUE), Base64.getDecoder().decode(cert));

            System.out.println("\nCreating certificate");
            rv = RtPkcs11Library.getPkcs11Interface().C_CreateObject(hSession, CertTmpl, new NativeLong(CertTmpl.length), hCert);
            Pkcs11Exception.throwIfNotOk("C_CreateObject failed", rv);

            System.out.println("Acquiring info about certificate");
            NativeLongByReference ulInfoSize = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11ExtendedInterface().C_EX_GetCertificateInfoText(hSession, hCert.getValue(), ppbtInfo, ulInfoSize);
            Pkcs11Exception.throwIfNotOk("C_EX_GetCertificateInfoText failed", rv);

            Pointer pbtInfo = ppbtInfo.getPointer(0);
            System.out.println("Certificate is: ");
            for (int i = 0; i < ulInfoSize.getValue().intValue(); ++i) {
                System.out.printf(" %02X", pbtInfo.getByteArray(0, ulInfoSize.getValue().intValue())[i]);
                if ((i + 1) % 8 == 0)
                    System.out.println();
            }

            System.out.println("\nPKCS7 signing");
            NativeLongByReference ulSignatureLen = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11ExtendedInterface().C_EX_PKCS7Sign(hSession, pbtData, new NativeLong(pbtData.length), hCert.getValue(),
                    pptSignature, ulSignatureLen, hPrvKey.getValue(),
                    null, new NativeLong(0), new NativeLong(0));
            Pkcs11Exception.throwIfNotOk("C_EX_PKCS7Sign failed", rv);

            Pointer pbtSignature = pptSignature.getPointer(0);

            System.out.println("Signed data is:");
            for (int i = 0; i < ulSignatureLen.getValue().intValue(); ++i) {
                System.out.printf(" %02X", pbtSignature.getByteArray(0, ulSignatureLen.getValue().intValue())[i]);
                if ((i + 1) % 8 == 0)
                    System.out.println();
            }
            System.out.println("\nData has been signed successfully.");
        } catch (Exception e) {
            System.out.println(e.getMessage());
        } finally {
            NativeLong rv = RtPkcs11Library.getPkcs11ExtendedInterface().C_EX_FreeBuffer(ppbtCsr.getPointer(0));
            Util.checkIfNotOk("C_EX_FreeBuffer failed", rv.longValue());

            rv = RtPkcs11Library.getPkcs11ExtendedInterface().C_EX_FreeBuffer(pptSignature.getPointer(0));
            Util.checkIfNotOk("C_EX_FreeBuffer failed", rv.longValue());

            rv = RtPkcs11Library.getPkcs11ExtendedInterface().C_EX_FreeBuffer(ppbtInfo.getPointer(0));
            Util.checkIfNotOk("C_EX_FreeBuffer failed", rv.longValue());

            System.out.println("Deleting public key");
            rv = RtPkcs11Library.getPkcs11Interface().C_DestroyObject(hSession, hPubKey.getValue());
            Util.checkIfNotOk("C_DestroyObject failed", rv.longValue());

            System.out.println("Deleting private key");
            rv = RtPkcs11Library.getPkcs11Interface().C_DestroyObject(hSession, hPrvKey.getValue());
            Util.checkIfNotOk("C_DestroyObject failed", rv.longValue());

            System.out.println("Deleting certificate");
            rv = RtPkcs11Library.getPkcs11Interface().C_DestroyObject(hSession, hCert.getValue());
            Util.checkIfNotOk("C_DestroyObject failed", rv.longValue());

            System.out.println("Logging out");
            rv = RtPkcs11Library.getPkcs11Interface().C_Logout(hSession);
            Util.checkIfNotOk("C_Logout failed", rv.longValue());

            System.out.println("Closing session");
            rv = RtPkcs11Library.getPkcs11Interface().C_CloseSession(hSession);
            Util.checkIfNotOk("C_CloseSession failed", rv.longValue());

            System.out.println("Finalizing PKCS11 library");
            rv = RtPkcs11Library.getPkcs11Interface().C_Finalize(null);
            Util.checkIfNotOk("C_Finalize failed", rv.longValue());

            System.out.println("Test has been completed.");
        }
    }
}
