/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2018, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

package ru.rutoken.samples.pkcs11Wrapper;

import com.sun.jna.Memory;
import com.sun.jna.NativeLong;
import com.sun.jna.Pointer;
import com.sun.jna.ptr.NativeLongByReference;
import ru.rutoken.pkcs11jna.CK_ATTRIBUTE;
import ru.rutoken.pkcs11jna.CK_MECHANISM;
import ru.rutoken.pkcs11jna.Pkcs11Constants;
import ru.rutoken.samples.Constants;

import java.util.Base64;

public class PKIExtensionsRSA {
    // Шаблон для импорта сертификата ключа подписи
    private final static CK_ATTRIBUTE[] CertTmpl = (CK_ATTRIBUTE[]) (new CK_ATTRIBUTE()).toArray(6);
    // Данные для шифрования
    private final static byte[] pbtEncDecData = {'1', '2', '3', '4', '5', '6', '7', '8', 'a', 'b', 'c', 'd'};
    // Данные для подписи
    private final static byte[] pbtData = {'1'};
    // Хендлы объектов, которые будут созданы вызовами C_CreateObject
    private final static NativeLongByReference hPubKey = new NativeLongByReference();
    private final static NativeLongByReference hPrvKey = new NativeLongByReference();
    private final static NativeLongByReference hCert = new NativeLongByReference();
    // Указатели на объекты, которые будут созданы вызовами C_CreateObject
    private final static Pointer ppbtCsr = new Memory(Pointer.SIZE);
    private final static Pointer ppbtInfo = new Memory(Pointer.SIZE);
    // Шаблон для генерации закрытого ключа RSA
    private static CK_ATTRIBUTE[] privRsaKeyTmpl = (CK_ATTRIBUTE[]) (new CK_ATTRIBUTE()).toArray(7);
    // Шаблон для генерации открытого ключа RSA
    private static CK_ATTRIBUTE[] pubRsaKeyTmpl = (CK_ATTRIBUTE[]) (new CK_ATTRIBUTE()).toArray(8);

    private static void initPrivRsaKeyTmpl() {
        privRsaKeyTmpl[0].setAttr(new NativeLong(Pkcs11Constants.CKA_CLASS), new NativeLong(Pkcs11Constants.CKO_PRIVATE_KEY));
        privRsaKeyTmpl[1].setAttr(new NativeLong(Pkcs11Constants.CKA_LABEL), "Sample RSA Private Key (Aktiv Co.)");
        privRsaKeyTmpl[2].setAttr(new NativeLong(Pkcs11Constants.CKA_ID), "Sample RSA Keypair ID (Aktiv Co.)");
        privRsaKeyTmpl[3].setAttr(new NativeLong(Pkcs11Constants.CKA_KEY_TYPE), new NativeLong(Pkcs11Constants.CKK_RSA));
        privRsaKeyTmpl[4].setAttr(new NativeLong(Pkcs11Constants.CKA_DECRYPT), true);
        privRsaKeyTmpl[5].setAttr(new NativeLong(Pkcs11Constants.CKA_TOKEN), true);
        privRsaKeyTmpl[6].setAttr(new NativeLong(Pkcs11Constants.CKA_PRIVATE), true);
    }

    private static void initPubRsaKeyTmpl() {
        pubRsaKeyTmpl[0].setAttr(new NativeLong(Pkcs11Constants.CKA_CLASS), new NativeLong(Pkcs11Constants.CKO_PUBLIC_KEY));
        pubRsaKeyTmpl[1].setAttr(new NativeLong(Pkcs11Constants.CKA_LABEL), "Sample RSA Public Key (Aktiv Co.)");
        pubRsaKeyTmpl[2].setAttr(new NativeLong(Pkcs11Constants.CKA_ID), "Sample RSA Keypair ID (Aktiv Co.)");
        pubRsaKeyTmpl[3].setAttr(new NativeLong(Pkcs11Constants.CKA_KEY_TYPE), new NativeLong(Pkcs11Constants.CKK_RSA));
        pubRsaKeyTmpl[4].setAttr(new NativeLong(Pkcs11Constants.CKA_ENCRYPT), true);
        pubRsaKeyTmpl[5].setAttr(new NativeLong(Pkcs11Constants.CKA_TOKEN), true);
        pubRsaKeyTmpl[6].setAttr(new NativeLong(Pkcs11Constants.CKA_PRIVATE), false);
        pubRsaKeyTmpl[7].setAttr(new NativeLong(Pkcs11Constants.CKA_MODULUS_BITS), new NativeLong(1024));
    }

    private static void initCertTemplate() {
        CertTmpl[0].setAttr(new NativeLong(Pkcs11Constants.CKA_CLASS), new NativeLong(Pkcs11Constants.CKO_CERTIFICATE));
        CertTmpl[1].setAttr(new NativeLong(Pkcs11Constants.CKA_CERTIFICATE_TYPE), new NativeLong(Pkcs11Constants.CKC_X_509));
        CertTmpl[2].setAttr(new NativeLong(Pkcs11Constants.CKA_ID), "Sample GOST R 34.10-2012 (256 bits) Keypair (Aktiv Co.)");
        CertTmpl[3].setAttr(new NativeLong(Pkcs11Constants.CKA_TOKEN), true);
        CertTmpl[4].setAttr(new NativeLong(Pkcs11Constants.CKA_PRIVATE), false);
    }

    public static void main(String[] args) {
        NativeLong hSession = new NativeLong(Pkcs11Constants.CK_INVALID_HANDLE);

        try {
            System.out.println("Example of rtpkcs11ecp PKI extensions usage via JNA");

            System.out.println("Library initialization and acquiring of function list");
            NativeLong rv = RtPkcs11Library.getPkcs11Interface().C_Initialize(null);
            Pkcs11Exception.throwIfNotOk("C_Initialize failed", rv);

            initPrivRsaKeyTmpl();
            initPubRsaKeyTmpl();
            initCertTemplate();

            ppbtCsr.setPointer(0, null);
            ppbtInfo.setPointer(0, null);

            System.out.println("Acquiring list of slots with connected tokens");
            NativeLongByReference slotsCount = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11Interface().C_GetSlotList(Pkcs11Constants.CK_TRUE, null, slotsCount);
            Pkcs11Exception.throwIfNotOk("C_GetSlotList failed", rv);

            if (slotsCount.getValue().intValue() == 0)
                throw new Exception("No Rutoken is available!");

            NativeLong[] pSlotList = new NativeLong[slotsCount.getValue().intValue()];
            rv = RtPkcs11Library.getPkcs11Interface().C_GetSlotList(Pkcs11Constants.CK_TRUE, pSlotList, slotsCount);
            Pkcs11Exception.throwIfNotOk("C_GetSlotList failed", rv);

            System.out.println("Opening session");
            NativeLongByReference phSession = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11Interface().C_OpenSession(pSlotList[0],
                    new NativeLong(Pkcs11Constants.CKF_SERIAL_SESSION | Pkcs11Constants.CKF_RW_SESSION),
                    null, null, phSession);
            Pkcs11Exception.throwIfNotOk("C_OpenSession failed", rv);

            hSession = phSession.getValue();

            System.out.println("Logging in as user");
            rv = RtPkcs11Library.getPkcs11Interface().C_Login(hSession, new NativeLong(Pkcs11Constants.CKU_USER), Constants.DEFAULT_USER_PIN, new NativeLong(Constants.DEFAULT_USER_PIN.length));
            Pkcs11Exception.throwIfNotOk("C_Login failed", rv);

            System.out.println("Generate key pair");
            rv = RtPkcs11Library.getPkcs11Interface().C_GenerateKeyPair(hSession, new CK_MECHANISM(new NativeLong(Pkcs11Constants.CKM_RSA_PKCS_KEY_PAIR_GEN),
                            null, new NativeLong(0)), pubRsaKeyTmpl, new NativeLong(pubRsaKeyTmpl.length), privRsaKeyTmpl,
                    new NativeLong(privRsaKeyTmpl.length), hPubKey, hPrvKey);
            Pkcs11Exception.throwIfNotOk("C_GenerateKeyPair failed", rv);


            System.out.println("Creating certificate request");
            NativeLongByReference ulCsrSize = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11ExtendedInterface().C_EX_CreateCSR(hSession, hPubKey.getValue(), Constants.DN, new NativeLong(Constants.DN.length),
                    ppbtCsr, ulCsrSize, hPrvKey.getValue(), null, new NativeLong(0),
                    Constants.EXTS, new NativeLong(Constants.EXTS.length));
            Pkcs11Exception.throwIfNotOk("C_EX_CreateCSR failed", rv);

            System.out.println("Certificate request is: ");
            Pointer pbtCsr = ppbtCsr.getPointer(0);
            Util.printCsr(pbtCsr.getByteArray(0, ulCsrSize.getValue().intValue()));

            System.out.println("\nEnter certificate in base64 format:");
            String cert = Util.readCert();
            CertTmpl[5].setAttr(new NativeLong(Pkcs11Constants.CKA_VALUE), Base64.getDecoder().decode(cert));

            System.out.println("\nCreating certificate");
            rv = RtPkcs11Library.getPkcs11Interface().C_CreateObject(hSession, CertTmpl, new NativeLong(CertTmpl.length), hCert);
            Pkcs11Exception.throwIfNotOk("C_CreateObject failed", rv);

            System.out.println("Acquiring info about certificate");
            NativeLongByReference ulInfoSize = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11ExtendedInterface().C_EX_GetCertificateInfoText(hSession, hCert.getValue(), ppbtInfo, ulInfoSize);
            Pkcs11Exception.throwIfNotOk("C_EX_GetCertificateInfoText failed", rv);

            Pointer pbtInfo = ppbtInfo.getPointer(0);
            System.out.println("Certificate is: ");
            for (int i = 0; i < ulInfoSize.getValue().intValue(); ++i) {
                System.out.printf(" %02X", pbtInfo.getByteArray(0, ulInfoSize.getValue().intValue())[i]);
                if ((i + 1) % 16 == 0)
                    System.out.println();
            }

            System.out.println("\nInitialize hash function");
            rv = RtPkcs11Library.getPkcs11Interface().C_DigestInit(hSession, new CK_MECHANISM(new NativeLong(Pkcs11Constants.CKM_SHA_1), null, new NativeLong(0)));
            Pkcs11Exception.throwIfNotOk("C_DigestInit failed", rv);

            System.out.println("Get hash code size");
            NativeLongByReference hashSize = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11Interface().C_Digest(hSession, pbtData, new NativeLong(pbtData.length), null, hashSize);
            Pkcs11Exception.throwIfNotOk("C_Digest failed", rv);

            System.out.println("Get hash code");
            byte[] hash = new byte[hashSize.getValue().intValue()];
            rv = RtPkcs11Library.getPkcs11Interface().C_Digest(hSession, pbtData, new NativeLong(pbtData.length), hash, hashSize);
            Pkcs11Exception.throwIfNotOk("C_Digest failed", rv);

            System.out.println("Hash buffer is:");
            System.out.print("    ");
            for (int i = 0; i < hash.length; ++i) {
                if ((i + 1) % 16 == 0)
                    System.out.printf("0x%02X\n    ", hash[i]);
                else
                    System.out.printf("0x%02X ", hash[i]);
            }
            System.out.println("\nHashing has been completed.");

            System.out.println("\nInitialize sign function");
            rv = RtPkcs11Library.getPkcs11Interface().C_SignInit(hSession, new CK_MECHANISM(new NativeLong(Pkcs11Constants.CKM_RSA_PKCS), null, new NativeLong(0)), hPrvKey.getValue());
            Pkcs11Exception.throwIfNotOk("C_SignInit failed", rv);

            System.out.println("Get size for signature");
            NativeLongByReference signatureSize = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11Interface().C_Sign(hSession, hash, new NativeLong(hash.length), null, signatureSize);
            Pkcs11Exception.throwIfNotOk("C_Sign failed", rv);

            System.out.println("Sign data");
            byte[] signature = new byte[signatureSize.getValue().intValue()];
            rv = RtPkcs11Library.getPkcs11Interface().C_Sign(hSession, hash, new NativeLong(hash.length), signature, signatureSize);
            Pkcs11Exception.throwIfNotOk("C_Sign failed", rv);

            System.out.println("Signature buffer is:\n");
            System.out.print("    ");
            for (int i = 0; i < signature.length; ++i) {
                if ((i + 1) % 16 == 0)
                    System.out.printf("0x%02X\n    ", signature[i]);
                else
                    System.out.printf("0x%02X ", signature[i]);
            }
            System.out.println("\nData has been signed successfully.");

            System.out.println("\nInitialize verify function");
            rv = RtPkcs11Library.getPkcs11Interface().C_VerifyInit(hSession, new CK_MECHANISM(new NativeLong(Pkcs11Constants.CKM_RSA_PKCS), null, new NativeLong(0)), hPubKey.getValue());
            Pkcs11Exception.throwIfNotOk("C_VerifyInit failed", rv);

            System.out.println("Verify signature");
            rv = RtPkcs11Library.getPkcs11Interface().C_Verify(hSession, hash, new NativeLong(hash.length), signature, signatureSize.getValue());
            Pkcs11Exception.throwIfNotOk("C_Verify failed", rv);

            System.out.println("Verifying has been completed successfully.");

            System.out.println("\nInitialize encryption");
            rv = RtPkcs11Library.getPkcs11ExtendedInterface().C_EncryptInit(hSession, new CK_MECHANISM(new NativeLong(Pkcs11Constants.CKM_RSA_PKCS), null, new NativeLong(0)), hPubKey.getValue());
            Pkcs11Exception.throwIfNotOk("C_EncryptInit failed", rv);

            System.out.println("Get size fo encrypted data");
            NativeLongByReference encryptedDataSize = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11ExtendedInterface().C_Encrypt(hSession, pbtEncDecData, new NativeLong(pbtEncDecData.length), null, encryptedDataSize);
            Pkcs11Exception.throwIfNotOk("C_Encrypt failed", rv);

            System.out.println("Encrypt data");
            byte[] encryptedData = new byte[encryptedDataSize.getValue().intValue()];
            rv = RtPkcs11Library.getPkcs11ExtendedInterface().C_Encrypt(hSession, pbtEncDecData, new NativeLong(pbtEncDecData.length), encryptedData, encryptedDataSize);
            Pkcs11Exception.throwIfNotOk("C_Encrypt failed", rv);

            System.out.println("Encrypted data is:");
            System.out.print("    ");
            for (int i = 0; i < encryptedData.length; ++i) {
                if ((i + 1) % 16 == 0)
                    System.out.printf("0x%02X\n    ", encryptedData[i]);
                else
                    System.out.printf("0x%02X ", encryptedData[i]);
            }
            System.out.println("\nEncryption has been completed successfully.");

            System.out.println("\nInitialize decryption");
            rv = RtPkcs11Library.getPkcs11Interface().C_DecryptInit(hSession, new CK_MECHANISM(new NativeLong(Pkcs11Constants.CKM_RSA_PKCS), null, new NativeLong(0)), hPrvKey.getValue());
            Pkcs11Exception.throwIfNotOk("C_DecryptInit failed", rv);

            System.out.println("Get size fo encrypted data");
            NativeLongByReference decryptedDataSize = new NativeLongByReference();
            rv = RtPkcs11Library.getPkcs11Interface().C_Decrypt(hSession, encryptedData, new NativeLong(encryptedData.length), null, decryptedDataSize);
            Pkcs11Exception.throwIfNotOk("C_Decrypt failed", rv);

            System.out.println("Decrypt data");
            byte[] decryptedData = new byte[decryptedDataSize.getValue().intValue()];
            rv = RtPkcs11Library.getPkcs11Interface().C_Decrypt(hSession, encryptedData, new NativeLong(encryptedData.length), decryptedData, decryptedDataSize);
            Pkcs11Exception.throwIfNotOk("C_Decrypt failed", rv);

            System.out.println("Origin data is:");
            System.out.print("    ");
            for (int i = 0; i < pbtEncDecData.length; ++i) {
                if ((i + 1) % 16 == 0)
                    System.out.printf("0x%02X\n    ", pbtEncDecData[i]);
                else
                    System.out.printf("0x%02X ", pbtEncDecData[i]);
            }

            System.out.println("\nDecrypted data is:");
            System.out.print("    ");
            for (int i = 0; i < decryptedData.length; ++i) {
                if ((i + 1) % 16 == 0)
                    System.out.printf("0x%02X\n    ", decryptedData[i]);
                else
                    System.out.printf("0x%02X ", decryptedData[i]);
            }
            System.out.println("\nDecryption has been completed successfully.");
        } catch (Exception e) {
            System.out.println(e.getMessage());
        } finally {
            NativeLong rv = RtPkcs11Library.getPkcs11ExtendedInterface().C_EX_FreeBuffer(ppbtCsr.getPointer(0));
            Util.checkIfNotOk("C_EX_FreeBuffer failed", rv.longValue());

            rv = RtPkcs11Library.getPkcs11ExtendedInterface().C_EX_FreeBuffer(ppbtInfo.getPointer(0));
            Util.checkIfNotOk("C_EX_FreeBuffer failed", rv.longValue());

            System.out.println("Deleting public key");
            rv = RtPkcs11Library.getPkcs11Interface().C_DestroyObject(hSession, hPubKey.getValue());
            Util.checkIfNotOk("C_DestroyObject failed", rv.longValue());

            System.out.println("Deleting private key");
            rv = RtPkcs11Library.getPkcs11Interface().C_DestroyObject(hSession, hPrvKey.getValue());
            Util.checkIfNotOk("C_DestroyObject failed", rv.longValue());

            System.out.println("Deleting certificate");
            rv = RtPkcs11Library.getPkcs11Interface().C_DestroyObject(hSession, hCert.getValue());
            Util.checkIfNotOk("C_DestroyObject failed", rv.longValue());

            System.out.println("Logging out");
            rv = RtPkcs11Library.getPkcs11Interface().C_Logout(hSession);
            Util.checkIfNotOk("C_Logout failed", rv.longValue());

            System.out.println("Closing session");
            rv = RtPkcs11Library.getPkcs11Interface().C_CloseSession(hSession);
            Util.checkIfNotOk("C_CloseSession failed", rv.longValue());

            System.out.println("Finalizing PKCS11 library");
            rv = RtPkcs11Library.getPkcs11Interface().C_Finalize(null);
            Util.checkIfNotOk("C_Finalize failed", rv.longValue());

            System.out.println("Test has been completed.");
        }
    }
}
