/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2018, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

package ru.rutoken.samples.sunPkcs11Wrapper;

import ru.rutoken.pkcs11jna.RtPkcs11Constants;
import ru.rutoken.samples.Constants;
import sun.security.pkcs11.wrapper.*;

public class GOSTR3410_2012_256 {
    private final static CK_ATTRIBUTE[] privGostKeyAttr = {
            new CK_ATTRIBUTE(PKCS11Constants.CKA_CLASS, PKCS11Constants.CKO_PRIVATE_KEY),
            new CK_ATTRIBUTE(PKCS11Constants.CKA_LABEL,
                    "Sample GOST R 34.10-2012 (256 bits) Private Key (Aktiv Co.)".toCharArray()),
            new CK_ATTRIBUTE(PKCS11Constants.CKA_ID,
                    "Sample GOST R 34.10-2012 (256 bits) Keypair (Aktiv Co.)".toCharArray()),
            new CK_ATTRIBUTE(PKCS11Constants.CKA_KEY_TYPE, RtPkcs11Constants.CKK_GOSTR3410),
            new CK_ATTRIBUTE(PKCS11Constants.CKA_TOKEN, true),
            new CK_ATTRIBUTE(PKCS11Constants.CKA_PRIVATE, true),
            new CK_ATTRIBUTE(RtPkcs11Constants.CKA_GOSTR3410_PARAMS, Constants.ATTR_CRYPTO_PRO_A),
            new CK_ATTRIBUTE(RtPkcs11Constants.CKA_GOSTR3411_PARAMS, Constants.ATTR_HASH_2012_256)
    };
    private final static CK_ATTRIBUTE[] pubGostKeyAttr = {
            new CK_ATTRIBUTE(PKCS11Constants.CKA_CLASS, PKCS11Constants.CKO_PUBLIC_KEY),
            new CK_ATTRIBUTE(PKCS11Constants.CKA_LABEL,
                    "Sample GOST R 34.10-2012 (256 bits) Public Key (Aktiv Co.)".toCharArray()),
            new CK_ATTRIBUTE(PKCS11Constants.CKA_ID,
                    "Sample GOST R 34.10-2012 (256 bits) Keypair (Aktiv Co.)".toCharArray()),
            new CK_ATTRIBUTE(PKCS11Constants.CKA_KEY_TYPE, RtPkcs11Constants.CKK_GOSTR3410),
            new CK_ATTRIBUTE(PKCS11Constants.CKA_TOKEN, true),
            new CK_ATTRIBUTE(PKCS11Constants.CKA_PRIVATE, false),
            new CK_ATTRIBUTE(RtPkcs11Constants.CKA_GOSTR3410_PARAMS, Constants.ATTR_CRYPTO_PRO_A),
            new CK_ATTRIBUTE(RtPkcs11Constants.CKA_GOSTR3411_PARAMS, Constants.ATTR_HASH_2012_256)
    };
    private final static CK_ATTRIBUTE[] signGostKeyAttr = {
            new CK_ATTRIBUTE(PKCS11Constants.CKA_CLASS, PKCS11Constants.CKO_PRIVATE_KEY),
            new CK_ATTRIBUTE(PKCS11Constants.CKA_LABEL,
                    "Sample GOST R 34.10-2012 (256 bits) Private Key (Aktiv Co.)".toCharArray())
    };
    private final static CK_ATTRIBUTE[] verifyGostKeyAttr = {
            new CK_ATTRIBUTE(PKCS11Constants.CKA_CLASS, PKCS11Constants.CKO_PUBLIC_KEY),
            new CK_ATTRIBUTE(PKCS11Constants.CKA_LABEL,
                    "Sample GOST R 34.10-2012 (256 bits) Public Key (Aktiv Co.)".toCharArray())
    };
    private final static CK_ATTRIBUTE[] value = {new CK_ATTRIBUTE(PKCS11Constants.CKA_VALUE)};
    private static PKCS11 pkcs11Wrapper;

    public static void main(String[] args) {
        long session = -1;

        try {
            System.out.println("Example of working with GOSTR3410-2012 algorithm using rtpkcs11ecp " +
                    "via sun.security.pkcs11.wrapper");

            System.out.println("Library initialization and acquiring of function list");
            pkcs11Wrapper = PKCS11.getInstance("rtpkcs11ecp", "C_GetFunctionList", new sun.security.pkcs11.wrapper.CK_C_INITIALIZE_ARGS(), false);

            System.out.println("Acquiring list of slots with connected tokens");
            long[] slotList = pkcs11Wrapper.C_GetSlotList(true);
            if (slotList.length == 0)
                throw new Exception("No Rutoken is available!");

            System.out.println("Opening session");
            session = pkcs11Wrapper.C_OpenSession(slotList[0],
                    PKCS11Constants.CKF_SERIAL_SESSION | PKCS11Constants.CKF_RW_SESSION, null, null);

            System.out.println("Logging in as user");
            pkcs11Wrapper.C_Login(session, PKCS11Constants.CKU_USER, new String(Constants.DEFAULT_USER_PIN).toCharArray());

            System.out.println("Generating GOST R 34.10-2012 (256 bits) key pair");
            pkcs11Wrapper.C_GenerateKeyPair(session,
                    new CK_MECHANISM(RtPkcs11Constants.CKM_GOSTR3410_KEY_PAIR_GEN), pubGostKeyAttr, privGostKeyAttr);

            System.out.println("Acquiring key for signing");
            long[] foundObjects = Util.findObject(pkcs11Wrapper, session, signGostKeyAttr);
            if (foundObjects.length == 0)
                throw new Exception("No signature key found!");

            byte[] digest = Util.hashData(pkcs11Wrapper, session, RtPkcs11Constants.CKM_GOSTR3411_12_256, Constants.MESSAGE);

            System.out.println("Data signing");
            pkcs11Wrapper.C_SignInit(session, new CK_MECHANISM(RtPkcs11Constants.CKM_GOSTR3410), foundObjects[0]);
            byte[] signature = pkcs11Wrapper.C_Sign(session, digest);

            System.out.println("Signed data:");
            for (int i = 0; i < signature.length; ++i) {
                System.out.printf(" %02X", signature[i]);
                if ((i + 1) % 16 == 0)
                    System.out.println();
            }
            System.out.println("Data has been signed successfully.");

            System.out.println("Acquiring key for verification");
            foundObjects = Util.findObject(pkcs11Wrapper, session, verifyGostKeyAttr);
            if (foundObjects.length == 0)
                throw new Exception("No verification key found!\n");

            pkcs11Wrapper.C_GetAttributeValue(session, foundObjects[0], value);
            byte[] pubKeyValue = value[0].getByteArray();
            System.out.println("Public key value: " + Util.bytesToHex(pubKeyValue));

            System.out.println("Verifying signature ");
            pkcs11Wrapper.C_VerifyInit(session, new CK_MECHANISM(RtPkcs11Constants.CKM_GOSTR3410), foundObjects[0]);
            pkcs11Wrapper.C_Verify(session, digest, signature);
            System.out.println("Verifying has been completed successfully");

            System.out.println("Searching objects on token");
            CK_ATTRIBUTE[][] Template = {signGostKeyAttr, verifyGostKeyAttr};
            for (CK_ATTRIBUTE[] tmpl : Template) {
                for (long found : Util.findObject(pkcs11Wrapper, session, tmpl)) {
                    System.out.println("Destroying object");
                    pkcs11Wrapper.C_DestroyObject(session, found);
                }
            }

            System.out.println("Test has been completed successfully");
        } catch (Exception e) {
            System.out.println("Some error occurred. Error code: " + e.getMessage());
        } finally {
            System.out.println("Logging out");
            try {
                pkcs11Wrapper.C_Logout(session);
            } catch (PKCS11Exception e) {
                System.out.println(e.getMessage());
            }
            System.out.println("Closing session");
            try {
                pkcs11Wrapper.C_CloseSession(session);
            } catch (PKCS11Exception e) {
                System.out.println(e.getMessage());
            }
            System.out.println("Finalizing PKCS11");
            try {
                pkcs11Wrapper.C_Finalize(null);
            } catch (PKCS11Exception e) {
                System.out.println(e.getMessage());
            }
        }
    }
}
