/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2018, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/*! \file p11.h
 *  \brief Заголовочный файл с описанием функций и структур для управления сессиями и ключами.
 */

#ifndef RUTOKEN_ENGINE_P11_H
#define RUTOKEN_ENGINE_P11_H

#include <openssl/evp.h>

#include <cryptoki.h>

#include "mutex.h"


#ifdef __cplusplus
extern "C" {
#endif

/*! \brief Структура для хранения таблицы функций сессии. */
typedef struct rt_eng_p11_session_vtable_st rt_eng_p11_session_vtable;

/*! \brief Структура для хранения pkcs11 сессии и её таблицы функций. */
typedef struct {
    /*! Указатель на таблицу функций. */
    const rt_eng_p11_session_vtable *vtable;
    /*! Указатель на сессию. */
    void *self;
} rt_eng_p11_session;

/*! \brief Таблица функций сессии. */
struct rt_eng_p11_session_vtable_st {
    /*! \brief Функция удаления сессии. */
    void (*free)(rt_eng_p11_session *that);
    /*! \brief Функция увеличения счётчика ссылок.
     *  \return 1 в случае успеха, 0 в случае ошибки.
     */
    int (*up_ref)(rt_eng_p11_session *that);
    /*! \brief Функция блокирования сессии.
     *  \return 1 в случае успеха, 0 в случае ошибки.
     */
    int (*lock)(rt_eng_p11_session *that);
    /*! \brief Функция освобождения сессии.
     *  \return 1 в случае успеха, 0 в случае ошибки.
     */
    int (*unlock)(rt_eng_p11_session *that);
};

/*! \brief Функция создания пустой сессии.
 *  \return описатель обёрнутой сессии в случае успеха, NULL в случае ошибки.
 */
rt_eng_p11_session rt_eng_p11_session_new_null(void);

// Wrap the PKCS#11 session handle for using from multiple threads. This wrapping object owns the session handle.
// The mutex_new argument is optional. If it is NULL then built-in methods will be used.
/*! \brief Функция создания сессии.
 *  \param function_list -- список функций, полученных от rtpkcs11ecp.
 *  \param handle -- описатель сессии, полученной от rtpkcs11ecp.
 *  \param hold -- параметр передачи прав управления сессией. При его равенстве 1, сессия будет закрыта средствами OpenSSL при завершении работы. При его равенстве 0, сессия должна быть закрыта пользователем самостоятельно.
 *  \param mutex_new (необязательный) -- мьютекс для управления сессией. При отсутствии необходимости его использования, устанавливается в значение NULL.
 *  \return описатель обёрнутой сессии в случае успеха, NULL в случае ошибки.
 */
rt_eng_p11_session rt_eng_p11_session_new(CK_FUNCTION_LIST_PTR function_list, CK_SESSION_HANDLE handle,
                                          int hold, rt_eng_mutex_new_t mutex_new);

/*! \brief Установка текущей сессии для получения энтропии.
 *  \param session -- описатель обёрнутой сессии.
 *  \return 1 в случае успеха, 0 в случае ошибки.
 */
int rt_eng_rand_set0_p11_session(rt_eng_p11_session session);

/*! \brief Получение описателя ключевой пары.
 *  \param session -- описатель обёрнутой сессии.
 *  \param private_key (необязательный) -- описатель закрытого ключа, полученный от rtpkcs11ecp. Если для работы приложения отсутствует необходимость использовать закрытый ключ, то следует установить данный параметр в значение NULL.
 *  \param public_key -- описатель открытого ключа, полученный от rtpkcs11ecp.
 *  \return описатель ключевой пары в случае успеха, NULL в случае ошибки.
 */
EVP_PKEY *rt_eng_new_p11_ossl_evp_pkey(rt_eng_p11_session session, CK_OBJECT_HANDLE private_key,
                                       CK_OBJECT_HANDLE public_key);

// This function must be called before removing the key from a token.
// It prevents the engine from using a wrong key handle.
/*! \brief Данная функция делает описатель ключевой пары недействительным.
 *  При использовании ключевой пары, находящейся на токене, могут возникать ситуации, при которых описатель объектов на токене становится более не актуальным, например при сбросе прав доступа к объектам на токене или при физическом удаление ключа из памяти токена. Во избежании неопределённого поведения описатель ключевой пары необходимо делать недействительным.
 *  \param pkey -- описатель ключевой пары.
 *  \return 1 в случае успеха, 0 в случае ошибки.
 */
int rt_eng_invalidate_p11_ossl_evp_pkey(EVP_PKEY *pkey);


#ifdef __cplusplus
}
#endif

#endif // RUTOKEN_ENGINE_P11_H
