/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2018, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

#include <assert.h>

#include <Common.h>

#include <openssl/cms.h>

int main(void)
{
	EVP_PKEY* key;                                           // Описатель ключевой пары
	ENGINE* rtEngine;                                        // rtengine
	BIO* outBio;                                             // Описатель потока вывода

	X509_REQ* csr;                                           // Описатель запроса
	X509_NAME* subject;                                      // Описатель субъекта
	STACK_OF(X509_EXTENSION) * extensions;                   // Описатель контейнера расширений
	STACK_OF(X509_EXTENSION) * retCheckValue;                // Код возврата
	X509_EXTENSION* keyUsageExt;                             // Описатель расширения 'key usage'
	X509_EXTENSION* extendedKeyUsageExt;                     // Описатель расширения 'extended key usage'
	X509_EXTENSION* subjectSignTool;                         // Описатель расширения 'subject sign tool'

	int r;                                                   // Код возврата
	int errorCode = 1;                                       // Флаг ошибки

	printf("Sample has started.\n\n");
	/*************************************************************************
	* Создание rtengine и регистрация его в OpenSSL                          *
	*************************************************************************/
	r = rt_eng_init();
	CHECK("  rt_eng_init", r == 1, exit);

	/*************************************************************************
	* Получение rtengine                                                     *
	*************************************************************************/
	rtEngine = rt_eng_get0_engine();
	assert(rtEngine);

	/*************************************************************************
	* Установка rtengine реализацией по умолчанию                            *
	*************************************************************************/
	r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
	CHECK("  ENGINE_set_default", r == 1, finalize_engine);

	/*************************************************************************
	* Получение ключевой пары                                                *
	*************************************************************************/
	printf("  get_key_pair...\n");
	key = get_key_pair();
	CHECK("  get_key_pair", key != NULL, unregister_engine);

	/*************************************************************************
	* Создание запроса                                                       *
	*************************************************************************/
	csr = X509_REQ_new();
	CHECK("  X509_REQ_new", csr != NULL, free_key);

	/*************************************************************************
	* Получение имени субъекта                                               *
	*************************************************************************/
	subject = X509_REQ_get_subject_name(csr);
	CHECK("  X509_REQ_get_subject_name", subject != NULL, free_csr);

	/*************************************************************************
	* Установка значения для поля 'Country'                                  *
	*************************************************************************/
	r = X509_NAME_add_entry_by_txt(subject, "C", MBSTRING_ASC, (unsigned char*)"RU", -1, -1, 0);
	CHECK("  X509_NAME_add_entry_by_txt", r == 1, free_csr);

	/*************************************************************************
	* Установка значения для поля 'Common name'                              *
	*************************************************************************/
	r = X509_NAME_add_entry_by_txt(subject, "CN", MBSTRING_UTF8, (unsigned char*)"Иванов", -1, -1, 0);
	CHECK("  X509_NAME_add_entry_by_txt", r == 1, free_csr);

	/*************************************************************************
	* Установка значения для поля 'E-mail'                                   *
	*************************************************************************/
	r = X509_NAME_add_entry_by_txt(subject, "emailAddress", MBSTRING_ASC, (unsigned char*)"ivanov@mail.ru", -1, -1, 0);
	CHECK("  X509_NAME_add_entry_by_txt", r == 1, free_csr);

	/*************************************************************************
	* Установка значения для поля 'State'                                    *
	*************************************************************************/
	r = X509_NAME_add_entry_by_txt(subject, "ST", MBSTRING_UTF8, (unsigned char*)"Москва", -1, -1, 0);
	CHECK("  X509_NAME_add_entry_by_txt", r == 1, free_csr);

	/*************************************************************************
	* Создание расширения 'Subject sign tool'                                *
	*************************************************************************/
	subjectSignTool = X509V3_EXT_nconf_nid(NULL, NULL, NID_subjectSignTool, "ASN1:FORMAT:UTF8,UTF8String:СКЗИ \"Рутокен ЭЦП 2.0\"");
	CHECK("  X509V3_EXT_nconf_nid", subjectSignTool != NULL, free_csr);

	/*************************************************************************
	* Создание расширения 'Key usage'                                        *
	*************************************************************************/
	keyUsageExt = X509V3_EXT_nconf_nid(NULL, NULL, NID_key_usage, "digitalSignature,nonRepudiation,keyEncipherment,dataEncipherment");
	CHECK("  X509V3_EXT_nconf_nid", keyUsageExt != NULL, free_subjectSignTool);

	/*************************************************************************
	* Создание расширения 'Extended key usage'                               *
	*************************************************************************/
	extendedKeyUsageExt = X509V3_EXT_nconf_nid(NULL, NULL, NID_ext_key_usage, "emailProtection");
	CHECK("  X509V3_EXT_nconf_nid", extendedKeyUsageExt != NULL, free_keyUsage);

	/*************************************************************************
	* Добавление расширения 'Subject sign tool' в контейнер                  *
	*************************************************************************/
	extensions = NULL;
	retCheckValue = X509v3_add_ext(&extensions, subjectSignTool, -1);
	CHECK("  X509v3_add_ext", retCheckValue != NULL, free_extKeyUsage);

	/*************************************************************************
	* Добавление расширения 'Key usage' в контейнер                          *
	*************************************************************************/
	retCheckValue = X509v3_add_ext(&extensions, keyUsageExt, -1);
	CHECK("  X509v3_add_ext", retCheckValue != NULL, free_extensions);

	/*************************************************************************
	* Добавление расширения 'Extended key usage' в контейнер                 *
	*************************************************************************/
	retCheckValue = X509v3_add_ext(&extensions, extendedKeyUsageExt, -1);
	CHECK("  X509v3_add_ext", retCheckValue != NULL, free_extensions);

	/*************************************************************************
	* Установка расширений для запроса                                       *
	*************************************************************************/
	r = X509_REQ_add_extensions(csr, extensions);
	CHECK("  X509_REQ_add_extensions", r == 1, free_extensions);

	/*************************************************************************
	* Установка открытого ключа                                              *
	*************************************************************************/
	r = X509_REQ_set_pubkey(csr, key);
	CHECK("  X509_REQ_set_pubkey", r == 1, free_extensions);

	/*************************************************************************
	* Подпись запроса                                                        *
	*************************************************************************/
	r = X509_REQ_sign(csr, key, EVP_get_digestbynid(NID_id_GostR3410_2012_256));
	CHECK("  X509_REQ_sign", r > 1, free_extensions);

	/*************************************************************************
	* Открытие потока вывода в файл                                          *
	*************************************************************************/
	outBio = BIO_new_file("csr.pem", "w");
	CHECK("  BIO_new_file", outBio != NULL, free_extensions);

	/*************************************************************************
	* Запись запроса                                                         *
	*************************************************************************/
	r = PEM_write_bio_X509_REQ(outBio, csr);
	CHECK("  PEM_write_bio_X509_REQ", r == 1, free_out_bio);

	/*************************************************************************
	* Установка признака успешного завершения программы                      *
	*************************************************************************/
	errorCode = 0;
free_out_bio:

	/*************************************************************************
	* Закрытие потока вывода                                                 *
	*************************************************************************/
	BIO_free_all(outBio);
free_extensions:

	/*************************************************************************
	* Освобождение контейнера расширений                                     *
	*************************************************************************/
	sk_X509_EXTENSION_pop_free(extensions, X509_EXTENSION_free);
free_extKeyUsage:

	/*************************************************************************
	* Освобождение расширения                                                *
	*************************************************************************/
	X509_EXTENSION_free(extendedKeyUsageExt);
free_keyUsage:

	/*************************************************************************
	* Освобождение расширения                                                *
	*************************************************************************/
	X509_EXTENSION_free(keyUsageExt);
free_subjectSignTool:

	/*************************************************************************
	* Освобождение расширения                                                *
	*************************************************************************/
	X509_EXTENSION_free(subjectSignTool);
free_csr:

	/*************************************************************************
	* Освобождение запроса                                                   *
	*************************************************************************/
	X509_REQ_free(csr);
free_key:

	/*************************************************************************
	* Освобождение описателя ключевой пары                                   *
	*************************************************************************/
	printf("  free_key_pair...\n");
	r = free_key_pair(key);
	CHECK_RELEASE("  free_key_pair", r == 0, errorCode);
unregister_engine:

	/*************************************************************************
	* Разрегистрация rtengine из OpenSSL                                     *
	*************************************************************************/
	ENGINE_unregister_pkey_asn1_meths(rtEngine);
	ENGINE_unregister_pkey_meths(rtEngine);
	ENGINE_unregister_digests(rtEngine);
	ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

	/*************************************************************************
	* Деинициализация rtengine                                               *
	*************************************************************************/
	r = rt_eng_final();
	CHECK_RELEASE("  rt_eng_final", r == 1, errorCode);
exit:
	if (errorCode) {
		printf("\n\nSample has failed. Some error has occurred.\n");
	} else {
		printf("\n\nSample has been completed successfully.\n");
	}
	return errorCode;
}
