/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2018, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

#include <assert.h>

#include <Common.h>

int main(void)
{
	EVP_PKEY* key;                                        // Описатель ключевой пары отправителя
	EVP_PKEY* peerKey;                                    // Описатель открытого ключа получателя

	ENGINE* rtEngine;                                     // rtengine
	EVP_PKEY_CTX* ctx;                                    // Контекст ВКО
	BIO* inBio;                                           // Описатель потока ввода
	FILE* outputFile;                                     // Описатель потока вывода
	uint8_t ukm[] = {1, 2, 3, 4, 5, 6, 7, 8};             // Вектор инициализации

	unsigned char* derivedKey;                            // Буфер с ключом обмена
	size_t keylen;                                        // Длина ключа обмена

	size_t numberOfBytes;                                 // Число записанных байт
	int r;                                                // Код возврата
	int errorCode = 1;                                    // Флаг ошибки

	printf("Sample has started.\n\n");
	/*************************************************************************
	* Создание rtengine и регистрация его в OpenSSL                          *
	*************************************************************************/
	r = rt_eng_init();
	CHECK("  rt_eng_init", r == 1, exit);

	/*************************************************************************
	* Получение rtengine                                                     *
	*************************************************************************/
	rtEngine = rt_eng_get0_engine();
	assert(rtEngine);

	/*************************************************************************
	* Установка rtengine реализацией по умолчанию                            *
	*************************************************************************/
	r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
	CHECK("  ENGINE_set_default", r == 1, finalize_engine);

	/*************************************************************************
	* Открытие поточного ввода из файла для ключа                            *
	*************************************************************************/
	inBio = BIO_new_file("key.pem", "r");
	CHECK("  BIO_new_file", inBio != NULL, unregister_engine);

	/*************************************************************************
	* Чтение ключа отправителя из файла                                      *
	*************************************************************************/
	key = PEM_read_bio_PrivateKey(inBio, NULL, NULL, NULL);
	CHECK("  PEM_read_bio_PrivateKey", key != NULL, free_in_bio);

	/*************************************************************************
	* Получение ключа получателя                                             *
	*************************************************************************/
	printf("  get_public_key...\n");
	peerKey = get_public_key();
	CHECK("  get_public_key", peerKey != NULL, free_key);

	/*************************************************************************
	* Создание контекста выработки ключей обмена                             *
	*************************************************************************/
	ctx = EVP_PKEY_CTX_new(key, rtEngine);
	CHECK("  EVP_PKEY_CTX_new", ctx != NULL, free_peer_key);

	/*************************************************************************
	* Инициализация контекста выработки ключей обмена                        *
	*************************************************************************/
	r = EVP_PKEY_derive_init(ctx);
	CHECK("  EVP_PKEY_derive_init", r == 1, free_context);

	/*************************************************************************
	* Установка ключа получателя                                             *
	*************************************************************************/
	r = EVP_PKEY_derive_set_peer(ctx, peerKey);
	CHECK("  EVP_PKEY_derive_set_peer", r == 1, free_context);

	/*************************************************************************
	* Установка вектора инициализации                                        *
	*************************************************************************/
	r = EVP_PKEY_CTX_ctrl(ctx, -1, EVP_PKEY_OP_DERIVE, EVP_PKEY_CTRL_SET_IV, sizeof(ukm), ukm);
	CHECK("  EVP_PKEY_CTX_ctrl", r > 0, free_context);

	/*************************************************************************
	* Получение размера ключа                                                *
	*************************************************************************/
	keylen = 0;
	r = EVP_PKEY_derive(ctx, NULL, &keylen);
	CHECK("  EVP_PKEY_derive", r == 1, free_context);

	/*************************************************************************
	* Выделение памяти для ключа обмена                                      *
	*************************************************************************/
	derivedKey = OPENSSL_malloc(keylen);
	CHECK("  OPENSSL_malloc", derivedKey != NULL, free_context);

	/*************************************************************************
	* Выработка ключа обмена                                                 *
	*************************************************************************/
	r = EVP_PKEY_derive(ctx, derivedKey, &keylen);
	CHECK("  EVP_PKEY_derive", r == 1, free_derived_key);

	/*************************************************************************
	* Открытие поточного вывода в файл                                       *
	*************************************************************************/
	outputFile = fopen("derived_key", "wb");
	CHECK("  fopen", outputFile != NULL, free_derived_key);

	/*************************************************************************
	* Запись ключа обмена в файл                                             *
	*************************************************************************/
	numberOfBytes = fwrite(derivedKey, 1, keylen, outputFile);
	CHECK("  fwrite", numberOfBytes == keylen, close_file);

	/*************************************************************************
	* Установка признака успешного завершения программы                      *
	*************************************************************************/
	errorCode = 0;
close_file:

	/*************************************************************************
	* Закрытие потока вывода                                                 *
	*************************************************************************/
	r = fclose(outputFile);
	CHECK_RELEASE("  fclose", r == 0, errorCode);
free_derived_key:

	/*************************************************************************
	* Освобождение буфера с ключом обмена                                    *
	*************************************************************************/
	OPENSSL_free(derivedKey);
free_context:

	/*************************************************************************
	* Освобождение контекста ВКО                                             *
	*************************************************************************/
	EVP_PKEY_CTX_free(ctx);
free_peer_key:

	/*************************************************************************
	* Освобождение описателя ключа получателя                                *
	*************************************************************************/
	printf("  free_public_key...\n");
	r = free_public_key(peerKey);
	CHECK_RELEASE("  free_public_key", r == 0, errorCode);
free_key:

	/*************************************************************************
	* Освобождение описателя ключевой пары отправителя                       *
	*************************************************************************/
	EVP_PKEY_free(key);
free_in_bio:

	/*************************************************************************
	* Закрытие потока ввода для ключа                                        *
	*************************************************************************/
	BIO_free_all(inBio);
unregister_engine:

	/*************************************************************************
	* Разрегистрация rtengine из OpenSSL                                     *
	*************************************************************************/
	ENGINE_unregister_pkey_asn1_meths(rtEngine);
	ENGINE_unregister_pkey_meths(rtEngine);
	ENGINE_unregister_digests(rtEngine);
	ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

	/*************************************************************************
	* Деинициализация rtengine                                               *
	*************************************************************************/
	r = rt_eng_final();
	CHECK_RELEASE("  rt_eng_final", r == 1, errorCode);
exit:
	if (errorCode) {
		printf("\n\nSample has failed. Some error has occurred.\n");
	} else {
		printf("\n\nSample has been completed successfully.\n");
	}
	return errorCode;
}
