/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2018, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

#include <assert.h>
#include <string.h>

#include <Common.h>

size_t unWrapKey(EVP_PKEY* key, unsigned char** unwrappedKey)
{
	EVP_PKEY_CTX* ctx;                                 // Контекст шифрования

	unsigned char* wrappedKey;                         // Зашифрованный ключ
	size_t wrappedKeySize;                             // Длина зашифрованного ключа
	size_t unwrappedKeySize = 0;                       // Длина ключа шифрования содержимого
	FILE* input;                                       // Описатель потока ввода

	size_t numberOfBytes;                              // Число прочитанных байт
	size_t result = 0;                                 // Результирующее значение функции
	int r;                                             // Код возврата

	/*************************************************************************
	* Открытие поточного ввода из файла                                      *
	*************************************************************************/
	input = fopen("wrapped_key", "rb");
	CHECK("    fopen", input != NULL, exit);

	/*************************************************************************
	* Определение размера файла                                              *
	*************************************************************************/
	r = fseek(input, 0, SEEK_END);
	CHECK("    fseek", r == 0, close_file);
	wrappedKeySize = ftell(input);
	CHECK("    ftell", wrappedKeySize > 0, close_file);
	r = fseek(input, 0, SEEK_SET);
	CHECK("    fseek", r == 0, close_file);

	/*************************************************************************
	* Выделение памяти для зашифрованного ключа                              *
	*************************************************************************/
	wrappedKey = OPENSSL_malloc(wrappedKeySize);
	CHECK("    OPENSSL_malloc", wrappedKey != NULL, close_file);

	/*************************************************************************
	* Чтение зашифрованного ключа из файла                                   *
	*************************************************************************/
	numberOfBytes = fread(wrappedKey, 1, wrappedKeySize, input);
	CHECK("    fread", numberOfBytes == wrappedKeySize, free_wrappedKey);

	/*************************************************************************
	* Создание контекста расшифрования                                       *
	*************************************************************************/
	ctx = EVP_PKEY_CTX_new(key, NULL);
	CHECK("    EVP_PKEY_CTX_new", ctx != NULL, free_wrappedKey);

	/*************************************************************************
	* Инициализация контекста расшифрования                                  *
	*************************************************************************/
	r = EVP_PKEY_decrypt_init(ctx);
	CHECK("    EVP_PKEY_decrypt_init", r == 1, free_context);

	/*************************************************************************
	* Определение размера расшифрованного ключа                              *
	*************************************************************************/
	r = EVP_PKEY_decrypt(ctx, NULL, &unwrappedKeySize, wrappedKey, wrappedKeySize);
	CHECK("    EVP_PKEY_decrypt", r == 1, free_context);

	/*************************************************************************
	* Выделение памяти для расшифрованного ключа                             *
	*************************************************************************/
	*unwrappedKey = OPENSSL_secure_malloc(unwrappedKeySize);
	CHECK("    OPENSSL_secure_malloc", *unwrappedKey != NULL, free_context);

	/*************************************************************************
	* Расшифрование                                                          *
	*************************************************************************/
	r = EVP_PKEY_decrypt(ctx, *unwrappedKey, &unwrappedKeySize, wrappedKey, wrappedKeySize);
	CHECK("    EVP_PKEY_decrypt", r == 1, free_unwrappedKey);

	/*************************************************************************
	* Установка результирующего значения                                     *
	*************************************************************************/
	result = unwrappedKeySize;
	goto free_context;
free_unwrappedKey:

	/*************************************************************************
	* Освобождение ключа шифрования содержимого                              *
	*************************************************************************/
	OPENSSL_secure_free(*unwrappedKey);
free_context:

	/*************************************************************************
	* Освобождение контекста                                                 *
	*************************************************************************/
	EVP_PKEY_CTX_free(ctx);
free_wrappedKey:

	/*************************************************************************
	* Освобождение зашифрованного ключа                                      *
	*************************************************************************/
	OPENSSL_free(wrappedKey);
close_file:

	/*************************************************************************
	* Закрытие потока ввода зашифрованного ключа                             *
	*************************************************************************/
	fclose(input);
exit:
	return result;
}

int decryptFile(unsigned char* cek)
{
	EVP_CIPHER_CTX* ctx;                               // Контекст расшифрования
	const EVP_CIPHER* cipher;                          // Описатель алгоритма шифрования

	FILE* input;                                       // Описатель потока ввода
	FILE* output;                                      // Описатель потока вывода
	size_t numberOfBytes;                              // Число прочитанных или записанных байт
	unsigned char* decryptedData;                      // Расшифрованные данные
	unsigned char* ciphertext;                         // Шифротекст
	unsigned char iv[8];                               // Инициализирующий вектор
	int dataLen;                                       // Длина расшифрованных данных
	int ciphertextLen;                                 // Длина шифротекста
	int sizeIv;                                        // Длина инициализируещего вектора
	int count;                                         // Количество расшифрованных байт

	int r;                                             // Код возврата
	int errorCode = 1;                                 // Флаг ошибки

	/*************************************************************************
	* Открытие поточного вывода в файл                                       *
	*************************************************************************/
	input = fopen("text_encrypted_with_cek", "rb");
	CHECK("    fopen", input != NULL, exit);

	/*************************************************************************
	* Определение размера файла                                              *
	*************************************************************************/
	r = fseek(input, 0, SEEK_END);
	CHECK("    fseek", r == 0, close_input);
	ciphertextLen = ftell(input);
	CHECK("    ftell", ciphertextLen > 0, close_input);
	r = fseek(input, 0, SEEK_SET);
	CHECK("    fseek", r == 0, close_input);

	/*************************************************************************
	* Получение алгоритма шифрования                                         *
	*************************************************************************/
	cipher = EVP_get_cipherbynid(rt_eng_nid_gost28147_paramset_a_cfb);
	CHECK("    EVP_get_cipherbynid", cipher != NULL, close_input);

	/*************************************************************************
	* Получение длины инициализируещего вектора                              *
	*************************************************************************/
	sizeIv = EVP_CIPHER_iv_length(cipher);
	CHECK("    EVP_CIPHER_iv_length", sizeIv == 8, close_input);
	ciphertextLen -= sizeIv;

	/*************************************************************************
	* Чтение вектора инициализации                                           *
	*************************************************************************/
	numberOfBytes = fread(iv, 1, sizeIv, input);
	CHECK("    fread", numberOfBytes == (size_t)sizeIv, close_input);

	/*************************************************************************
	* Выделение памяти шифротекста                                           *
	*************************************************************************/
	ciphertext = OPENSSL_malloc(ciphertextLen);
	CHECK("    OPENSSL_malloc", ciphertext != NULL, close_input);

	/*************************************************************************
	* Чтение данных шифротекста                                              *
	*************************************************************************/
	numberOfBytes = fread(ciphertext, 1, ciphertextLen, input);
	CHECK("    fread", numberOfBytes == (size_t)ciphertextLen, free_ciphertext);

	/*************************************************************************
	* Создание контекста расшифрования                                       *
	*************************************************************************/
	ctx = EVP_CIPHER_CTX_new();
	CHECK("    EVP_PKEY_CTX_new", ctx != NULL, free_ciphertext);

	/*************************************************************************
	* Инициализация контекста расшифрования                                  *
	*************************************************************************/
	r = EVP_DecryptInit_ex(ctx, cipher, NULL, cek, iv);
	CHECK("    EVP_DecryptInit_ex", r == 1, free_context);

	/*************************************************************************
	* Выделение памяти для расшифрованных даных                              *
	*************************************************************************/
	decryptedData = OPENSSL_malloc(ciphertextLen + EVP_CIPHER_CTX_block_size(ctx));
	CHECK("    OPENSSL_malloc", decryptedData != NULL, free_context);

	/*************************************************************************
	* Расшифрование данных                                                   *
	*************************************************************************/
	r = EVP_DecryptUpdate(ctx, decryptedData, &count, ciphertext, ciphertextLen);
	CHECK("    EVP_DecryptUpdate", r == 1, free_decrypted_data);
	dataLen = count;

	/*************************************************************************
	* Завершение расшифрования данных                                        *
	*************************************************************************/
	r = EVP_DecryptFinal_ex(ctx, decryptedData + count, &count);
	CHECK("    EVP_DecryptFinal_ex", r == 1, free_decrypted_data);
	dataLen += count;

	/*************************************************************************
	* Открытие поточного вывода в файл                                       *
	*************************************************************************/
	output = fopen("decrypted_text_with_unwrapped_key", "wb");
	CHECK("    fopen", output != NULL, free_decrypted_data);

	/*************************************************************************
	* Запись расшифрованных данных в файл                                    *
	*************************************************************************/
	numberOfBytes = fwrite(decryptedData, 1, dataLen, output);
	CHECK("    fwrite", numberOfBytes == (size_t)dataLen, close_output);

	/*************************************************************************
	* Установка признака успешного завершения программы                      *
	*************************************************************************/
	errorCode = 0;
close_output:

	/*************************************************************************
	* Закрытие потока вывода                                                 *
	*************************************************************************/
	r = fclose(output);
	CHECK_RELEASE("    fclose", r == 0, errorCode);
free_decrypted_data:

	/*************************************************************************
	* Освобождение расшифрованных данных                                     *
	*************************************************************************/
	OPENSSL_free(decryptedData);
free_context:

	/*************************************************************************
	* Освобождение контекста шифрования                                      *
	*************************************************************************/
	EVP_CIPHER_CTX_free(ctx);
free_ciphertext:

	/*************************************************************************
	* Освобождение шифротекста                                               *
	*************************************************************************/
	OPENSSL_free(ciphertext);
close_input:

	/*************************************************************************
	* Закрытие потока ввода                                                  *
	*************************************************************************/
	r = fclose(input);
	CHECK_RELEASE("    fclose", r == 0, errorCode);
exit:
	return errorCode;
}

int main(void)
{
	EVP_PKEY* key;                                        // Описатель ключевой пары
	ENGINE* rtEngine;                                     // rtengine

	unsigned char* cek;                                   // Раcшифрованный ключ
	size_t cekSize;                                       // Длина расшифрованного ключа

	int r;                                                // Код возврата
	int errorCode = 1;                                    // Флаг ошибки

	printf("Sample has started.\n\n");
	/*************************************************************************
	* Выделение защищенной области памяти                                    *
	*************************************************************************/
	CRYPTO_secure_malloc_init(4096, 32);

	/*************************************************************************
	* Создание rtengine и регистрация его в OpenSSL                          *
	*************************************************************************/
	r = rt_eng_init();
	CHECK("  rt_eng_init", r == 1, exit);

	/*************************************************************************
	* Получение rtengine                                                     *
	*************************************************************************/
	rtEngine = rt_eng_get0_engine();
	assert(rtEngine);

	/*************************************************************************
	* Установка rtengine реализацией по умолчанию                            *
	*************************************************************************/
	r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
	CHECK("  ENGINE_set_default", r == 1, finalize_engine);

	/*************************************************************************
	* Получение ключевой пары                                                *
	*************************************************************************/
	printf("  get_key_pair...\n");
	key = get_key_pair();
	CHECK("  get_key_pair", key != NULL, unregister_engine);

	/*************************************************************************
	* Расшифрование ключа из файла                                           *
	*************************************************************************/
	cekSize = unWrapKey(key, &cek);
	CHECK("  unWrapKey", cekSize > 0, free_key);

	/*************************************************************************
	* Расшифрование шифротекста и его запись в файл                          *
	*************************************************************************/
	r = decryptFile(cek);
	CHECK("  decryptFile", r == 0, free_cek);

	/*************************************************************************
	* Установка признака успешного завершения программы                      *
	*************************************************************************/
	errorCode = 0;
free_cek:

	/*************************************************************************
	* Освобождение ключа шифрования содержимого                              *
	*************************************************************************/
	OPENSSL_secure_free(cek);
free_key:

	/*************************************************************************
	* Освобождение описателя ключевой пары                                   *
	*************************************************************************/
	printf("  free_key_pair...\n");
	r = free_key_pair(key);
	CHECK_RELEASE("  free_key_pair", r == 0, errorCode);
unregister_engine:

	/*************************************************************************
	* Разрегистрация rtengine из OpenSSL                                     *
	*************************************************************************/
	ENGINE_unregister_pkey_asn1_meths(rtEngine);
	ENGINE_unregister_pkey_meths(rtEngine);
	ENGINE_unregister_digests(rtEngine);
	ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

	/*************************************************************************
	* Деинициализация rtengine                                               *
	*************************************************************************/
	r = rt_eng_final();
	CHECK_RELEASE("  rt_eng_final", r == 1, errorCode);
exit:
	CRYPTO_secure_malloc_done();
	if (errorCode) {
		printf("\n\nSample has failed. Some error has occurred.\n");
	} else {
		printf("\n\nSample has been completed successfully.\n");
	}
	return errorCode;
}
