/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2018, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
* Пример работы с Рутокен ЭЦП при помощи библиотеки PKCS#11 на языке C   *
*------------------------------------------------------------------------*
* Использование команды получения информации о сертификате на токене:    *
*  - установление соединения с Рутокен ЭЦП в первом доступном слоте;     *
*  - выполнение аутентификации Пользователя;                             *
*  - импорт сертификата на Рутокен;                                      *
*  - сброс прав доступа Пользователя и закрытие соединения с Рутокен.    *
*------------------------------------------------------------------------*
* В примере используется ключевая пара из                                *
* CreateCSR-PKCS10-GOST34.10-2001, и необходимо с помощью запроса        *
* из проекта CreateCSR-PKCS10-GOST34.10-2001 получить сертификат в       *
* der кодировке. Сертификат можно получить в любом УЦ. Полученный        *
* сертификат необходимо назвать "cert_2001.cer" и положить в папку,      *
* содержащую исполняемый файл примера.                                   *
*************************************************************************/

#include <Common.h>

/*************************************************************************
* Шаблон для импорта сертификата                                         *
*************************************************************************/
CK_ATTRIBUTE certificateTemplate[] =
{
	{ CKA_VALUE, 0, 0 },                                                               // Значение сертификата (заполняется в процессе работы)
	{ CKA_CLASS, &certificateObject, sizeof(certificateObject) },                      // Класс - сертификат
	{ CKA_ID, &keyPairIdGost2001_1, sizeof(keyPairIdGost2001_1) - 1 },                 // Идентификатор сертификата
	{ CKA_TOKEN, &attributeTrue, sizeof(attributeTrue) },                              // Сертификат является объектом токена
	{ CKA_PRIVATE, &attributeFalse, sizeof(attributeFalse) },                          // Сертификат доступен без аутентификации
	{ CKA_CERTIFICATE_TYPE, &certificateType, sizeof(certificateType) },               // Тип сертификата - X.509
	{ CKA_CERTIFICATE_CATEGORY, &tokenUserCertificate, sizeof(tokenUserCertificate) }, // Категория сертификата - пользовательский
};

int main(void)
{
	HMODULE module;                                   // Хэндл загруженной библиотеки PKCS#11
	CK_SESSION_HANDLE session;                        // Хэндл открытой сессии

	CK_FUNCTION_LIST_PTR functionList;                // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
	CK_C_GetFunctionList getFunctionList;             // Указатель на функцию C_GetFunctionList

	CK_SLOT_ID_PTR slots;                             // Указатель на массив идентификаторов слотов
	CK_ULONG slotCount;                               // Количество идентификаторов слотов в массиве

	FILE* certFile;                                   // Описатель потока ввода сертификата
	CK_BYTE_PTR certDer;                              // Массив с сертификатом в DER формате
	CK_ULONG certSize;                                // Размер массива сертификата

	CK_OBJECT_HANDLE certificate;                     // Хэндл сертификата

	CK_RV rv;                                         // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
	int r;                                            // Код возврата для функций, возвращающих int

	int errorCode = 1;                                // Флаг ошибки

	/*************************************************************************
	* Выполнить действия для начала работы с библиотекой PKCS#11             *
	*************************************************************************/
	printf("Initialization...\n");

	/*************************************************************************
	* Загрузить библиотеку                                                   *
	*************************************************************************/
	module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
	CHECK(" LoadLibrary", module != NULL, exit);

	/*************************************************************************
	* Получить адрес функции запроса структуры с указателями на функции      *
	*************************************************************************/
	getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
	CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

	/*************************************************************************
	* Получить структуру с указателями на функции                            *
	*************************************************************************/
	rv = getFunctionList(&functionList);
	CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

	/*************************************************************************
	* Инициализировать библиотеку                                            *
	*************************************************************************/
	rv = functionList->C_Initialize(NULL_PTR);
	CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

	/*************************************************************************
	* Получить количество слотов c подключенными токенами                    *
	*************************************************************************/
	rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
	CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

	CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

	/*************************************************************************
	* Получить список слотов c подключенными токенами                        *
	*************************************************************************/
	slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
	CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

	rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
	CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
	printf(" Slots available: %d\n", (int)slotCount);

	/*************************************************************************
	* Открыть RW сессию в первом доступном слоте                             *
	*************************************************************************/
	rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
	CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

	/*************************************************************************
	* Выполнить аутентификацию Пользователя                                  *
	*************************************************************************/
	rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
	CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
	printf("Initialization has been completed successfully.\n");

	/*************************************************************************
	* Импорт сертификата                                                     *
	*************************************************************************/
	printf("\nImport certificate... \n");

	/*************************************************************************
	* Открыть поточный ввод сертификата из файла                             *
	*************************************************************************/
	certFile = fopen("cert_2001.cer", "rb");
	CHECK_AND_LOG(" fopen", certFile != NULL, "\"cert_2001.cer\" doesn't exist", logout);

	/*************************************************************************
	* Определить размер файла, содержащего сертификат                        *
	*************************************************************************/
	r = fseek(certFile, 0, SEEK_END);
	CHECK(" fseek", r == 0, close_certFile);
	certSize = ftell(certFile);
	CHECK(" ftell", certSize > 0, close_certFile);
	r = fseek(certFile, 0, SEEK_SET);
	CHECK(" fseek", r == 0, close_certFile);

	/*************************************************************************
	* Выделить память для сертификата                                        *
	*************************************************************************/
	certDer = (CK_BYTE_PTR)malloc(certSize);
	CHECK(" malloc", certDer != NULL, close_certFile);

	/*************************************************************************
	* Прочитать сертификат                                                   *
	*************************************************************************/
	r = fread(certDer, 1, (int)certSize, certFile);
	CHECK(" fread", r == (int)certSize, free_certificate);

	/*************************************************************************
	* Задать шаблон сертификата для импорта                                  *
	*************************************************************************/
	certificateTemplate[0].pValue = certDer;
	certificateTemplate[0].ulValueLen = certSize;

	/*************************************************************************
	* Создание сертификата на токене                                         *
	*************************************************************************/
	rv = functionList->C_CreateObject(session, certificateTemplate, arraysize(certificateTemplate),
	                                  &certificate);
	CHECK_AND_LOG(" C_CreateObject", rv == CKR_OK, rvToStr(rv), free_certificate);

	printf("Certificate has been created successfully\n");

	/*************************************************************************
	* Выставить признак успешного завершения программы                       *
	*************************************************************************/
	errorCode = 0;

	/*************************************************************************
	* Выполнить действия для завершения работы с библиотекой PKCS#11         *
	*************************************************************************/
	printf("\nFinalizing... \n");

	/*************************************************************************
	* Очистить память из-под строки с сертификатом                           *
	*************************************************************************/
free_certificate:
	free(certDer);

	/*************************************************************************
	* Закрыть поток ввода сертификата                                        *
	*************************************************************************/
close_certFile:
	r = fclose(certFile);
	CHECK_RELEASE(" fclose", r == 0, errorCode);

	/*************************************************************************
	* Сбросить права доступа                                                 *
	*************************************************************************/
logout:
	rv = functionList->C_Logout(session);
	CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Закрыть открытую сессию в слоте                                        *
	*************************************************************************/
close_session:
	rv = functionList->C_CloseSession(session);
	CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Очистить память из-под слотов                                          *
	*************************************************************************/
free_slots:
	free(slots);

	/*************************************************************************
	* Деинициализировать библиотеку                                          *
	*************************************************************************/
finalize_pkcs11:
	rv = functionList->C_Finalize(NULL_PTR);
	CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Выгрузить библиотеку из памяти                                         *
	*************************************************************************/
unload_pkcs11:
	CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
	if (errorCode) {
		printf("\n\nSome error occurred. Sample failed.\n");
	} else {
		printf("\n\nSample has been completed successfully.\n");
	}

	return errorCode;
}
