/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2018, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
* Пример работы с Рутокен ЭЦП при помощи библиотеки PKCS#11 на языке C   *
*------------------------------------------------------------------------*
* Использование команды проверки подписанных данных ключевой парой       *
* ГОСТ Р 34.10-2001 в формате PKCS#7:                                    *
*  - установление соединения с Рутокен ЭЦП в первом доступном слоте;     *
*  - выполнение аутентификации Пользователя;                             *
*  - проверка подписанного сообщения в формате PKCS#7, которое не хранит *
*    исходные данные;                                                    *
*  - сброс прав доступа Пользователя и закрытие соединения с Рутокен.    *
*                                                                        *
* На данный момент функции C_EX_PKCS7VerifyInit, C_EX_PKCS7VerifyUpdate  *
* и C_EX_PKCS7VerifyFinal поддерживают только алгоритм ГОСТ Р 34.10-2001 *
*------------------------------------------------------------------------*
* Пример использует объекты, созданные в памяти примерами                *
* ImportCertificate-GOST34.10-2001 и SignPKCS7Detached-GOST34.10-2001.   *
* Также необходимо предоставить сертификат УЦ, в котором был выписан     *
* сертификат в примере ImportCertificate-GOST34.10-2001, в der           *
* кодировке. Файл, содержащий сертификат, необходимо назвать             *
* "CA_cert.cer" и положить в папку, содержащую исполняемый файл примера. *
*************************************************************************/

#include <Common.h>

/*************************************************************************
* Исходные данные, которые были подписаны                                *
*************************************************************************/
CK_BYTE data[] =
{
	0x01, 0x00, 0x02, 0x35, 0x35,
	0x02, 0x00, 0x01, 0x01,
	0x81, 0x00, 0x09, 0x34, 0x30, 0x34, 0x34, 0x34, 0x35, 0x39, 0x39, 0x38,
	0x82, 0x00, 0x0A, 0x37, 0x37, 0x38, 0x31, 0x35, 0x36, 0x34, 0x36, 0x31, 0x31,
	0x83, 0x00, 0x13, 0x41, 0x6B, 0x74, 0x69, 0x76, 0x20, 0x52, 0x75, 0x74, 0x6F, 0x6B, 0x65, 0x6E, 0x20, 0x42, 0x61, 0x6E, 0x6B, 0x2E,
	0x84, 0x00, 0x14, 0x34, 0x37, 0x37, 0x37, 0x38, 0x38, 0x38, 0x39, 0x39, 0x39, 0x31, 0x31, 0x31, 0x31, 0x31, 0x32, 0x32, 0x32, 0x37, 0x36,
	0x85, 0x00, 0x0A, 0x33, 0x32, 0x32, 0x38, 0x37, 0x33, 0x36, 0x37, 0x36, 0x35,
	0x86, 0x00, 0x03, 0x52, 0x55, 0x42,
	0xFF, 0x00, 0x0D, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30
};

int main(void)
{
	HMODULE module;                                    // Хэндл загруженной библиотеки PKCS#11
	CK_SESSION_HANDLE session;                         // Хэндл открытой сессии

	CK_FUNCTION_LIST_PTR functionList;                 // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
	CK_C_GetFunctionList getFunctionList;              // Указатель на функцию C_GetFunctionList

	CK_FUNCTION_LIST_EXTENDED_PTR functionListEx;      // Указатель на список функций расширения PKCS#11, хранящийся в структуре CK_FUNCTION_LIST_EXTENDED
	CK_C_EX_GetFunctionListExtended getFunctionListEx; // Указатель на функцию C_EX_GetFunctionListExtended

	FILE* inputCms;                                    // Описатель потока ввода CMS
	FILE* inputCaCert;                                 // Описатель потока ввода сертификата УЦ

	CK_SLOT_ID_PTR slots;                              // Указатель на массив идентификаторов слотов
	CK_ULONG slotCount;                                // Количество идентификаторов слотов в массиве

	CK_BYTE_PTR cms;                                   // Указатель на CMS
	CK_ULONG cmsSize;                                  // Размер CMS, в байтах

	CK_VENDOR_BUFFER rootCertByteArrays[1];            // Массив буферов, содержащих сертификаты в виде массива байт
	CK_ULONG rootCertByteArraysCount = 1;              // Количество буферов, содержащих сертификаты в виде массива байт

	CK_VENDOR_X509_STORE store;                        // структура, содержащая указатели на необходимые для проверки подписи сертификаты

	CK_VENDOR_BUFFER_PTR signerCertificates;           // Указатель на массив буферов, содержащих сертификаты, использованные при подписи
	CK_ULONG signerCertificatesCount;                  // Количество элементов в массиве буферов, содержащих сертификаты, использованные при подписи

	CK_ULONG i;                                        // Счетчик

	CK_RV rv;                                          // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
	int r;                                             // Код возврата для функций, возвращающих int

	int errorCode = 1;                                 // Флаг ошибки

	/*************************************************************************
	* Выполнить действия для начала работы с библиотекой PKCS#11             *
	*************************************************************************/
	printf("Initialization...\n");

	/*************************************************************************
	* Загрузить библиотеку                                                   *
	*************************************************************************/
	module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
	CHECK(" LoadLibrary", module != NULL, exit);

	/*************************************************************************
	* Получить адрес функции запроса структуры с указателями на функции      *
	*************************************************************************/
	getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
	CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

	/*************************************************************************
	* Получить адрес функции запроса структуры с указателями на функции      *
	* расширения стандарта PKCS#11                                           *
	*************************************************************************/
	getFunctionListEx = (CK_C_EX_GetFunctionListExtended)GetProcAddress(module, "C_EX_GetFunctionListExtended");
	CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

	/*************************************************************************
	* Получить структуру с указателями на функции                            *
	*************************************************************************/
	rv = getFunctionList(&functionList);
	CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

	/*************************************************************************
	* Получить структуру с указателями на функции расширения стандарта       *
	*************************************************************************/
	rv = getFunctionListEx(&functionListEx);
	CHECK_AND_LOG(" Get function list extended", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

	/*************************************************************************
	* Инициализировать библиотеку                                            *
	*************************************************************************/
	rv = functionList->C_Initialize(NULL_PTR);
	CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

	/*************************************************************************
	* Получить количество слотов c подключенными токенами                    *
	*************************************************************************/
	rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
	CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

	CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

	/*************************************************************************
	* Получить список слотов c подключенными токенами                        *
	*************************************************************************/
	slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
	CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

	rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
	CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
	printf(" Slots available: %d\n", (int)slotCount);

	/*************************************************************************
	* Открыть RW сессию в первом доступном слоте                             *
	*************************************************************************/
	rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
	CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

	/*************************************************************************
	* Выполнить аутентификацию Пользователя                                  *
	*************************************************************************/
	rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
	CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
	printf("Initialization has been completed successfully.\n");

	/*************************************************************************
	* Чтение CMS                                                             *
	*************************************************************************/
	printf("\nReading CMS...\n");

	/*************************************************************************
	* Открыть поточный ввод CMS из файла                                     *
	*************************************************************************/
	inputCms = fopen("cms_detached_data", "rb");
	CHECK_AND_LOG(" fopen", inputCms != NULL, "\"cms_detached_data\" doesn't exist", logout);

	/*************************************************************************
	* Определить размер файла                                                *
	*************************************************************************/
	r = fseek(inputCms, 0, SEEK_END);
	CHECK(" fseek", r == 0, close_inputCms);
	cmsSize = ftell(inputCms);
	CHECK(" ftell", cmsSize > 0, close_inputCms);
	r = fseek(inputCms, 0, SEEK_SET);
	CHECK(" fseek", r == 0, close_inputCms);

	/*************************************************************************
	* Выделить память для CMS                                                *
	*************************************************************************/
	cms = (CK_BYTE_PTR)malloc(cmsSize);
	CHECK(" malloc", cms != NULL, close_inputCms);

	/*************************************************************************
	* Прочитать CMS                                                          *
	*************************************************************************/
	r = fread(cms, 1, (int)cmsSize, inputCms);
	CHECK(" fread", r == (int)cmsSize, free_cms);

	/*************************************************************************
	* Чтение сертификата УЦ                                                  *
	*************************************************************************/
	printf("\nReading CA certificate...\n");

	/*************************************************************************
	* Открыть поточный ввод сертификата УЦ из файла                          *
	*************************************************************************/
	inputCaCert = fopen("CA_cert.cer", "rb");
	CHECK_AND_LOG(" fopen", inputCaCert != NULL, "\"CA_cert.cer\" doesn't exist", free_cms);

	/*************************************************************************
	* Определить размер файла                                                *
	*************************************************************************/
	r = fseek(inputCaCert, 0, SEEK_END);
	CHECK(" fseek", r == 0, close_input_ca_cert);
	rootCertByteArrays[0].ulSize = ftell(inputCaCert);
	CHECK(" ftell", rootCertByteArrays[0].ulSize > 0, close_input_ca_cert);
	r = fseek(inputCaCert, 0, SEEK_SET);
	CHECK(" fseek", r == 0, close_input_ca_cert);

	/*************************************************************************
	* Выделить память для сертификата УЦ                                     *
	*************************************************************************/
	rootCertByteArrays[0].pData = (CK_BYTE_PTR)malloc(rootCertByteArrays[0].ulSize);
	CHECK(" malloc", rootCertByteArrays[0].pData != NULL, close_input_ca_cert);

	/*************************************************************************
	* Прочитать сертификат УЦ                                                *
	*************************************************************************/
	r = fread(rootCertByteArrays[0].pData, 1, (int)rootCertByteArrays[0].ulSize, inputCaCert);
	CHECK(" fread", r == (int)rootCertByteArrays[0].ulSize, free_certificate);

	/*************************************************************************
	* Заполнить структуру CK_VENDOR_X509_STORE                               *
	*************************************************************************/
	store.pTrustedCertificates = rootCertByteArrays;           // указатель на массив доверенных сертификатов
	store.ulTrustedCertificateCount = rootCertByteArraysCount; // количество доверенных сертификатов в массиве
	store.pCertificates = NULL_PTR;                            // указатель на массив, содержащий сертификаты для проверки подписи
	store.ulCertificateCount = 0;                              // количество сертификатов в цепочке сертификатов
	store.pCrls = NULL_PTR;                                    // указатель на массив списков отзыва сертификатов
	store.ulCrlCount = 0;                                      // количество списков отзыва сертификатов в массиве

	/*************************************************************************
	* Проверка подписи                                                       *
	*************************************************************************/
	printf("\nVerifying...\n");

	/*************************************************************************
	* Инициализировать операцию проверки подписи                             *
	*************************************************************************/
	rv = functionListEx->C_EX_PKCS7VerifyInit(session, cms, cmsSize, &store, OPTIONAL_CRL_CHECK, 0);
	CHECK_AND_LOG(" C_EX_PKCS7VerifyInit", rv == CKR_OK, rvToStr(rv), free_certificate);

	/*************************************************************************
	* Добавить данные, для которых была сформирована подпись                 *
	*************************************************************************/
	rv = functionListEx->C_EX_PKCS7VerifyUpdate(session, data, sizeof(data));
	CHECK_AND_LOG(" C_EX_PKCS7VerifyUpdate", rv == CKR_OK, rvToStr(rv), free_certificate);

	/*************************************************************************
	* Проверить подпись                                                      *
	*************************************************************************/
	rv = functionListEx->C_EX_PKCS7VerifyFinal(session, &signerCertificates, &signerCertificatesCount);
	CHECK_AND_LOG(" C_EX_PKCS7VerifyFinal", rv == CKR_OK, rvToStr(rv), free_certificate);

	/*************************************************************************
	* Распечатать буферы, содержащие сертификаты, использованные при подписи *
	*************************************************************************/
	for (i = 0; i < signerCertificatesCount; ++i) {
		printf(" %lu signer certificate's data is:\n", i + 1);
		printHex(signerCertificates[i].pData, signerCertificates[i].ulSize);
	}

	printf("Data has been verified successfully.\n");

	/*************************************************************************
	* Выставить признак успешного завершения программы                       *
	*************************************************************************/
	errorCode = 0;

	/*************************************************************************
	* Выполнить действия для завершения работы с библиотекой PKCS#11         *
	*************************************************************************/
	printf("\nFinalizing... \n");

	/*************************************************************************
	* Очистить память, содержащую сертификаты, использованные при подписи    *
	*************************************************************************/
	for (i = 0; i < signerCertificatesCount; ++i) {
		rv = functionListEx->C_EX_FreeBuffer((CK_BYTE_PTR)signerCertificates[i].pData);
		CHECK_RELEASE_AND_LOG(" C_EX_FreeBuffer", rv == CKR_OK, rvToStr(rv), errorCode);
	}

	rv = functionListEx->C_EX_FreeBuffer((CK_BYTE_PTR)signerCertificates);
	CHECK_RELEASE_AND_LOG(" C_EX_FreeBuffer", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Очистить память, содержащую массив байт с сертификатом                 *
	*************************************************************************/
free_certificate:
	free(rootCertByteArrays[0].pData);

	/*************************************************************************
	* Закрыть поток ввода сертификата УЦ                                     *
	*************************************************************************/
close_input_ca_cert:
	r = fclose(inputCaCert);
	CHECK_RELEASE(" fclose", r == 0, errorCode);

	/*************************************************************************
	* Освободить память, выделенную для CMS                                  *
	*************************************************************************/
free_cms:
	free(cms);

	/*************************************************************************
	* Закрыть поток ввода CMS                                                *
	*************************************************************************/
close_inputCms:
	r = fclose(inputCms);
	CHECK_RELEASE(" fclose", r == 0, errorCode);

	/*************************************************************************
	* Сбросить права доступа                                                 *
	*************************************************************************/
logout:
	rv = functionList->C_Logout(session);
	CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Закрыть открытую сессию в слоте                                        *
	*************************************************************************/
close_session:
	rv = functionList->C_CloseSession(session);
	CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Очистить память из-под слотов                                          *
	*************************************************************************/
free_slots:
	free(slots);

	/*************************************************************************
	* Деинициализировать библиотеку                                          *
	*************************************************************************/
finalize_pkcs11:
	rv = functionList->C_Finalize(NULL_PTR);
	CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Выгрузить библиотеку из памяти                                         *
	*************************************************************************/
unload_pkcs11:
	CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
	if (errorCode) {
		printf("\n\nSome error occurred. Sample failed.\n");
	} else {
		printf("\n\nSample has been completed successfully.\n");
	}
	return errorCode;
}
