/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2018, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
* Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
*------------------------------------------------------------------------*
* Использование команд вычисления/проверки ЭП на ключах                  *
* ГОСТ Р 34.10-2012 для длины ключа 256 бит:                             *
*  - установление соединения с Рутокен в первом доступном слоте;         *
*  - выполнение аутентификации Пользователя;                             *
*  - формирование цифровой подписи сообщения на демонстрационном ключе;  *
*  - проверка цифровой подписи на демонстрационном ключе;                *
*  - сброс прав доступа Пользователя на Рутокен и закрытие соединения    *
*    с Рутокен.                                                          *
*------------------------------------------------------------------------*
* Пример использует объекты, созданные в памяти Рутокен примером         *
* CreateGOST34.10-2012-256.                                              *
*************************************************************************/

#include <Common.h>

/*************************************************************************
* Данные для цифровой подписи                                            *
*************************************************************************/
CK_BYTE data[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	               0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	               0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	               0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07 };

/*************************************************************************
* Шаблон для поиска закрытого ключа для цифровой подписи                 *
*************************************************************************/
CK_ATTRIBUTE privateKeyTemplate[] =
{
	{ CKA_ID, &keyPairIdGost2012_256_1, sizeof(keyPairIdGost2012_256_1) - 1 }, // ID пары
	{ CKA_CLASS, &privateKeyObject, sizeof(privateKeyObject)},                 // Класс - закрытый ключ
};

/*************************************************************************
* Шаблон для поиска открытого ключа для проверки цифровой подписи        *
*************************************************************************/
CK_ATTRIBUTE publicKeyTemplate[] =
{
	{ CKA_ID, &keyPairIdGost2012_256_1, sizeof(keyPairIdGost2012_256_1) - 1 }, // ID пары
	{ CKA_CLASS, &publicKeyObject, sizeof(publicKeyObject) },                  // Класс - открытый ключ
};

int main(void)
{
	HMODULE module;                                  // Хэндл загруженной библиотеки PKCS#11
	CK_SESSION_HANDLE session;                       // Хэндл открытой сессии

	CK_FUNCTION_LIST_PTR functionList;               // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
	CK_C_GetFunctionList getFunctionList;            // Указатель на функцию C_GetFunctionList

	CK_SLOT_ID_PTR slots;                            // Массив идентификаторов слотов
	CK_ULONG slotCount;                              // Количество идентификаторов слотов в массиве

	CK_MECHANISM_TYPE_PTR mechanisms;                // Массив поддерживаемых механизмов
	CK_ULONG mechanismCount;                         // Количество поддерживаемых механизмов

	CK_OBJECT_HANDLE_PTR objects1;                   // Массив хэндлов объектов, соответствующих критериям поиска
	CK_ULONG objectCount1;                           // Количество хэндлов объектов в массиве

	CK_OBJECT_HANDLE_PTR objects2;                   // Массив хэндлов объектов, соответствующих критериям поиска
	CK_ULONG objectCount2;                           // Количество хэндлов объектов в массиве

	CK_BYTE_PTR signature;                           // Указатель на буфер, содержащий цифровую подпись для данных
	CK_ULONG signatureSize;                          // Размер буфера, содержащего цифровую подпись для данных, в байтах

	CK_BYTE_PTR hash1;                               // Указатель на временный буфер для хэш-кода от данных
	CK_BYTE_PTR hash2;                               // Указатель на временный буфер для хэш-кода от данных

	CK_ULONG hashSize1;                              // Размер временного буфера в байтах
	CK_ULONG hashSize2;                              // Размер временного буфера в байтах

	CK_RV rv;                                        // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
	int r;                                           // Код возврата для функций, возвращающих int

	CK_ULONG i;                                      // Вспомогательная переменная-счетчик в циклах

	int isGostR3410Supported = 0;                    // Флаги для проверки поддержки токеном CKM_GOSTR3410
	int isGostR3411_2012_256Supported = 0;           // Флаги для проверки поддержки токеном CKM_GOSTR3411_12_256

	int errorCode = 1;                               // Флаг ошибки

	/*************************************************************************
	* Выполнить действия для начала работы с библиотекой PKCS#11             *
	*************************************************************************/
	printf("Initialization...\n");

	/*************************************************************************
	* Загрузить библиотеку                                                   *
	*************************************************************************/
	module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
	CHECK(" LoadLibrary", module != NULL, exit);

	/*************************************************************************
	* Получить адрес функции запроса структуры с указателями на функции      *
	*************************************************************************/
	getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
	CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

	/*************************************************************************
	* Получить структуру с указателями на функции                            *
	*************************************************************************/
	rv = getFunctionList(&functionList);
	CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

	/*************************************************************************
	* Инициализировать библиотеку                                            *
	*************************************************************************/
	rv = functionList->C_Initialize(NULL_PTR);
	CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

	/*************************************************************************
	* Получить количество слотов c подключенными токенами                    *
	*************************************************************************/
	rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
	CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

	CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

	/*************************************************************************
	* Получить список слотов c подключенными токенами                        *
	*************************************************************************/
	slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
	CHECK(" Memory allocation for slots", slots != NULL, finalize_pkcs11);

	rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
	CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
	printf(" Slots available: %d\n", (int)slotCount);

	/*************************************************************************
	* Получить список поддерживаемых токеном механизмов                      *
	*************************************************************************/
	rv = functionList->C_GetMechanismList(slots[0], NULL_PTR, &mechanismCount);
	CHECK_AND_LOG(" C_GetMechanismList (number of mechanisms)", rv == CKR_OK, rvToStr(rv), free_slots);

	CHECK_AND_LOG(" Checking mechanisms available", mechanismCount > 0, " No mechanisms available", free_slots);

	mechanisms = (CK_MECHANISM_TYPE_PTR)malloc(mechanismCount * sizeof(CK_MECHANISM_TYPE));
	CHECK(" Memory allocation for mechanisms", mechanisms != NULL_PTR, free_slots);

	rv = functionList->C_GetMechanismList(slots[0], mechanisms, &mechanismCount);
	CHECK_AND_LOG(" C_GetMechanismList", rv == CKR_OK, rvToStr(rv), free_mechanisms);

	/*************************************************************************
	* Определение поддерживаемых токеном механизмов                          *
	*************************************************************************/
	for (i = 0; i < mechanismCount; ++i) {
		if (mechanisms[i] == CKM_GOSTR3410) {
			isGostR3410Supported = 1;
			break;
		}
	}

	CHECK_AND_LOG(" Checking CKM_GOSTR3410 support", isGostR3410Supported,
	              "CKM_GOSTR3410 isn`t supported!\n", free_mechanisms);

	for (i = 0; i < mechanismCount; ++i) {
		if (mechanisms[i] == CKM_GOSTR3411_12_256) {
			isGostR3411_2012_256Supported = 1;
			break;
		}
	}

	CHECK_AND_LOG(" Checking CKM_GOSTR3411_12_256 support", isGostR3411_2012_256Supported,
	              "CKM_GOSTR3411_12_256 isn`t supported!\n", free_mechanisms);

	/*************************************************************************
	* Открыть RW сессию в первом доступном слоте                             *
	*************************************************************************/
	rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
	CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_mechanisms);

	/*************************************************************************
	* Выполнить аутентификацию Пользователя                                  *
	*************************************************************************/
	rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
	CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
	printf("Initialization has been completed successfully.\n");

	/*************************************************************************
	* Сформировать цифровую подпись данных по алгоритму                      *
	* ГОСТ Р 34.10-2012(256)                                                 *
	*************************************************************************/
	printf("\nSigning data...\n");

	/*************************************************************************
	* Получить массив хэндлов закрытых ключей                                *
	*************************************************************************/
	printf(" Getting signing key...\n");

	r = findObjects(functionList, session, privateKeyTemplate, arraysize(privateKeyTemplate),
	                &objects1, &objectCount1);
	CHECK(" findObjects", r == 0, logout);

	CHECK_AND_LOG(" Checking number of keys found", objectCount1 > 0, "No objects found\n", logout);

	/*************************************************************************
	* Вычислить хэш-код данных                                               *
	*************************************************************************/
	printf(" Hashing data...\n");

	/*************************************************************************
	* Инициализировать хэш-функцию                                           *
	*************************************************************************/
	rv = functionList->C_DigestInit(session, &gostR3411_2012_256HashMech);
	CHECK_AND_LOG("  C_DigestInit", rv == CKR_OK, rvToStr(rv), free_objects1);

	/*************************************************************************
	* Определить размер хэш-кода                                             *
	*************************************************************************/
	rv = functionList->C_Digest(session, data, sizeof(data), NULL_PTR, &hashSize1);
	CHECK_AND_LOG("  C_Digest(get size)", rv == CKR_OK, rvToStr(rv), free_objects1);

	/*************************************************************************
	* Вычислить хэш-код данных                                               *
	*************************************************************************/

	hash1 = (CK_BYTE*)malloc(hashSize1 * sizeof(CK_BYTE));
	CHECK("  Memory allocation for hash", hash1 != NULL, free_objects1);

	rv = functionList->C_Digest(session, data, sizeof(data), hash1, &hashSize1);
	CHECK_AND_LOG("  C_Digest (get hash)", rv == CKR_OK, rvToStr(rv), free_hash1);

	/*************************************************************************
	* Распечатать буфер, содержащий хэш-код                                  *
	*************************************************************************/
	printf("  Hashed buffer is: \n");
	printHex(hash1, hashSize1);
	printf(" Hashing has been completed.\n");

	/*************************************************************************
	* Сформировать цифровую подпись данных                                   *
	* по алгоритму ГОСТ Р 34.10-2012(256)                                    *
	*************************************************************************/
	printf(" Signing data...\n");
	/*************************************************************************
	* Инициализировать операцию подписи данных                               *
	*************************************************************************/
	rv = functionList->C_SignInit(session, &gostR3410_2012_256SigVerMech, objects1[0]);
	CHECK_AND_LOG("  C_SignInit", rv == CKR_OK, rvToStr(rv), free_hash1);

	/*************************************************************************
	* Определить размер данных подписи                                       *
	*************************************************************************/
	rv = functionList->C_Sign(session, hash1, hashSize1, NULL_PTR, &signatureSize);
	CHECK_AND_LOG("  C_Sign (get size)", rv == CKR_OK, rvToStr(rv), free_hash1);

	/*************************************************************************
	 * Подписать данные                                                      *
	 ************************************************************************/
	signature = (CK_BYTE*)malloc(signatureSize * sizeof(CK_BYTE));
	CHECK("  Memory allocation for signature", signature != NULL, free_hash1);

	rv = functionList->C_Sign(session, hash1, hashSize1, signature, &signatureSize);
	CHECK_AND_LOG("  C_Sign (signing)", rv == CKR_OK, rvToStr(rv), free_signature);

	/*************************************************************************
	* Распечатать буфер, содержащий подпись                                  *
	*************************************************************************/
	printf("  Signature buffer is: \n");
	printHex(signature, signatureSize);
	printf("Data has been signed successfully.\n");

	/*************************************************************************
	* Выполнить проверку подписи по алгоритму ГОСТ Р 34.10-2012(256)         *
	*************************************************************************/
	printf("\nVerifying signature...\n");

	/*************************************************************************
	* Получить массив хэндлов открытых ключей                                *
	*************************************************************************/
	printf(" Getting key to verify...\n");
	r = findObjects(functionList, session, publicKeyTemplate, arraysize(publicKeyTemplate),
	                &objects2, &objectCount2);
	CHECK(" findObjects", r == 0, free_signature);

	CHECK_AND_LOG(" Checking number of keys found", objectCount2 > 0, "No objects found\n", free_signature);

	/*************************************************************************
	* Вычислить хэш-код от данных                                            *
	*************************************************************************/
	printf(" Hashing data...\n");

	/*************************************************************************
	* Инициализировать хэш-функцию                                           *
	*************************************************************************/
	rv = functionList->C_DigestInit(session, &gostR3411_2012_256HashMech);
	CHECK_AND_LOG("  C_DigestInit", rv == CKR_OK, rvToStr(rv), free_objects2);

	/*************************************************************************
	* Определить размер хэш-кода                                             *
	*************************************************************************/
	rv = functionList->C_Digest(session, data, sizeof(data), NULL_PTR, &hashSize2);
	CHECK_AND_LOG("  C_Digest (get size)", rv == CKR_OK, rvToStr(rv), free_objects2);

	/*************************************************************************
	* Вычислить хэш-код данных                                               *
	*************************************************************************/
	hash2 = (CK_BYTE*)malloc(hashSize2 * sizeof(CK_BYTE));
	CHECK("  Memory allocation for hash", hash2 != NULL, free_objects2);

	rv = functionList->C_Digest(session, data, sizeof(data), hash2, &hashSize2);
	CHECK_AND_LOG("  C_Digest (get hash)", rv == CKR_OK, rvToStr(rv), free_hash2);

	/*************************************************************************
	* Распечатать буфер, содержащий хэш-код                                  *
	*************************************************************************/
	printf("  Hashed buffer is: \n");
	printHex(hash2, hashSize2);
	printf(" Hashing has been completed.\n");

	/*************************************************************************
	* Проверка подписи                                                       *
	*************************************************************************/
	printf(" Verifying data...\n");
	/*************************************************************************
	* Инициализировать операцию проверки подписи                             *
	*************************************************************************/
	rv = functionList->C_VerifyInit(session, &gostR3410_2012_256SigVerMech, objects2[0]);
	CHECK_AND_LOG("  C_VerifyInit", rv == CKR_OK, rvToStr(rv), free_hash2);

	/*************************************************************************
	* Проверить подпись для данных                                           *
	*************************************************************************/
	rv = functionList->C_Verify(session, hash2, hashSize2, signature, signatureSize);
	CHECK_AND_LOG("  C_Verify", rv == CKR_OK, rvToStr(rv), free_hash2);

	printf("Verifying has been completed successfully.\n");

	/*************************************************************************
	* Выставить признак успешного завершения программы                       *
	*************************************************************************/
	errorCode = 0;

	/*************************************************************************
	* Выполнить действия для завершения работы с библиотекой PKCS#11         *
	*************************************************************************/
	printf("\nFinalizing... \n");

	/*************************************************************************
	* Очистить память, выделенную под хэш-коды, подпись и объекты            *
	*************************************************************************/
free_hash2:
	free(hash2);

free_objects2:
	free(objects2);

free_signature:
	free(signature);

free_hash1:
	free(hash1);

free_objects1:
	free(objects1);

	/*************************************************************************
	* Сбросить права доступа                                                 *
	*************************************************************************/
logout:
	rv = functionList->C_Logout(session);
	CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Закрыть открытую сессию в слоте                                        *
	*************************************************************************/
close_session:
	rv = functionList->C_CloseSession(session);
	CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Очистить память, выделенную под механизмы и слоты                      *
	*************************************************************************/
free_mechanisms:
	free(mechanisms);

free_slots:
	free(slots);

	/*************************************************************************
	* Деинициализировать библиотеку                                          *
	*************************************************************************/
finalize_pkcs11:
	rv = functionList->C_Finalize(NULL_PTR);
	CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Выгрузить библиотеку из памяти                                         *
	*************************************************************************/
unload_pkcs11:
	CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
	if (errorCode) {
		printf("\n\nSome error occurred. Sample failed.\n");
	} else {
		printf("\n\nSample has been completed successfully.\n");
	}

	return errorCode;
}
