/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2018, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
* Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
*------------------------------------------------------------------------*
* Использование команд выработки ключа обмена                            *
* и маскирования сессионного ключа:                                      *
*  - установление соединения с Рутокен в первом доступном слоте;         *
*  - выполнение аутентификации Пользователя;                             *
*  - генерация сессионного ключа;                                        *
*  - генерация UKM;                                                      *
*  - выработка ключа обмена;                                             *
*  - маскирование сессионного ключа на выработанном ключе обемена;       *
*  - сброс прав доступа Пользователя на Рутокен и закрытие соединения    *
*    с Рутокен.                                                          *
*------------------------------------------------------------------------*
* Пример использует объекты, созданные в памяти Рутокен примером         *
* CreateGOST34.10-2012-256.                                              *
*************************************************************************/

#include <Common.h>

/*************************************************************************
* Шаблон для поиска закрытого ключа отправителя                          *
*************************************************************************/
CK_ATTRIBUTE privateKeyTemplate[] =
{
	{ CKA_ID, keyPairIdGost2012_256_1, sizeof(keyPairIdGost2012_256_1) - 1},        // ID пары
	{ CKA_CLASS, &privateKeyObject, sizeof(privateKeyObject) },                     // Класс - закрытый ключ
	{ CKA_KEY_TYPE, &keyTypeGostR3410_2012_256, sizeof(keyTypeGostR3410_2012_256)}  // Тип ключа - ГОСТ Р 34.10-2012(256)
};

/*************************************************************************
* Шаблон для создания ключа обмена                                       *
*************************************************************************/
CK_ATTRIBUTE derivedKeyTemplate[] =
{
	{ CKA_LABEL, &derivedKeyLabel, sizeof(derivedKeyLabel) - 1},          // Метка ключа
	{ CKA_CLASS, &secretKeyObject, sizeof(secretKeyObject) },             // Класс - секретный ключ
	{ CKA_KEY_TYPE, &keyTypeGost28147, sizeof(keyTypeGost28147)},         // Тип ключа - ГОСТ 28147-89
	{ CKA_TOKEN, &attributeFalse, sizeof(attributeFalse)},                // Ключ является объектом сессии
	{ CKA_MODIFIABLE, &attributeTrue, sizeof(attributeTrue)},             // Ключ может быть изменен после создания
	{ CKA_PRIVATE, &attributeTrue, sizeof(attributeTrue)},                // Ключ доступен только после аутентификации на токене
	{ CKA_EXTRACTABLE, &attributeTrue, sizeof(attributeTrue)},            // Ключ может быть извлечен в зашифрованном виде
	{ CKA_SENSITIVE, &attributeFalse, sizeof(attributeFalse)}             // Ключ может быть извлечен в открытом виде
};

/*************************************************************************
* Шаблон сессионного ключа                                               *
*************************************************************************/
CK_ATTRIBUTE sessionKeyTemplate[] =
{
	{ CKA_LABEL, &sessionKeyLabel, sizeof(sessionKeyLabel) - 1 },          // Метка ключа
	{ CKA_CLASS, &secretKeyObject, sizeof(secretKeyObject) },              // Класс - секретный ключ
	{ CKA_KEY_TYPE, &keyTypeGost28147, sizeof(keyTypeGost28147)},          // Тип ключа - ГОСТ 28147-89
	{ CKA_TOKEN, &attributeFalse, sizeof(attributeFalse)},                 // Ключ является объектом сессии
	{ CKA_MODIFIABLE, &attributeTrue, sizeof(attributeTrue)},              // Ключ может быть изменен после создания
	{ CKA_PRIVATE, &attributeTrue, sizeof(attributeTrue)},                 // Ключ доступен только после аутентификации на токене
	{ CKA_VALUE, NULL_PTR, 0},                                             // Значение ключа
	{ CKA_EXTRACTABLE, &attributeTrue, sizeof(attributeTrue)},             // Ключ может быть извлечен в зашифрованном виде
	{ CKA_SENSITIVE, &attributeFalse, sizeof(attributeFalse)}              // Ключ может быть извлечен в открытом виде
};

/*************************************************************************
* Размер массива параметров для выработки ключа обмена по алгоритму      *
* VKO GOST R 34.10-2012-256                                              *
* 4 байта  - идентификатор функции диверсификации                        *
* 4 байта  - размер открытого ключа (64 байта)                           *
* 64 байт  - буфер с открытым ключом                                     *
* 4 байта  - размер буфера с UKM (8 байт)                                *
* 8 байт   - буфер с UKM                                                 *
*************************************************************************/
#define DERIVE_PARAMS_256_LENGTH 84

/*************************************************************************
* Параметры для выработки ключа обмена по схеме VKO GOST R 34.10-2012-256*
* Содержат в себе данные по диверсификации ключа, открытый ключ и UKM    *
*************************************************************************/
CK_BYTE deriveParameters2012_256[DERIVE_PARAMS_256_LENGTH] = { 0x00, };

const CK_ULONG keyLengthOffset = 4;       // Смещение длины ключа в массиве
const CK_ULONG publicKeyValueOffset = 8;  // Смещение значения ключа в массиве
const CK_ULONG ukmLengthOffset = 72;      // Смещение длины UKM в массиве
const CK_ULONG ukmDataOffset = 76;        // Смещение UKM в массиве

/*************************************************************************
* Функция записи четырёхбайтного значения длины в буфер                  *
*************************************************************************/
void ulongToBuffer(CK_BYTE_PTR buffer, CK_ULONG value)
{
	buffer[0] = value & 0xFF;
	buffer[1] = (value >> 8) & 0xFF;
	buffer[2] = (value >> 16) & 0xFF;
	buffer[3] = (value >> 24) & 0xFF;
}

/*************************************************************************
* Функция выработки ключа обмена                                         *
*************************************************************************/
static int deriveKey(CK_FUNCTION_LIST_PTR functionList,   // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
                     CK_SESSION_HANDLE session,           // Хэндл сессии
                     CK_ATTRIBUTE_PTR privateKeyTemplate, // Указатель на шаблон для поиска закрытого ключа
                     CK_ULONG privateKeyTemplateSize,     // Количество атрибутов в шаблоне для поиска закрытого ключа
                     CK_BYTE_PTR publicKeyValue,          // Указатель на буфер, содержащий открытый ключ получателя
                     CK_ULONG publicKeyValueSize,         // Размер буфера с открытым ключом получателя
                     CK_BYTE_PTR ukm,                     // Указатель на буфер, содержащий UKM
                     CK_ULONG ukmSize,                    // Размер буфера с UKM
                     CK_OBJECT_HANDLE_PTR derivedKey      // Хэндл выработанного ключа обмена
                     )
{
	CK_OBJECT_HANDLE_PTR privateKeys;                     // Массив хэндлов закрытых ключей
	CK_ULONG privateKeysCount;                            // Количество хэндлов объектов в массиве

	CK_BYTE derivedKeyValue[GOST_28147_KEY_SIZE];         // Буфер, содержащий тело ключа обмена

	CK_ATTRIBUTE valueAttr = { CKA_VALUE, NULL_PTR, 0 };  // Структура данных типа CK_ATTRIBUTE для хранения значения атрибута CKA_VALUE

	CK_RV rv;                                             // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
	int r;                                                // Код возврата для функций, возвращающих int

	int errorCode = 1;                                    // Флаг ошибки

	/*************************************************************************
	* Получить массив хэндлов закрытых ключей                                *
	*************************************************************************/
	printf(" Getting private key...\n");

	r = findObjects(functionList, session, privateKeyTemplate, privateKeyTemplateSize,
	                &privateKeys, &privateKeysCount);
	CHECK(" findObjects", r == 0, exit);

	CHECK_AND_LOG(" Checking number of keys found", privateKeysCount > 0, " No objects found", exit);

	/*************************************************************************
	* Поместить в структуру типа CK_MECHANISM параметры, необходимые         *
	* для выработки ключа обмена                                             *
	*************************************************************************/
	ulongToBuffer(deriveParameters2012_256, CKM_KDF_GOSTR3411_2012_256);
	ulongToBuffer(deriveParameters2012_256 + keyLengthOffset, publicKeyValueSize);
	memcpy(deriveParameters2012_256 + publicKeyValueOffset, publicKeyValue, publicKeyValueSize);
	ulongToBuffer(deriveParameters2012_256 + ukmLengthOffset, ukmSize);
	memcpy(deriveParameters2012_256 + ukmDataOffset, ukm, ukmSize);
	gostR3410_12DerivationMech.pParameter = deriveParameters2012_256;
	gostR3410_12DerivationMech.ulParameterLen = sizeof(deriveParameters2012_256);

	rv = functionList->C_DeriveKey(session, &gostR3410_12DerivationMech, privateKeys[0],
	                               derivedKeyTemplate, arraysize(derivedKeyTemplate), derivedKey);

	CHECK_AND_LOG(" C_DeriveKey", rv == CKR_OK, rvToStr(rv), free_privateKeys);

	/*************************************************************************
	* Получить и распечатать значение выработанного ключа                    *
	*************************************************************************/
	valueAttr.pValue = derivedKeyValue;
	valueAttr.ulValueLen = sizeof(derivedKeyValue);

	rv = functionList->C_GetAttributeValue(session, *derivedKey, &valueAttr, 1);
	CHECK_AND_LOG(" C_GetAttributeValue", rv == CKR_OK, rvToStr(rv), destroy_derivedKey);

	printf(" Derived key value:\n");
	printHex(derivedKeyValue, sizeof(derivedKeyValue));

	errorCode = 0;

	/*************************************************************************
	* Очистить память, выделенную под объекты ключей,                        *
	* уничтожить выработанный ключ (при необходимости)                       *
	*************************************************************************/
destroy_derivedKey:
	if (errorCode != 0) {
		rv = functionList->C_DestroyObject(session, *derivedKey);
		CHECK_RELEASE_AND_LOG(" C_DestroyObject", rv == CKR_OK, rvToStr(rv), errorCode);
		*derivedKey = CK_INVALID_HANDLE;
	}
free_privateKeys:
	free(privateKeys);
exit:
	return errorCode;
}

int main(void)
{
	HMODULE module;                                      // Хэндл загруженной библиотеки PKCS#11
	CK_SESSION_HANDLE session;                           // Хэндл открытой сессии

	CK_FUNCTION_LIST_PTR functionList;                   // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
	CK_C_GetFunctionList getFunctionList;                // Указатель на функцию C_GetFunctionList

	CK_SLOT_ID_PTR slots;                                // Массив идентификаторов слотов
	CK_ULONG slotCount;                                  // Количество идентификаторов слотов в массиве

	CK_MECHANISM_TYPE_PTR mechanisms;                    // Массив поддерживаемых механизмов
	CK_ULONG mechanismCount;                             // Количество поддерживаемых механизмов

	CK_ATTRIBUTE valueAttr = { CKA_VALUE, NULL_PTR, 0 }; // Структура данных типа CK_ATTRIBUTE для хранения значения атрибута CKA_VALUE

	CK_OBJECT_HANDLE derivedKey;                         // Хэндл выработанного ключа обмена

	CK_BYTE publicKeyValue[GOST_3410_12_256_KEY_SIZE] =  // Буфер, содержащий значение шаблонного открытого ключа получателя
	{
		0x76, 0x25, 0x13, 0x0F, 0x19, 0x17, 0x3D, 0x3B, 0x24, 0xCC, 0xA7, 0xC7, 0x72, 0xB3, 0x5D, 0x83,
		0xB0, 0xBB, 0x42, 0xC9, 0x66, 0xD5, 0xC1, 0x5A, 0x0A, 0x9F, 0xD4, 0x24, 0xF0, 0x46, 0xB2, 0xCD,
		0x85, 0xDD, 0xC5, 0x73, 0xAE, 0x72, 0x8D, 0x6F, 0xC8, 0x9C, 0xE2, 0x5B, 0x89, 0x05, 0xE8, 0x9D,
		0x75, 0x93, 0xBF, 0xE9, 0x38, 0xC3, 0x43, 0x27, 0x09, 0x59, 0x7E, 0x7D, 0x51, 0xA8, 0x35, 0x53
	};

	CK_BYTE sessionKeyValue[GOST_28147_KEY_SIZE];        // Буфер, содержащий сессионный ключ

	CK_BYTE_PTR wrappedKeyValue;                         // Указатель на буфер, содержащий маскированный сессионный ключ
	CK_ULONG wrappedKeyValueSize;                        // Размер буфера со значением маскированного сессионного ключа, в байтах

	CK_BYTE ukm[UKM_LENGTH];                             // Буфер, содержащий UKM

	CK_OBJECT_HANDLE sessionKey;                         // Хэндл сессионного ключа, который будет маскироваться

	CK_RV rv;                                            // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
	int r;                                               // Код возврата для функций, возвращающих int

	CK_ULONG i;                                          // Вспомогательная переменная-счетчик в циклах

	int isGostR3410_12DeriveSupported = 0;               // Флаги для проверки поддержки токеном CKM_GOSTR3410_12_DERIVE
	int isGostWrapSupported = 0;                         // Флаги для проверки поддержки токеном CKM_GOST28147_KEY_WRAP

	int errorCode = 1;                                   // Флаг ошибки

	/*************************************************************************
	* Выполнить действия для начала работы с библиотекой PKCS#11             *
	*************************************************************************/
	printf("Initialization...\n");

	/*************************************************************************
	* Загрузить библиотеку                                                   *
	*************************************************************************/
	module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
	CHECK(" LoadLibrary", module != NULL, exit);

	/*************************************************************************
	* Получить адрес функции запроса структуры с указателями на функции      *
	*************************************************************************/
	getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
	CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

	/*************************************************************************
	* Получить структуру с указателями на функции                            *
	*************************************************************************/
	rv = getFunctionList(&functionList);
	CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

	/*************************************************************************
	* Инициализировать библиотеку                                            *
	*************************************************************************/
	rv = functionList->C_Initialize(NULL_PTR);
	CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

	/*************************************************************************
	* Получить количество слотов c подключенными токенами                    *
	*************************************************************************/
	rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
	CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

	CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

	/*************************************************************************
	* Получить список слотов c подключенными токенами                        *
	*************************************************************************/
	slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
	CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

	rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
	CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
	printf(" Slots available: %d\n", (int)slotCount);

	/*************************************************************************
	* Получить список поддерживаемых токеном механизмов                      *
	*************************************************************************/
	rv = functionList->C_GetMechanismList(slots[0], NULL_PTR, &mechanismCount);
	CHECK_AND_LOG(" C_GetMechanismList (number of mechanisms)", rv == CKR_OK, rvToStr(rv), free_slots);

	CHECK_AND_LOG(" Checking mechanisms available", mechanismCount > 0, " No mechanisms available", free_slots);

	mechanisms = (CK_MECHANISM_TYPE_PTR)malloc(mechanismCount * sizeof(CK_MECHANISM_TYPE));
	CHECK(" Memory allocation for mechanisms", mechanisms != NULL_PTR, free_slots);

	rv = functionList->C_GetMechanismList(slots[0], mechanisms, &mechanismCount);
	CHECK_AND_LOG(" C_GetMechanismList", rv == CKR_OK, rvToStr(rv), free_mechanisms);

	/*************************************************************************
	* Определение поддерживаемых токеном механизмов                          *
	*************************************************************************/
	for (i = 0; i < mechanismCount; ++i) {
		if (mechanisms[i] == CKM_GOSTR3410_12_DERIVE) {
			isGostR3410_12DeriveSupported = 1;
			break;
		}
	}

	CHECK_AND_LOG(" Checking CKM_GOSTR3410_12_DERIVE support\n", isGostR3410_12DeriveSupported,
	              "CKM_GOSTR3410_12_DERIVE isn`t supported!\n", free_mechanisms);

	for (i = 0; i < mechanismCount; ++i) {
		if (mechanisms[i] == CKM_GOST28147_KEY_WRAP) {
			isGostWrapSupported = 1;
			break;
		}
	}

	CHECK_AND_LOG(" Checking CKM_GOST28147_KEY_WRAP support", isGostWrapSupported,
	              " CKM_GOST28147_KEY_WRAP isn`t supported!\n", free_mechanisms);

	/*************************************************************************
	* Открыть RW сессию в первом доступном слоте                             *
	*************************************************************************/
	rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
	CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_mechanisms);

	/*************************************************************************
	* Выполнить аутентификацию Пользователя                                  *
	*************************************************************************/
	rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
	CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
	printf("Initialization has been completed successfully.\n");

	/*************************************************************************
	* Выполнить предварительные действия для выработки ключа и маскирования  *
	*************************************************************************/
	printf("\nPreparing data for deriving and wrapping...\n");

	/*************************************************************************
	* Установить параметры в структуре типа CK_GOSTR3410_DERIVE_PARAMS       *
	* для выработки ключа обмена                                             *
	*************************************************************************/
	rv = functionList->C_GenerateRandom(session, ukm, sizeof(ukm));
	CHECK_AND_LOG(" C_GenerateRandom", rv == CKR_OK, rvToStr(rv), logout);

	/*************************************************************************
	* Для маскирования ключа должно использоваться то же значение UKM,       *
	* что и для выработки                                                    *
	*************************************************************************/
	gost28147WrapMech.ulParameterLen = sizeof(ukm);
	gost28147WrapMech.pParameter = ukm;

	/*************************************************************************
	* Сгенерировать значение сессионного ключа                               *
	*************************************************************************/
	rv = functionList->C_GenerateRandom(session, sessionKeyValue, sizeof(sessionKeyValue));
	CHECK_AND_LOG(" C_GenerateRandom", rv == CKR_OK, rvToStr(rv), logout);

	for (i = 0; i < arraysize(sessionKeyTemplate); ++i) {
		if (sessionKeyTemplate[i].type == CKA_VALUE) {
			sessionKeyTemplate[i].pValue = sessionKeyValue;
			sessionKeyTemplate[i].ulValueLen = sizeof(sessionKeyValue);
			break;
		}
	}

	printf(" Session key data is:\n");
	printHex(sessionKeyValue, sizeof(sessionKeyValue));

	printf("Preparing has been completed successfully.\n");

	/*************************************************************************
	* Выработать ключ обмена                                                 *
	*************************************************************************/
	printf("\nDeriving key...\n");

	r = deriveKey(functionList, session, privateKeyTemplate, arraysize(privateKeyTemplate),
	              publicKeyValue, sizeof(publicKeyValue), ukm, sizeof(ukm), &derivedKey);
	CHECK(" deriveKey", r == 0, logout);

	printf("Key has been derived successfully.\n");

	/*************************************************************************
	* Маскировать сессионный ключ с помощью выработанного ключа обмена       *
	*************************************************************************/
	printf("\nWrapping key...\n");

	/*************************************************************************
	* Создать ключ, который будет замаскирован                               *
	*************************************************************************/
	printf(" Creating the GOST 28147-89 key to wrap...\n");
	rv = functionList->C_CreateObject(session, sessionKeyTemplate, arraysize(sessionKeyTemplate), &sessionKey);
	CHECK_AND_LOG("  C_CreateObject", rv == CKR_OK, rvToStr(rv), destroy_derived);

	/*************************************************************************
	* Замаскировать ключ                                                     *
	*************************************************************************/
	rv = functionList->C_WrapKey(session, &gost28147WrapMech, derivedKey, sessionKey, NULL_PTR, &wrappedKeyValueSize);
	CHECK_AND_LOG("  C_WrapKey", rv == CKR_OK, rvToStr(rv), destroy_sessionKey);

	wrappedKeyValue = (CK_BYTE_PTR)malloc(wrappedKeyValueSize * sizeof(CK_BYTE));
	CHECK("  Memory allocation for wrapped key", wrappedKeyValue != NULL_PTR, destroy_sessionKey);

	rv = functionList->C_WrapKey(session, &gost28147WrapMech, derivedKey, sessionKey, wrappedKeyValue, &wrappedKeyValueSize);
	CHECK_AND_LOG("  C_WrapKey", rv == CKR_OK, rvToStr(rv), free_wrapped);

	/*************************************************************************
	* Распечатать буфер, содержащий маскированный ключ                       *
	*************************************************************************/
	printf(" Wrapped key data is:\n");
	printHex(wrappedKeyValue, wrappedKeyValueSize);

	printf("Key has been wrapped successfully.\n");

	/*************************************************************************
	* Выставить признак успешного завершения программы                       *
	*************************************************************************/
	errorCode = 0;

	/*************************************************************************
	* Выполнить действия для завершения работы с библиотекой PKCS#11         *
	*************************************************************************/
	printf("\nFinalizing... \n");
free_wrapped:
	free(wrappedKeyValue);
destroy_sessionKey:
	rv = functionList->C_DestroyObject(session, sessionKey);
	CHECK_RELEASE_AND_LOG(" C_DestroyObject (sessionKey)", rv == CKR_OK, rvToStr(rv), errorCode);
destroy_derived:
	rv = functionList->C_DestroyObject(session, derivedKey);
	CHECK_RELEASE_AND_LOG(" C_DestroyObject (derivedKey)", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Сбросить права доступа                                                 *
	*************************************************************************/
logout:
	rv = functionList->C_Logout(session);
	CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Закрыть открытую сессию в слоте                                        *
	*************************************************************************/
close_session:
	rv = functionList->C_CloseSession(session);
	CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Очистить память, выделенную под слоты и механизмы                      *
	*************************************************************************/
free_mechanisms:
	free(mechanisms);

free_slots:
	free(slots);

	/*************************************************************************
	* Деинициализировать библиотеку                                          *
	*************************************************************************/
finalize_pkcs11:
	rv = functionList->C_Finalize(NULL_PTR);
	CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Выгрузить библиотеку из памяти                                         *
	*************************************************************************/
unload_pkcs11:
	CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
	if (errorCode) {
		printf("\n\nSome error occurred. Sample failed.\n");
	} else {
		printf("\n\nSample has been completed successfully.\n");
	}

	return errorCode;
}
