/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2018, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
* Данный пример демонстрирует работу с ключами на токене с помощью       *
* pkicore. Во время выполнения программы в консоль выводятся             *
* идентификаторы найденных ключей. Открытые ключи записываются  в файлы  *
* с расширением key.                                                     *
*************************************************************************/

#include <common.h>

using namespace std;
using namespace rutoken::pkicore;

int main()
{
	try {
		cout << boolalpha;

		/**********************************************************************
		* Инициализируем pkicore, передав путь до директории с библиотекой    *
		* rtPKCS11ECP.                                                        *
		**********************************************************************/
		rutoken::pkicore::initialize(".");
		SCOPE_EXIT() {
			/**********************************************************************
			* Завершаем работу с pkicore при выходе из текущего блока.            *
			**********************************************************************/
			rutoken::pkicore::deinitialize();
		};

		/**********************************************************************
		* Получаем список подключенных устройств и продолжаем работу с первым *
		* доступным устройством.                                              *
		**********************************************************************/
		auto devices = Pkcs11Device::enumerate();
		if (devices.empty()) {
			throw runtime_error("There must be at least one device connected");
		}

		auto device = move(devices.front());

		/**********************************************************************
		* Аутентифицируемся на устройстве.                                    *
		**********************************************************************/
		device.login("12345678");
		SCOPE_EXIT(&device) {
			/**********************************************************************
			* Сбрасываем права доступа при выходе из текущего блока.              *
			**********************************************************************/
			device.logout();
		};

		/**********************************************************************
		* Получаем список ключей на устройстве.                               *
		**********************************************************************/
		auto keys = device.enumeratePrivateKeys();
		cout << keys.size() << " keys have been found\n" << endl;

		/**********************************************************************
		* Выводим информацию о ключах.                                        *
		**********************************************************************/
		for (auto it = keys.begin(); it != keys.end(); ++it) {
			auto& key = *it;

			/**********************************************************************
			* Выводим hex-кодированный идентификатор ключа.                       *
			**********************************************************************/
			auto id = toHex(key.getId());
			cout << "Key " << id << '\n';

			/**********************************************************************
			* Получаем открытый ключ в DER кодировке.                             *
			**********************************************************************/
			auto pubkey = key.getPublicKey().toDer();

			/**********************************************************************
			* Записываем открытый ключ в файл.                                    *
			**********************************************************************/
			const auto fileName = id + ".key";

			cout << "\t writing public key to " << fileName << '\n';

			writeFile(fileName, pubkey.data(), pubkey.size());

			cout << endl;
		}
	} catch (const exception& e) {
		cerr << e.what() << endl;
		return 1;
	}

	return 0;
}
