/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2017, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
* Пример работы с Рутокен PINPad при помощи библиотеки PKCS#11           *
* на языке C                                                             *
*------------------------------------------------------------------------*
* Использование команд вычисления/проверки ЭП на ключах                  *
* ГОСТ Р 34.10-2012 (512):                                               *
*  - установление соединения с Рутокен PINPad в первом доступном слоте;  *
*  - выполнение аутентификации c правами Пользователя;                   *
*  - подпись отображаемых платежных данных с атрибутами на экране PINPad;*
*  - проверка подписи;                                                   *
*  - сброс прав доступа Пользователя на Рутокен PINPad и закрытие        *
*    соединения с Рутокен PINPad.                                        *
*------------------------------------------------------------------------*
* Пример использует объекты, созданные в памяти Рутокен PINPad примером  *
* CreateGOST34.10-2012-PINPad.                                           *
*************************************************************************/

#include <Common.h>

/* Формат сообщения, распознаваемого PINPad:
   <!PINPADFILE RU>             // обязательный заголовок строки в кодировке CP-1251, которая будет отображаться на экране устройства
   <!PINPADFILE UTF8>           // обязательный заголовок строки в кодировке UTF-8, которая будет отображаться на экране устройства
   <!PINPADFILE INVISIBLE RU>   // обязательный заголовок строки в кодировке CP-1251, которая будет подписана PINPad без отображения на экране устройства
   <!PINPADFILE INVISIBLE UTF8> // обязательный заголовок строки в кодировке UTF-8, которая будет подписана PINPad без отображения на экране устройства
   <N>some text                 // наименование поля - текст будет отображен в левой части строки на экране PINPad
   <V>some text                 // значение поля - текст будет отображен в правой части строки на экране PINPad
   <T>some text                 // информационное поле - текст будет отображен на всей строке на экране PINPad
 */

/*************************************************************************
* Данные в формате PINPADFILE для подписи без отображения в кодировке    *
* UTF-8                                                                  *
*************************************************************************/
char data[] = "<!PINPADFILE INVISIBLE UTF8><N>ФИО:<V>Петров Петр Петрович Москва, Пионерская ул, д. 3, кв. 72"
              "<N>Перевод со счета:<V>42301810001000075212<N>Сумма:<V>150000<N>Валюта:<V>RUR"
              "<N>Наименование получателя:<V>Иванова Елена Ивановна<N>Номер счета получателя:<V>40817810338295201618"
              "<N>БИК банка получателя:<V>044525225<N>Наименование банка получателя:<V>ОАО 'СБЕРБАНК РОССИИ' Г. МОСКВА"
              "<N>Номер счета банка получателя:<V>30101810400000000225<N>Назначение платежа:<V>перевод личных средств";


/*************************************************************************
* Данные в формате PINPADFILE для подписи c отображением в кодировке     *
* UTF-8                                                                  *
*************************************************************************/


/* char data[] = "<!PINPADFILE UTF8><N>ФИО:<V>Петров Петр Петрович Москва, Пионерская ул, д. 3, кв. 72"
                "<N>Перевод со счета:<V>42301810001000075212<N>Сумма:<V>150000<N>Валюта:<V>RUR"
                "<N>Наименование получателя:<V>Иванова Елена Ивановна<N>Номер счета получателя:<V>40817810338295201618"
                "<N>БИК банка получателя:<V>044525225<N>Наименование банка получателя:<V>ОАО 'СБЕРБАНК РОССИИ' Г. МОСКВА"
                "<N>Номер счета банка получателя:<V>30101810400000000225<N>Назначение платежа:<V>перевод личных средств";
 */

/*************************************************************************
* Данные для подписи с атрибутами в виде двоичной строки                 *
*************************************************************************/
CK_BYTE attrData[] = { 0x31, 0x82, 0x01, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09,
	                   0x03, 0x31, 0x0b, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x07, 0x01, 0x30, 0x1c,
	                   0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x05, 0x31, 0x0f, 0x17, 0x0d, 0x31,
	                   0x31, 0x31, 0x32, 0x31, 0x39, 0x31, 0x34, 0x30, 0x30, 0x30, 0x32, 0x5a, 0x30, 0x4f, 0x06, 0x09,
	                   0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x04, 0x31, 0x42, 0x04, 0x40,

	                   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,      // HASH (pbtDataAttr + 77)
	                   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	                   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	                   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	                   0x30, 0x81, 0x8e, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x0f, 0x31, 0x81,
	                   0x80, 0x30, 0x7e, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x01, 0x2a,
	                   0x30, 0x08, 0x06, 0x06, 0x2a, 0x85, 0x03, 0x02, 0x02, 0x09, 0x30, 0x08, 0x06, 0x06, 0x2a, 0x85,
	                   0x03, 0x02, 0x02, 0x15, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x01,
	                   0x16, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x01, 0x02, 0x30, 0x0a,
	                   0x06, 0x08, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x03, 0x07, 0x30, 0x0e, 0x06, 0x08, 0x2a, 0x86,
	                   0x48, 0x86, 0xf7, 0x0d, 0x03, 0x02, 0x02, 0x02, 0x00, 0x80, 0x30, 0x0d, 0x06, 0x08, 0x2a, 0x86,
	                   0x48, 0x86, 0xf7, 0x0d, 0x03, 0x02, 0x02, 0x01, 0x40, 0x30, 0x07, 0x06, 0x05, 0x2b, 0x0e, 0x03,
	                   0x02, 0x07, 0x30, 0x0d, 0x06, 0x08, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x03, 0x02, 0x02, 0x01,
	                   0x28 };

/*************************************************************************
* Смещение хэш-кода внутри данных                                        *
*************************************************************************/
const CK_ULONG hashOffset = 77;

/*************************************************************************
* Шаблон для поиска открытого ключа ГОСТ Р 34.10-2012(512)               *
*************************************************************************/
CK_ATTRIBUTE publicKeyTemplate[] =
{
	{ CKA_ID, &keyPairIdGost1_512, sizeof(keyPairIdGost1_512) - 1 },        // ID пары
	{ CKA_KEY_TYPE, &keyTypeGostR3410_512, sizeof(keyTypeGostR3410_512) },  // Ключ ГОСТ Р 34.10-2012(512)
	{ CKA_CLASS, &publicKeyObject, sizeof(publicKeyObject) }                // Класс - открытый ключ
};

/*************************************************************************
* Шаблон для поиска закрытого ключа ГОСТ Р 34.10-2012(512)               *
*************************************************************************/
CK_ATTRIBUTE privateKeyTemplate[] =
{
	{ CKA_ID, &keyPairIdGost1_512, sizeof(keyPairIdGost1_512) - 1 },        // ID пары
	{ CKA_KEY_TYPE, &keyTypeGostR3410_512, sizeof(keyTypeGostR3410_512) },  // Ключ ГОСТ Р 34.10-2012(512)
	{ CKA_CLASS, &privateKeyObject, sizeof(privateKeyObject) }              // Класс - закрытый ключ
};

int main(void)
{
	HMODULE module;                                                          // Хэндл загруженной библиотеки PKCS#11
	CK_SESSION_HANDLE session;                                               // Хэндл открытой сессии

	CK_FUNCTION_LIST_PTR functionList;                                       // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
	CK_C_GetFunctionList getFunctionList;                                    // Указатель на функцию C_GetFunctionList

	CK_C_EX_GetFunctionListExtended getFunctionListEx;                       // Указатель на функцию C_EX_GetFunctionListExtended
	CK_FUNCTION_LIST_EXTENDED_PTR functionListEx;                            // Указатель на список функций расширения PKCS#11, хранящийся в структуре CK_FUNCTION_LIST_EXTENDED

	CK_TOKEN_INFO_EXTENDED tokenInfoEx;                                      // Структура данных типа CK_TOKEN_INFO_EXTENDED с информацией о токене

	CK_SLOT_ID_PTR slots;                                                    // Указатель на массив идентификаторов слотов
	CK_ULONG slotCount;                                                      // Количество идентификаторов слотов в массиве

	CK_MECHANISM_TYPE_PTR mechanisms;                                        // Указатель на массив поддерживаемых механизмов
	CK_ULONG mechanismCount;                                                 // Количество поддерживаемых механизмов

	CK_OBJECT_HANDLE_PTR privateKeys;                                        // Массив хэндлов объектов, соответствующих критериям поиска
	CK_ULONG privateKeysCount;                                               // Количество хэндлов объектов в массиве

	CK_OBJECT_HANDLE_PTR publicKeys;                                         // Массив хэндлов объектов, соответствующих критериям поиска
	CK_ULONG publicKeysCount;                                                // Количество хэндлов объектов в массиве

	CK_BYTE_PTR signature;                                                   // Указатель на буфер, содержащий подпись для исходных данных
	CK_ULONG signatureSize;                                                  // Размер буфера, содержащего подпись для исходных данных, в байтах

	CK_BYTE_PTR hash1;                                                       // Указатель на временный буфер для хэш-кода
	CK_ULONG hash1Size;                                                      // Размер временного буфера в байтах

	CK_BYTE_PTR hash2;                                                       // Указатель на временный буфер для хэш-кода
	CK_ULONG hash2Size;                                                      // Размер временного буфера в байтах

	CK_BYTE_PTR hash3;                                                       // Указатель на временный буфер для хэш-кода
	CK_ULONG hash3Size;                                                      // Размер временного буфера в байтах

	CK_BYTE_PTR hash4;                                                       // Указатель на временный буфер для хэш-кода
	CK_ULONG hash4Size;                                                      // Размер временного буфера в байтах

	CK_BBOOL confirmOp = CK_FALSE;                                           // Вспомогательная переменная для хранения значения атрибута CKA_VENDOR_KEY_CONFIRM_OP
	CK_ATTRIBUTE confirmOpAttr;                                              // Шаблон для получения значения атрибута CKA_VENDOR_KEY_CONFIRM_OP
	int isOperationInvisible;                                                // Вспомогательная переменная для хранения признака необходимости использовать функции расширения для подписи

	CK_RV rv;                                                                // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
	int r;                                                                   // Код возврата для функций, возвращающих int

	CK_ULONG i;                                                              // Вспомогательная переменная-счетчик в циклах

	int isGostR3410_512Supported = 0;                                        // Флаги для проверки поддержки токеном CKM_GOSTR3410_512
	int isGostR3411_12_512Supported = 0;                                     // Флаги для проверки поддержки токеном CKM_GOSTR3411_12_512

	int errorCode = 1;                                                       // Флаг ошибки

	/*************************************************************************
	* Выполнить действия для начала работы с библиотекой PKCS#11             *
	*************************************************************************/
	printf("Initialization...\n");

	/*************************************************************************
	* Загрузить библиотеку                                                   *
	*************************************************************************/
	module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
	CHECK(" LoadLibrary", module != NULL, exit);

	/*************************************************************************
	* Получить адрес функции запроса структуры с указателями на функции      *
	*************************************************************************/
	getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
	CHECK(" GetProcAddress (C_GetFunctionList)", getFunctionList != NULL, unload_pkcs11);

	/*************************************************************************
	* Получить структуру с указателями на функции                            *
	*************************************************************************/
	rv = getFunctionList(&functionList);
	CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

	/*************************************************************************
	* Получить адрес функции запроса структуры с указателями на функции      *
	* расширения                                                             *
	*************************************************************************/
	getFunctionListEx = (CK_C_EX_GetFunctionListExtended)GetProcAddress(module, "C_EX_GetFunctionListExtended");
	CHECK(" GetProcAddress (C_EX_GetFunctionListExtended)", getFunctionListEx != NULL, unload_pkcs11);

	/*************************************************************************
	* Получить структуру с указателями на функции расширения                 *
	*************************************************************************/
	rv = getFunctionListEx(&functionListEx);
	CHECK_AND_LOG(" Get extended function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

	/*************************************************************************
	* Инициализировать библиотеку                                            *
	*************************************************************************/
	rv = functionList->C_Initialize(NULL_PTR);
	CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

	/*************************************************************************
	* Получить количество слотов c подключенными токенами                    *
	*************************************************************************/
	rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
	CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

	CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

	/*************************************************************************
	* Получить список слотов c подключенными токенами                        *
	*************************************************************************/
	slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
	CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

	rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
	CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
	printf(" Slots available: %d\n", (int)slotCount);

	/*************************************************************************
	* Определить класс токена                                                *
	*************************************************************************/
	printf(" Determining token type\n");

	/*************************************************************************
	* Получить расширенную информацию о подключенном токене                  *
	*************************************************************************/
	tokenInfoEx.ulSizeofThisStructure = sizeof(tokenInfoEx);
	rv = functionListEx->C_EX_GetTokenInfoExtended(slots[0], &tokenInfoEx);
	CHECK_AND_LOG(" C_EX_GetTokenInfoExtended", rv == CKR_OK, rvToStr(rv), free_slots);

	/*************************************************************************
	* Проверить наличие PINPad в нулевом слоте                               *
	*************************************************************************/
	CHECK_AND_LOG(" Checking token type", tokenInfoEx.ulTokenType == TOKEN_TYPE_RUTOKEN_PINPAD_FAMILY,
	              " Device in slot 0 is not Rutoken PINPad", free_slots);

	/*************************************************************************
	* Получить список поддерживаемых токеном механизмов                      *
	*************************************************************************/
	rv = functionList->C_GetMechanismList(slots[0], NULL_PTR, &mechanismCount);
	CHECK_AND_LOG(" C_GetMechanismList (number of mechanisms)", rv == CKR_OK, rvToStr(rv), free_slots);

	CHECK_AND_LOG(" Checking mechanisms available", mechanismCount > 0, " No mechanisms available", free_slots);

	mechanisms = (CK_MECHANISM_TYPE_PTR)malloc(mechanismCount * sizeof(CK_MECHANISM_TYPE));
	CHECK(" Memory allocation for mechanisms", mechanisms != NULL_PTR, free_slots);

	rv = functionList->C_GetMechanismList(slots[0], mechanisms, &mechanismCount);
	CHECK_AND_LOG(" C_GetMechanismList", rv == CKR_OK, rvToStr(rv), free_mechanisms);

	/*************************************************************************
	* Определение поддерживаемых токеном механизмов                          *
	*************************************************************************/
	for (i = 0; i < mechanismCount; ++i) {
		if (mechanisms[i] == CKM_GOSTR3410_512) {
			isGostR3410_512Supported = 1;
			break;
		}
	}

	for (i = 0; i < mechanismCount; ++i) {
		if (mechanisms[i] == CKM_GOSTR3411_12_512) {
			isGostR3411_12_512Supported = 1;
			break;
		}
	}

	CHECK_AND_LOG(" Checking CKM_GOSTR3410_512 support", isGostR3410_512Supported,
	              " CKM_GOSTR3410_512 isn`t supported!\n", free_mechanisms);
	CHECK_AND_LOG(" Checking CKM_GOSTR3411_12_512 support", isGostR3411_12_512Supported,
	              " CKM_GOSTR3411_12_512 isn`t supported!\n", free_mechanisms);

	/*************************************************************************
	* Открыть RW сессию в первом доступном слоте                             *
	*************************************************************************/
	rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
	CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_mechanisms);

	/*************************************************************************
	* Выполнить авторизацию с правами пользователя                           *
	*************************************************************************/
	rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
	CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
	printf("Initialization has been completed successfully.\n");

	/*************************************************************************
	* Выполнить подпись платежной информации по алгоритму                    *
	* ГОСТ Р 34.10-2012(512)                                                 *
	*************************************************************************/
	printf("\nSigning data...\n");

	/*************************************************************************
	* Получить массив хэндлов закрытых ключей                                *
	*************************************************************************/
	printf(" Getting private key...\n");
	r = findObjects(functionList, session, privateKeyTemplate, arraysize(privateKeyTemplate),
	                &privateKeys, &privateKeysCount);
	CHECK(" findObjects", r == 0, logout);

	CHECK_AND_LOG(" Checking number of keys found", privateKeysCount > 0, " No objects found", logout);

	/*************************************************************************
	* Вычислить хэш-код данных                                               *
	*************************************************************************/
	printf(" Hashing data without attributes...\n");

	/*************************************************************************
	* Инициализировать хэш-функцию                                           *
	*************************************************************************/
	rv = functionList->C_DigestInit(session, &gostR3411_12_512HashMech);
	CHECK_AND_LOG("  C_DigestInit", rv == CKR_OK, rvToStr(rv), free_privateKeys);

	/*************************************************************************
	* Определить размер хэш-кода                                             *
	*************************************************************************/
	rv = functionList->C_Digest(session, (CK_BYTE_PTR)data, arraysize(data), NULL_PTR, &hash1Size);
	CHECK_AND_LOG("  C_Digest (get size)", rv == CKR_OK, rvToStr(rv), free_privateKeys);

	/*************************************************************************
	* Вычислить хэш-код данных                                               *
	*************************************************************************/
	hash1 = (CK_BYTE_PTR)malloc(hash1Size * sizeof(CK_BYTE));
	CHECK("  Memory allocation for hash", hash1 != NULL_PTR, free_privateKeys);

	rv = functionList->C_Digest(session, (CK_BYTE_PTR)data, arraysize(data), hash1, &hash1Size);
	CHECK_AND_LOG("  C_Digest (get hash)", rv == CKR_OK, rvToStr(rv), free_hash1);

	/*************************************************************************
	* Распечатать буфер, содержащий хэш-код                                  *
	*************************************************************************/
	printf("  Hashed buffer is: \n");
	printHex(hash1, hash1Size);
	printf(" Hashing has been completed.\n");

	memcpy(attrData + hashOffset, hash1, hash1Size);

	/*************************************************************************
	* Сформировать хэш-код от данных с атрибутами                            *
	*************************************************************************/
	printf(" Hashing data with attributes...\n");

	/*************************************************************************
	* Инициализировать хэш-функцию                                           *
	*************************************************************************/
	rv = functionList->C_DigestInit(session, &gostR3411_12_512HashMech);
	CHECK_AND_LOG("  C_DigestInit", rv == CKR_OK, rvToStr(rv), free_hash1);

	/*************************************************************************
	* Определить размер хэш-кода                                             *
	*************************************************************************/
	rv = functionList->C_Digest(session, (CK_BYTE_PTR)attrData, arraysize(attrData), NULL_PTR, &hash2Size);
	CHECK_AND_LOG("  C_Digest(get size)", rv == CKR_OK, rvToStr(rv), free_hash1);

	/*************************************************************************
	* Вычислить хэш-код данных                                               *
	*************************************************************************/
	hash2 = (CK_BYTE_PTR)malloc(hash2Size * sizeof(CK_BYTE));
	CHECK("  Memory allocation for hash", hash2 != NULL_PTR, free_hash1);

	rv = functionList->C_Digest(session, (CK_BYTE_PTR)attrData, arraysize(attrData), hash2, &hash2Size);
	CHECK_AND_LOG("  C_Digest (get hash)", rv == CKR_OK, rvToStr(rv), free_hash2);

	/*************************************************************************
	* Распечатать буфер, содержащий хэш-код                                  *
	*************************************************************************/
	printf("  Hashed buffer is: \n");
	printHex(hash2, hash2Size);
	printf(" Hashing has been completed.\n");

	/*************************************************************************
	* Получить значение флага подтверждения операции подписи                 *
	*************************************************************************/
	printf(" Checking whether signature confirmation is required...\n");

	confirmOpAttr.type = CKA_VENDOR_KEY_CONFIRM_OP;
	confirmOpAttr.pValue = &confirmOp;
	confirmOpAttr.ulValueLen = sizeof(confirmOp);
	rv = functionList->C_GetAttributeValue(session, privateKeys[0], &confirmOpAttr, 1);
	CHECK_AND_LOG("  C_GetAttributeValue", rv == CKR_OK, rvToStr(rv), free_hash2);

	/*************************************************************************
	* Инициализировать операцию подписи данных                               *
	*************************************************************************/

	/*************************************************************************
	* При подписи сообщения с заголовком <!PINPADFILE INVISIBLE RU> или      *
	* <!PINPADFILE INVISIBLE UTF8> на ключе, имеющем атрибут                 *
	* CKA_VENDOR_KEY_CONFIRM_OP равным CK_TRUE, а так же для подписи на      *
	* ключе, имеющем атрибут CKA_VENDOR_KEY_CONFIRM_OP равным CK_FALSE, для  *
	* инициализации подписи должна использоваться функция                    *
	* C_EX_SignInvisibleInit, для подписи - C_EX_SignInvisible.              *
	*
	* При подписи сообщения с заголовком <!PINPADFILE RU> или                *
	* <!PINPADFILE UTF8> на ключе, имеющем атрибут CKA_VENDOR_KEY_CONFIRM_OP *
	* равным CK_TRUE, для инициализации подписи должна использоваться        *
	* функция C_SignInit, для подписи - C_Sign.                              *
	*************************************************************************/

	isOperationInvisible = ((strncmp(data, "<!PINPADFILE INVISIBLE RU>", 26) == 0 || strncmp(data, "<!PINPADFILE INVISIBLE UTF8>", 28) == 0)
	                        && confirmOp == CK_TRUE) || confirmOp == CK_FALSE;

	/*************************************************************************
	* Инициализировать подпись и подписать данные                            *
	*************************************************************************/
	if (isOperationInvisible) {
		rv = functionListEx->C_EX_SignInvisibleInit(session, &gostR3410_512SigVerMech, privateKeys[0]);
		CHECK_AND_LOG(" C_EX_SignInvisibleInit", rv == CKR_OK, rvToStr(rv), free_hash2);

		rv = functionListEx->C_EX_SignInvisible(session, hash2, hash2Size, NULL_PTR, &signatureSize);
		CHECK_AND_LOG(" C_EX_SignInvisible (get size)", rv == CKR_OK, rvToStr(rv), free_hash2);

		signature = (CK_BYTE_PTR)malloc(signatureSize * sizeof(CK_BYTE));
		CHECK("  Memory allocation for signature", signature != NULL_PTR, free_hash2);

		rv = functionListEx->C_EX_SignInvisible(session, hash2, hash2Size, signature, &signatureSize);
		CHECK_AND_LOG(" C_EX_SignInvisible", rv == CKR_OK, rvToStr(rv), free_signature);
	} else {
		rv = functionList->C_SignInit(session, &gostR3410_512SigVerMech, privateKeys[0]);
		CHECK_AND_LOG(" C_SignInit", rv == CKR_OK, rvToStr(rv), free_hash2);

		rv = functionList->C_Sign(session, hash2, hash2Size, NULL_PTR, &signatureSize);
		CHECK_AND_LOG(" C_Sign (get size)", rv == CKR_OK, rvToStr(rv), free_hash2);

		signature = (CK_BYTE_PTR)malloc(signatureSize * sizeof(CK_BYTE));
		CHECK("  Memory allocation for signature", signature != NULL_PTR, free_hash2);

		rv = functionList->C_Sign(session, hash2, hash2Size, signature, &signatureSize);
		CHECK_AND_LOG(" C_Sign", rv == CKR_OK, rvToStr(rv), free_signature);
	}

	/*************************************************************************
	* Распечатать буфер, содержащий подпись                                  *
	*************************************************************************/
	printf("  Signature buffer is: \n");
	printHex(signature, signatureSize);
	printf("Data has been signed successfully.\n");

	/*************************************************************************
	* Выполнить проверку подписи данных по алгоритму ГОСТ Р 34.10-2012(512)  *
	*************************************************************************/
	printf("\nVerifying signature...\n");

	/*************************************************************************
	* Получить массив хэндлов открытых ключей                                *
	*************************************************************************/
	printf(" Getting public key...\n");
	rv = findObjects(functionList, session, publicKeyTemplate,
	                 arraysize(publicKeyTemplate), &publicKeys, &publicKeysCount);

	CHECK(" findObjects", r == 0, free_signature);
	CHECK_AND_LOG(" Checking number of keys found", publicKeysCount > 0, "No objects found", free_signature);

	/*************************************************************************
	* Инициализировать хэш-функцию                                           *
	*************************************************************************/
	rv = functionList->C_DigestInit(session, &gostR3411_12_512HashMech);
	CHECK_AND_LOG("  C_DigestInit", rv == CKR_OK, rvToStr(rv), free_publicKeys);

	/*************************************************************************
	* Определить размер хэш-кода                                             *
	*************************************************************************/
	rv = functionList->C_Digest(session, (CK_BYTE_PTR)data, arraysize(data), NULL_PTR, &hash3Size);
	CHECK_AND_LOG("  C_Digest (get size)", rv == CKR_OK, rvToStr(rv), free_publicKeys);

	/*************************************************************************
	* Вычислить хэш-код данных                                               *
	*************************************************************************/
	hash3 = (CK_BYTE_PTR)malloc(hash3Size * sizeof(CK_BYTE));
	CHECK("  Memory allocation for hash", hash3 != NULL_PTR, free_publicKeys);

	rv = functionList->C_Digest(session, (CK_BYTE_PTR)data, arraysize(data), hash3, &hash3Size);
	CHECK_AND_LOG("  C_Digest (get hash)", rv == CKR_OK, rvToStr(rv), free_hash3);

	/*************************************************************************
	* Распечатать буфер, содержащий хэш-код                                  *
	*************************************************************************/
	printf("  Hashed buffer is: \n");
	printHex(hash3, hash3Size);
	printf(" Hashing has been completed.\n");

	memcpy(attrData + hashOffset, hash3, hash3Size);

	/*************************************************************************
	* Сформировать хэш-код от данных с атрибутами                            *
	*************************************************************************/
	printf(" Hashing data...\n");

	/*************************************************************************
	* Инициализировать хэш-функцию                                           *
	*************************************************************************/
	rv = functionList->C_DigestInit(session, &gostR3411_12_512HashMech);
	CHECK_AND_LOG("  C_DigestInit", rv == CKR_OK, rvToStr(rv), free_hash3);

	/*************************************************************************
	* Определить размер хэш-кода                                             *
	*************************************************************************/
	rv = functionList->C_Digest(session, (CK_BYTE_PTR)attrData, arraysize(attrData), NULL_PTR, &hash4Size);
	CHECK_AND_LOG("  C_Digest (get size)", rv == CKR_OK, rvToStr(rv), free_hash3);

	/*************************************************************************
	* Вычислить хэш-код данных                                               *
	*************************************************************************/
	hash4 = (CK_BYTE_PTR)malloc(hash4Size * sizeof(CK_BYTE));
	CHECK("  Memory allocation for hash", hash4 != NULL_PTR, free_hash3);

	rv = functionList->C_Digest(session, (CK_BYTE_PTR)attrData, arraysize(attrData), hash4, &hash4Size);
	CHECK_AND_LOG("  C_Digest (get hash)", rv == CKR_OK, rvToStr(rv), free_hash4);

	/*************************************************************************
	* Распечатать буфер, содержащий хэш-код                                  *
	*************************************************************************/
	printf("  Hashed buffer is: \n");
	printHex(hash4, hash4Size);
	printf(" Hashing has been completed.\n");

	/*************************************************************************
	* Проверка подписи                                                       *
	*************************************************************************/
	printf(" Verifying data...\n");
	/*************************************************************************
	* Инициализировать операцию проверки подписи                             *
	*************************************************************************/
	rv = functionList->C_VerifyInit(session, &gostR3410_512SigVerMech, publicKeys[0]);
	CHECK_AND_LOG("  C_VerifyInit", rv == CKR_OK, rvToStr(rv), free_hash4);

	/*************************************************************************
	* Проверить подпись для данных                                           *
	*************************************************************************/
	rv = functionList->C_Verify(session, hash4, hash4Size, signature, signatureSize);
	CHECK_AND_LOG("  C_Verify", rv == CKR_OK, rvToStr(rv), free_hash4);

	printf("Verifying has been completed successfully.\n");

	/*************************************************************************
	* Выставить признак успешного завершения программы                       *
	*************************************************************************/
	errorCode = 0;

	/*************************************************************************
	* Выполнить действия для завершения работы с библиотекой PKCS#11         *
	*************************************************************************/
	printf("\nFinalizing... \n");

	/*************************************************************************
	* Очистить память, выделенную под подпись, объекты и хэш-коды            *
	*************************************************************************/
free_hash4:
	free(hash4);
free_hash3:
	free(hash3);
free_publicKeys:
	free(publicKeys);
free_signature:
	free(signature);
free_hash2:
	free(hash2);
free_hash1:
	free(hash1);
free_privateKeys:
	free(privateKeys);

	/*************************************************************************
	* Сбросить права доступа                                                 *
	*************************************************************************/
logout:
	rv = functionList->C_Logout(session);
	CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Закрыть открытую сессию в слоте                                        *
	*************************************************************************/
close_session:
	rv = functionList->C_CloseSession(session);
	CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Очистить память, выделенную под механизмы и слоты                      *
	*************************************************************************/
free_mechanisms:
	free(mechanisms);

free_slots:
	free(slots);

	/*************************************************************************
	* Деинициализировать библиотеку                                          *
	*************************************************************************/
finalize_pkcs11:
	rv = functionList->C_Finalize(NULL_PTR);
	CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

	/*************************************************************************
	* Выгрузить библиотеку из памяти                                         *
	*************************************************************************/
unload_pkcs11:
	CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
	if (errorCode) {
		printf("\n\nSome error occurred. Sample failed.\n");
	} else {
		printf("\n\nSample has been completed successfully.\n");
	}

	return errorCode;
}
