/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2017, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
* Данный пример демонстрирует создание упакованного CMS сообщения с      *
* помощью pkicore. Сертификат получателя в DER формате читается из файла *
* cert.cer. Упакованное сообщение записывается в файл enveloped_data.    *
*************************************************************************/

#include <common.h>

using namespace std;
using namespace rutoken::pkicore;

int main()
{
	try {
		cout << boolalpha;

		/**********************************************************************
		* Инициализируем pkicore, передав путь до директории с библиотекой    *
		* rtPKCS11ECP.                                                        *
		**********************************************************************/
		rutoken::pkicore::initialize(".");
		SCOPE_EXIT() {
			/**********************************************************************
			* Завершаем работу с pkicore при выходе из текущего блока.            *
			**********************************************************************/
			rutoken::pkicore::deinitialize();
		};

		/**********************************************************************
		* Получаем список подключенных устройств и продолжаем работу с первым *
		* доступным устройством.                                              *
		**********************************************************************/
		auto devices = Pkcs11Device::enumerate();
		if (devices.empty()) {
			throw runtime_error("There must be at least one device connected");
		}

		auto device = move(devices.front());

		/**********************************************************************
		* Аутентифицируемся на устройстве.                                    *
		**********************************************************************/
		device.login("12345678");
		SCOPE_EXIT(&device) {
			/**********************************************************************
			* Сбрасываем права доступа при выходе из текущего блока.              *
			**********************************************************************/
			device.logout();
		};

		/**********************************************************************
		* Читаем файл с сертификатом получателя сообщения в DER формате.      *
		**********************************************************************/
		const auto certFileName = "cert.cer";

		cout << "Reading certificate from " << certFileName << endl;

		auto certData = readFile(certFileName);
		ExternalCert cert(certData.data(), certData.size());

		/**********************************************************************
		* Задаем параметры и шифруем сообщение.                               *
		**********************************************************************/
		string message = "This is a message";
		cout << "Message for enveloping: " << message << endl;

		cms::Data data(vector<uint8_t>(message.begin(), message.end()));
		cms::EnvelopParams params(device, cert, cms::EnvelopParams::Flag::hardwareEncryption);

		auto envelopedData = cms::envelop(data, params);

		/**********************************************************************
		* Пишем зашифрованное сообщение в файл.                               *
		**********************************************************************/
		const auto outFileName = "enveloped_data";

		cout << "Writing enveloped message to " << outFileName << endl;

		auto outBuf = envelopedData.toBer();
		writeFile(outFileName, outBuf.data(), outBuf.size());
	} catch (const exception& e) {
		cerr << e.what() << endl;
		return 1;
	}

	return 0;
}
