/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2017, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
* Данный пример демонстрирует создание подписанного CMS сообщения с      *
* помощью pkicore. Для успешного выполнения программы на токене должен   *
* находиться сертификат с соответствующей ему ключевой парой. Для записи *
* сертификата можно воспользоваться примером ImportCert. Подписанное     *
* сообщение записывается в файл signed_data.                             *
*************************************************************************/

#include <common.h>

using namespace std;
using namespace rutoken::pkicore;

int main()
{
	try {
		cout << boolalpha;

		/**********************************************************************
		* Инициализируем pkicore, передав путь до директории с библиотекой    *
		* rtPKCS11ECP.                                                        *
		**********************************************************************/
		rutoken::pkicore::initialize(".");
		SCOPE_EXIT() {
			/**********************************************************************
			* Завершаем работу с pkicore при выходе из текущего блока.            *
			**********************************************************************/
			rutoken::pkicore::deinitialize();
		};

		/**********************************************************************
		* Получаем список подключенных устройств и продолжаем работу с первым *
		* доступным устройством.                                              *
		**********************************************************************/
		auto devices = Pkcs11Device::enumerate();
		if (devices.empty()) {
			throw runtime_error("There must be at least one device connected");
		}

		auto device = move(devices.front());

		/**********************************************************************
		* Получаем список сертификатов на устройстве и продолжаем работу      *
		* с первым найденным.                                                 *
		**********************************************************************/
		auto certs = device.enumerateCerts();
		if (certs.empty()) {
			throw runtime_error("There must be at least one user certificate on device");
		}

		auto cert = move(certs.front());

		/**********************************************************************
		* Аутентифицируемся на устройстве.                                    *
		**********************************************************************/
		device.login("12345678");
		SCOPE_EXIT(&device) {
			/**********************************************************************
			* Сбрасываем права доступа при выходе из текущего блока.              *
			**********************************************************************/
			device.logout();
		};

		/**********************************************************************
		* Задаем параметры и подписываем сообщение.                           *
		**********************************************************************/
		string message = "This is a message";
		cout << "Message for signing: " << message << endl;

		cms::Data data(vector<uint8_t>(message.begin(), message.end()));
		cms::SignParams params(cert, cms::SignParams::Flag::addUserCert | cms::SignParams::Flag::hardwareHash);

		auto signedData = cms::sign(data, params);

		/**********************************************************************
		* Пишем подписанное сообщение в файл.                                 *
		**********************************************************************/
		const auto outFileName = "signed_data";

		cout << "Writing signed message to " << outFileName << endl;

		auto outBuf = signedData.toBer();
		writeFile(outFileName, outBuf.data(), outBuf.size());
	} catch (const exception& e) {
		cerr << e.what() << endl;
		return 1;
	}

	return 0;
}
