/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2017, CJSC Aktiv-Soft. All rights reserved.         *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
* Данный пример демонстрирует создание запроса на получение сертификата  *
* в формате PKCS#10 с помощью pkicore. Для успешного выполнения          *
* программы на токене должен находиться секретный ключ. Для создания     *
* ключевой пары на токене можно воспользоваться примером                 *
* GenerateKeyPair. Запрос на получение сертификата в формате PEM         *
* записывается в файл pkcs10.pem.                                        *
*************************************************************************/

#include <common.h>

using namespace std;
using namespace rutoken::pkicore;

int main()
{
	try {
		cout << boolalpha;

		/**********************************************************************
		* Инициализируем pkicore, передав путь до директории с библиотекой    *
		* rtPKCS11ECP.                                                        *
		**********************************************************************/
		rutoken::pkicore::initialize(".");
		SCOPE_EXIT() {
			/**********************************************************************
			* Завершаем работу с pkicore при выходе из текущего блока.            *
			**********************************************************************/
			rutoken::pkicore::deinitialize();
		};

		/**********************************************************************
		* Получаем список подключенных устройств и продолжаем работу с первым *
		* доступным устройством.                                              *
		**********************************************************************/
		auto devices = Pkcs11Device::enumerate();
		if (devices.empty()) {
			throw runtime_error("There must be at least one device connected");
		}

		auto device = move(devices.front());

		/**********************************************************************
		* Аутентифицируемся на устройстве.                                    *
		**********************************************************************/
		device.login("12345678");
		SCOPE_EXIT(&device) {
			/**********************************************************************
			* Сбрасываем права доступа при выходе из текущего блока.              *
			**********************************************************************/
			device.logout();
		};

		/**********************************************************************
		* Получаем список ключей на устройстве и продолжаем работу с первым   *
		* найденным                                                           *
		**********************************************************************/
		auto keys = device.enumeratePrivateKeys();
		if (keys.empty()) {
			throw runtime_error("There must be at least one key on device");
		}

		auto key = move(keys.front());

		/**********************************************************************
		* Создаем субъект сертификата и заполняем его поля.                   *
		**********************************************************************/
		X500Dn subject;
		subject.setRdn(X500Dn::RdnId::commonName, "commonName")
		.setRdn(X500Dn::RdnId::surname, "surname")
		.setRdn(X500Dn::RdnId::givenName, "givenName");

		/**********************************************************************
		* Создаем key usage расширение сертификата и задаем его значение.    *
		**********************************************************************/
		X509KeyUsage keyUsage;
		keyUsage.addOption(X509KeyUsage::Option::digitalSignature)
		.addOption(X509KeyUsage::Option::keyAgreement)
		.setCritical();

		/**********************************************************************
		* Задаем поля запроса на получение сертификата.                       *
		**********************************************************************/
		Pkcs10RequestInfo info;
		info.setSubject(move(subject))
		.setExtension(move(keyUsage));

		/**********************************************************************
		* Создаем PKCS#10 запрос на получение сертификата.                    *
		**********************************************************************/
		auto request = createPkcs10Request(key, info);
		auto pem = request.toPem();

		/**********************************************************************
		* Записываем запрос в файл.                                           *
		**********************************************************************/
		const auto fileName = "pkcs10.pem";

		cout << "Writing certification request to " << fileName << endl;

		writeFile(fileName, pem.data(), pem.size());
	} catch (const exception& e) {
		cerr << e.what() << endl;
		return 1;
	}

	return 0;
}
