# ######################################################################################################################
# Function section
# ######################################################################################################################

# Import variable and it's value to parent scope
macro(import var)
    if(${ARGC} EQUAL 1)
        set(val ${${var}})
    else()
        set(val "${ARGN}")
        set(${var} ${val})
    endif()
    set(${var}
        ${val}
        PARENT_SCOPE)
endmacro()

# Return file name without smallest extension
function(get_filename_wse result file)
    get_filename_component(file ${file} NAME)
    string(REGEX REPLACE "\\.[^.]*$" "" ${result} ${file})
    import(${result})
endfunction()

# Create custom target and put it to internal folder
function(add_internal_target target)
    add_custom_target(${target})
    set_target_properties(${target} PROPERTIES FOLDER ${INTERNAL_TARGET_FOLDER})
endfunction()

# Function for adding rutoken dependencies from sdk
function(target_rutoken_dependencies target)
    foreach(package ${ARGN})
        if(${package}_LIBS)
            target_link_libraries(${target} PRIVATE ${${package}_LIBS})
        endif()

        if(${package}_INCLUDE_DIRS)
            target_include_directories(${target} PRIVATE ${${package}_INCLUDE_DIRS})
        endif()

        if(${package}_SO_LIBS)
            get_property(
                out_dir
                TARGET ${target}
                PROPERTY RUNTIME_OUTPUT_DIRECTORY)
            get_filename_component(out_dir ${out_dir} REALPATH)
            string(SHA1 hash ${out_dir})
            set(copy_target copy_${package}_SO_LIBS_to_${hash})

            if(NOT TARGET ${copy_target})
                add_internal_target(${copy_target})
                safe_copy(
                    ${copy_target}
                    FILES ${${package}_SO_LIBS}
                    DESTINATION ${out_dir})
            endif()

            add_dependencies(${target} ${copy_target})
        endif()
    endforeach()
endfunction()

# Function to glue list elements
function(JOIN values glue output)
    string(REPLACE ";" "${glue}" _tmp_str "${values}")
    import(${output} "${_tmp_str}")
endfunction()

# Function to copy files with preserving symlinks
function(safe_copy target)
    cmake_parse_arguments(_arg "" "DESTINATION;NEW_NAME" "FILES" ${ARGN})

    if(${_arg_UNPARSED_ARGUMENTS})
        message(FATAL_ERROR "safe_copy had unparsed arguments")
    endif()

    set(dest ${_arg_DESTINATION})
    add_custom_command(
        TARGET ${target}
        POST_BUILD
        COMMAND ${CMAKE_COMMAND} -E make_directory ${dest})

    set(new_name ${_arg_NEW_NAME})

    foreach(file ${_arg_FILES})
        set(file_name ${file})
        if(new_name)
            set(file_name ${new_name})
        endif()
        get_filename_component(file_name ${file_name} ABSOLUTE)
        get_filename_component(file_name ${file_name} NAME)

        add_custom_command(
            TARGET ${target}
            POST_BUILD
            COMMAND cmake -E remove -f ${dest}/${file_name})
    endforeach()

    if(UNIX)
        add_custom_command(
            TARGET ${target}
            POST_BUILD
            COMMAND cp -RP ${_arg_FILES} ${dest}/${new_name})
        return()
    endif()

    foreach(file ${_arg_FILES})
        set(file_name ${file})
        if(new_name)
            set(file_name ${new_name})
        endif()
        get_filename_component(file_name ${file_name} ABSOLUTE)
        get_filename_component(file_name ${file_name} NAME)

        set(copy_cmd ${CMAKE_COMMAND} -E copy)
        if(IS_DIRECTORY ${file})
            set(copy_cmd ${CMAKE_COMMAND} -E copy_directory)
        endif()

        add_custom_command(
            TARGET ${target}
            POST_BUILD
            COMMAND ${copy_cmd} ${file} ${dest}/${file_name})
    endforeach()
endfunction()

# Function to copy files with preserving symlinks Resolve
# https://stackoverflow.com/questions/1027247/is-it-better-to-specify-source-files-with-glob-or-each-file-individually-in-cmak
function(safe_glob target)
    if(NOT ${CMAKE_VERSION} VERSION_LESS 3.12)
        set(extra_options CONFIGURE_DEPENDS)
    endif()
    file(GLOB ${target} ${extra_options} ${ARGN})
    import(${target})
endfunction()

# ######################################################################################################################
# Global setup section
# ######################################################################################################################

# If build type is not specified -- build debug
if(NOT CMAKE_BUILD_TYPE)
    set(CMAKE_BUILD_TYPE Debug)
endif()

# Folders are used to hide internal targets and organize normal targets
set_property(GLOBAL PROPERTY USE_FOLDERS ON)

# Common compiler options
if(MSVC)
    add_compile_options(
        $<$<CONFIG:>:/MT> # ---------|
        $<$<CONFIG:Debug>:/MTd> # ---|-- Statically link the runtime libraries
        $<$<CONFIG:Release>:/MT> # --|
    )
endif()

# ######################################################################################################################
# Setup vars section
# ######################################################################################################################
function(setup_vars)
    # ARCH -- architecture of target processor. Possible values: x86, x86_64, arm64, x86_64+arm64.
    # Can not be provided by user. Use or modify with cautious.
    if(ARCH)
        message("Manual setting for ARCH variable is not allowed.")
        if(${CMAKE_GENERATOR} MATCHES "Visual Studio")
            message(FATAL_ERROR "Use -A Win32 or -A x64 instead.")
        endif()
        if(${CMAKE_SYSTEM_NAME} MATCHES "Darwin")
            message(FATAL_ERROR "Use -DCMAKE_OSX_ARCHITECTURES instead.")
        endif()
        message(FATAL_ERROR "Use -DCMAKE_C/CXX_FLAGS or -DCMAKE_C/CXX_COMPILER instead.")
    endif()

    if(${CMAKE_GENERATOR} MATCHES "Visual Studio")
        set(ARCH ${CMAKE_VS_PLATFORM_NAME})
        string(TOLOWER ${ARCH} ARCH)
        if(${ARCH} MATCHES "(x86_64|amd64|x64).*")
            set(ARCH "x86_64")
        elseif(${ARCH} MATCHES "(win32).*")
            set(ARCH "x86")
        else()
            message(FATAL_ERROR "Chosen architecture isn't supported yet")
        endif()
    elseif(${CMAKE_SYSTEM_NAME} MATCHES "Darwin")
        if(CMAKE_OSX_ARCHITECTURES)
            list(SORT CMAKE_OSX_ARCHITECTURES)
            list(REVERSE CMAKE_OSX_ARCHITECTURES)
            join("${CMAKE_OSX_ARCHITECTURES}" "+" ARCH)
        else()
            set(ARCH ${CMAKE_HOST_SYSTEM_PROCESSOR})
        endif()
    else()
        execute_process(COMMAND ${CMAKE_C_COMPILER} ${CMAKE_C_FLAGS} -print-multiarch OUTPUT_VARIABLE ARCH)
        if(ARCH MATCHES "^[ \n]*$") # your compiler doesn't support multiarch building
            execute_process(COMMAND ${CMAKE_C_COMPILER} ${CMAKE_C_FLAGS} -dumpmachine OUTPUT_VARIABLE ARCH)
        endif()

        string(TOLOWER ${ARCH} ARCH)
        if(${ARCH} MATCHES "(x86_64|x64).*")
            set(ARCH "x86_64")
        elseif(${ARCH} MATCHES "(x86|i[3456]86).*")
            set(ARCH "x86")
        elseif(${ARCH} MATCHES "(aarch64).*")
            set(ARCH "arm64")
        else()
            string(REGEX MATCH "[^-]+" ARCH ${ARCH})
            message(WARNING "The architecture wasn't tested. Trying to use deps for ${ARCH} arch.")
        endif()
    endif()

    import(ARCH)

    if(${CMAKE_SYSTEM_NAME} MATCHES "Windows")
        set(SO_LIB_EXT dll)
        set(PLATFORM_NAME "windows")
    elseif(${CMAKE_SYSTEM_NAME} MATCHES "Linux")
        set(SO_LIB_EXT so)
        set(PLATFORM_NAME "linux_glibc")
    elseif(${CMAKE_SYSTEM_NAME} MATCHES "FreeBSD")
        set(SO_LIB_EXT so)
        set(PLATFORM_NAME "freebsd")
    elseif(${CMAKE_SYSTEM_NAME} MATCHES "Darwin")
        set(SO_LIB_EXT dylib)
        set(PLATFORM_NAME "macos")
    endif()

    if(${CMAKE_SYSTEM_NAME} MATCHES "Darwin")
        set(PLATFORM_DIR "${PLATFORM_NAME}-x86_64")
        set(PLATFORM_DIR_POSTFIX "+arm64")
    else()
        set(PLATFORM_DIR "${PLATFORM_NAME}-${ARCH}")
    endif()

    # OUT_DIR -- name of output directory for target platform
    import(OUT_DIR "${PLATFORM_NAME}-${ARCH}-$<LOWER_CASE:$<CONFIG>>")

    # SDK_ROOT -- path to root of sdk
    import(SDK_ROOT ${CMAKE_CURRENT_LIST_DIR})

    # INTERNAL_TARGET_FOLDER -- name of folder for internal/hidden targets
    import(INTERNAL_TARGET_FOLDER Internal)

    # ##################################################################################################################
    # Setup rutoken packages
    # ##################################################################################################################

    # cmake-format off
    # In this section we setup libxml2, xmlsec, icu, openssl, rtengine, pkicore and openssl libs:
    # - ${LIB}_SO_LIBS -- path to shared library.
    # - ${LIB}_INCLUDE_DIRS -- directory to include.
    # - ${LIB}_LIB -- library to link.
    macro(setup_package package)
        cmake_parse_arguments(_arg "NOT_USE_POSTFIX" "DIR" "LIB_DIRS;LIB_NAMES;INCLUDE_DIRS;SO_PATTERNS" ${ARGN})

        if(${_arg_UNPARSED_ARGUMENTS})
            message(FATAL_ERROR "setup_package had unparsed arguments")
        endif()

        set(${package}_DIR "${SDK_ROOT}/${_arg_DIR}/${PLATFORM_DIR}")
        if(NOT _arg_NOT_USE_POSTFIX)
            set(${package}_DIR "${${package}_DIR}${PLATFORM_DIR_POSTFIX}")
        endif()

        if(NOT EXISTS "${${package}_DIR}")
            message(STATUS "Can't find directory ${${package}_DIR} for package ${package}")
        endif()

        foreach(dir ${_arg_INCLUDE_DIRS})
            if(NOT IS_ABSOLUTE ${dir})
                set(dir "${${package}_DIR}/${dir}")
            endif()

            list(APPEND ${package}_INCLUDE_DIRS "${dir}")
        endforeach()
        import(${package}_INCLUDE_DIRS)

        unset(lib_dirs)
        foreach(dir ${_arg_LIB_DIRS})
            list(APPEND lib_dirs "${${package}_DIR}/${dir}")
        endforeach()
        set(lib_names ${_arg_LIB_NAMES})
        if(lib_dirs)
            foreach(lib_name ${lib_names})
                unset(lib CACHE)
                find_library(
                    lib
                    NAMES ${lib_name}
                    PATHS ${lib_dirs}
                    NO_DEFAULT_PATH)

                if(lib)
                    list(APPEND ${package}_LIBS ${lib})
                endif()
            endforeach()
        endif()
        import(${package}_LIBS)

        foreach(pattern ${_arg_SO_PATTERNS})
            safe_glob(libs "${${package}_DIR}/${pattern}")
            list(APPEND ${package}_SO_LIBS "${libs}")
        endforeach()
        import(${package}_SO_LIBS)
    endmacro()

    setup_package(
        PKCS11
        DIR "pkcs11/lib"
        INCLUDE_DIRS "${SDK_ROOT}/pkcs11/include"
        SO_PATTERNS "*pkcs11*")
    if(NOT PLATFORM_DIR STREQUAL "linux_glibc-arm64") # There is no pkicore package for linux_glibc-arm64
        setup_package(
            PKICORE
            DIR "pkicore/cpp/bin"
            NOT_USE_POSTFIX
            INCLUDE_DIRS "include"
            SO_PATTERNS "lib/*.${SO_LIB_EXT}*"
            LIB_DIRS lib
            LIB_NAMES pki-core)
    endif()
    if(NOT PLATFORM_DIR STREQUAL "linux_glibc-x86") # There is no XML2 package for linux_glibc-x86
        setup_package(
            XML2
            DIR "openssl/bin/Common/libxml2"
            INCLUDE_DIRS "include" "include/libxml2"
            SO_PATTERNS "lib/*.${SO_LIB_EXT}*"
            LIB_DIRS lib
            LIB_NAMES libxml2 xml2)
    endif()
    if(${CMAKE_SYSTEM_NAME} MATCHES "Windows")
        setup_package(
            ICU
            DIR "openssl/bin/Common/icu"
            INCLUDE_DIRS "include" "include/unicode"
            SO_PATTERNS "lib/*.${SO_LIB_EXT}*"
            LIB_DIRS lib
            LIB_NAMES icu)
    endif()

    foreach(ver 1.1 3.0)
        if(${PLATFORM_NAME} MATCHES "macos" AND ${ver} MATCHES 3.0)
            setup_package(
            OPENSSL_${ver}
            DIR "openssl/bin/${ver}/openssl-${ver}"
            INCLUDE_DIRS libcrypto.framework/Headers libssl.framework/Headers 
            SO_PATTERNS "libcrypto.framework" "libssl.framework"
            LIB_DIRS "."
            LIB_NAMES libssl ssl libcrypto crypto)
        else()
            setup_package(
            OPENSSL_${ver}
            DIR "openssl/bin/${ver}/openssl-${ver}"
            INCLUDE_DIRS include
            SO_PATTERNS "lib/*.${SO_LIB_EXT}*"
            LIB_DIRS "lib"
            LIB_NAMES libssl ssl libcrypto crypto)
        endif()
        if(NOT PLATFORM_DIR STREQUAL "linux_glibc-x86") # There is no XML2 package for linux_glibc-x86
            setup_package(
                XMLSEC_${ver}
                DIR "openssl/bin/${ver}/xmlsec-${ver}"
                INCLUDE_DIRS include include/xmlsec1
                SO_PATTERNS "lib/*.${SO_LIB_EXT}*"
                LIB_DIRS lib
                LIB_NAMES xmlsec1 libxmlsec xmlsec1-openssl libxmlsec-openssl)
        endif()
        setup_package(
            RTENGINE_${ver}
            DIR "openssl/bin/${ver}/rtengine-${ver}"
            INCLUDE_DIRS include
            SO_PATTERNS "lib/*.${SO_LIB_EXT}*" "rtengine.framework"
            LIB_DIRS "." "lib"
            LIB_NAMES rtengine)
        list(APPEND RTENGINE_${ver}_INCLUDE_DIRS ${PKCS11_INCLUDE_DIRS})
        import(RTENGINE_${ver}_INCLUDE_DIRS)
    endforeach()
endfunction()

setup_vars()
