/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

package ru.rutoken.samples.pkcs11utils;

import com.sun.jna.Native;
import com.sun.jna.Platform;

import ru.rutoken.pkcs11jna.Pkcs11;
import ru.rutoken.pkcs11jna.RtPkcs11;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Collections;
import java.util.Comparator;
import java.util.Objects;

import static ru.rutoken.samples.BuildConfig.BUILD_DIR_PATH;

public class RtPkcs11Library {
    private static final String TEMP_DIR_PATH =
            Paths.get(System.getProperty("java.io.tmpdir"), "rutoken-sdk").toString();

    static {
        System.setProperty("jna.tmpdir", TEMP_DIR_PATH);
    }

    private static final RtPkcs11 INSTANCE = load();

    private RtPkcs11Library() {
    }

    public static Pkcs11 getPkcs11Interface() {
        return INSTANCE;
    }

    public static RtPkcs11 getPkcs11ExtendedInterface() {
        return INSTANCE;
    }

    private static void deleteTempDir() throws IOException {
        try {
            Files.walk(Paths.get(TEMP_DIR_PATH))
                    .sorted(Comparator.reverseOrder())
                    .map(Path::toFile)
                    .forEach(File::delete);
        } catch (NoSuchFileException ignore) {
        }
    }

    /**
     * Copies pkcs11ecp.framework from resources to a temporary directory.
     * <br/><br/>
     * The search path for pkcs11ecp.framework varies based on the Java SDK sample execution method:
     * <ul>
     * <li>When running with the <b>Application configuration in IntelliJ IDEA</b>, no .jar file is created, and
     * pkcs11ecp.framework remains in the same file system from which the sample is launched.</li>
     * <li>When executing a <b>.jar file</b>, pkcs11ecp.framework is embedded within it. To access the library,
     * we use a Zip File System Provider.</li>
     * </ul>
     *
     * @return Absolute path of pkcs11ecp library in temporary directory. This path can be used in Native.load function.
     * @see <a href="https://docs.oracle.com/javase/8/docs/technotes/guides/io/fsp/zipfilesystemprovider.html">
     * Zip File System Provider</a>
     */
    private static String copyPkcs11FrameworkToTempDir() throws IOException, URISyntaxException {
        Path sourcePath;
        Path targetPath = Paths.get(TEMP_DIR_PATH);

        Path jarPath = Paths.get(BUILD_DIR_PATH + "/libs/pkcs11.jar");
        if (Files.exists(jarPath)) { // Executing a jar file
            URI uri = URI.create("jar:" + jarPath.toUri());
            try (FileSystem zipfs = FileSystems.newFileSystem(uri, Collections.emptyMap())) {
                sourcePath = zipfs.getPath("/darwin");
                return doCopyPkcs11FrameworkToTempDir(sourcePath, targetPath);
            }
        } else { // Executing in the IntelliJ IDEA
            sourcePath = Paths.get(Objects.requireNonNull(RtPkcs11Library.class.getResource("/darwin")).toURI());
            return doCopyPkcs11FrameworkToTempDir(sourcePath, targetPath);
        }
    }

    private static String doCopyPkcs11FrameworkToTempDir(Path sourcePath, Path targetPath) throws IOException {
        final String[] executablePath = {""};
        Files.walkFileTree(sourcePath, new SimpleFileVisitor<Path>() {
            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                Files.createDirectories(targetPath.resolve(sourcePath.relativize(dir).toString()));
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Path currentTarget = targetPath.resolve(sourcePath.relativize(file).toString());
                Files.copy(file, currentTarget, StandardCopyOption.REPLACE_EXISTING);
                if (file.getFileName().toString().equals("rtpkcs11ecp"))
                    executablePath[0] = currentTarget.toString();
                return FileVisitResult.CONTINUE;
            }
        });

        return executablePath[0];
    }

    private static RtPkcs11 load() {
        try {
            // Clear temporary directory before extracting rtpkcs11ecp library to it as it can contain rtpkcs11ecp from
            // previous launch (JNA was unable to remove it for some reason).
            deleteTempDir();

            // We use Native.extractFromResourcePath function to avoid loading the system rtpkcs11ecp library.
            // JNA does not support work with non-system frameworks in this function, so we have to copy
            // rtpkcs11ecp.framework content in temporary directory.
            String libraryPath = Platform.isMac() ? copyPkcs11FrameworkToTempDir() :
                    Native.extractFromResourcePath("rtpkcs11ecp").getAbsolutePath();
            return Native.load(libraryPath, RtPkcs11.class);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}
