/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/*************************************************************************
 * Добавление доверенной метки времени в CMS в качестве неподписанного    *
 * атрибута первому найденному подписанту.                                *
 * Этот пример использует CMS, созданный примером SignCMS, запрос,        *
 * созданный примером CreateTSRequest, и ответ на запрос, который можно   *
 * получить через инструменты командной строки OpenSSL настоящего пакета  *
 * SDK (openssl/samples/tool/README.md). Ответ на запрос надо положить    *
 * в папку с собранными примерами под названием response.tsr              *
 *************************************************************************/

#include <assert.h>

#include <Common.h>

#include <openssl/cms.h>
#include <openssl/ts.h>
#include <openssl/x509.h>

X509_STORE* create_cert_store();
STACK_OF(X509) * get_tsa_cert();
int add_tst_to_cms(CMS_ContentInfo* cms, TS_RESP* response);

int main(void) {
    ENGINE* rtEngine; // rtengine
    BIO* inBio;       // Описатель потока ввода
    BIO* outBio;      // Описатель потока вывода

    CMS_ContentInfo* cms; // Описатель CMS структуры
    TS_REQ* request;      // Описатель структуры запроса метки времени
    TS_RESP* response; // Описатель структуры ответа сервера с меткой времени
    TS_VERIFY_CTX* ctx; // Описатель контекста проверки ответа сервера с меткой времени

    int r;             // Код возврата
    int errorCode = 1; // Флаг ошибки

    printf("Sample has started.\n\n");
    /*************************************************************************
     * Инициализация OPENSSL_crypto                                           *
     *************************************************************************/
    r = OPENSSL_init_crypto(OPENSSL_INIT_NO_LOAD_CONFIG | OPENSSL_INIT_NO_ATEXIT, NULL);
    CHECK("  OPENSSL_init_crypto", r, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_load_engine();
    CHECK("  rt_eng_load_engine", r == 1, exit);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    r = ENGINE_init(rtEngine);
    CHECK("  ENGINE_init", r == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK("  ENGINE_set_default", r == 1, finalize_engine);

    /*************************************************************************
     * Открытие потока ввода                                                  *
     *************************************************************************/
    inBio = BIO_new_file("cms_signed.pem", "rb");
    CHECK("  BIO_new_file", inBio != NULL, unregister_engine);

    /*************************************************************************
     * Чтение CMS структуры                                                   *
     *************************************************************************/
    cms = PEM_read_bio_CMS(inBio, NULL, NULL, NULL);
    CHECK("  PEM_read_bio_CMS", cms != NULL, free_in_bio);

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    BIO_free_all(inBio);
    inBio = BIO_new_file("request.tsq", "rb");
    CHECK("  BIO_new_file", inBio != NULL, free_cms);

    /*************************************************************************
     * Чтение запроса на метку времени                                        *
     *************************************************************************/
    request = d2i_TS_REQ_bio(inBio, NULL);
    CHECK("  d2i_TS_REQ_bio", request != NULL, free_cms);

    /*************************************************************************
     * Создание контекста проверки метки времени из запроса                   *
     *************************************************************************/
    ctx = TS_REQ_to_TS_VERIFY_CTX(request, NULL);
    CHECK("  TS_REQ_to_TS_VERIFY_CTX", ctx != NULL, free_ts_req);

    /*************************************************************************
     * Выставление флага проверки подписи метки                               *
     *************************************************************************/
    TS_VERIFY_CTX_add_flags(ctx, TS_VFY_SIGNATURE);

    /*************************************************************************
     * Создание хранилища сертификатов и добавление его в контекст проверки   *
     *************************************************************************/
    r = TS_VERIFY_CTX_set_store(ctx, create_cert_store()) != NULL;
    CHECK("  TS_VERIFY_CTX_set_store", r == 1, free_ctx);

    /*************************************************************************
     * Добавление сертификата TSA в контекст проверки                         *
     *************************************************************************/
    r = TS_VERIFY_CTS_set_certs(ctx, get_tsa_cert()) != NULL;
    CHECK("  TS_VERIFY_CTS_set_certs", r == 1, free_ctx);

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    BIO_free_all(inBio);
    inBio = BIO_new_file("response.tsr", "rb");
    CHECK("  BIO_new_file", inBio != NULL, free_ctx);

    /*************************************************************************
     * Чтение ответа сервера с меткой времени                                 *
     *************************************************************************/
    response = d2i_TS_RESP_bio(inBio, NULL);
    CHECK("  d2i_TS_RESP_bio", response != NULL, free_ctx);

    /*************************************************************************
     * Проверка ответа от сервера                                             *
     *************************************************************************/
    r = TS_RESP_verify_response(ctx, response);
    CHECK("  TS_RESP_verify_response", r == 1, free_ts_resp);

    /*************************************************************************
     * Добавление метки времени в качестве неподписанного атрибута CMS        *
     *************************************************************************/
    r = add_tst_to_cms(cms, response);
    CHECK("  add_tst_to_cms", r == 1, free_ts_resp);

    /*************************************************************************
     * Открытие поточного вывода в файл                                       *
     *************************************************************************/
    outBio = BIO_new_file("cms_signed_with_tst.pem", "wb");
    CHECK("  BIO_new_file", outBio != NULL, free_ts_resp);

    /*************************************************************************
     * Запись CMS структуры                                                   *
     *************************************************************************/
    r = PEM_write_bio_CMS(outBio, cms);
    CHECK("  PEM_write_bio_CMS", r > 0, free_out_bio);

    /*************************************************************************
     * Установка признака успешного завершения программы                      *
     *************************************************************************/
    errorCode = 0;
free_out_bio:

    /*************************************************************************
     * Закрытие потока вывода                                                 *
     *************************************************************************/
    BIO_free_all(outBio);
free_ts_resp:

    /*************************************************************************
     * Освобождение TS_RESP структуры                                         *
     *************************************************************************/
    TS_RESP_free(response);
free_ctx:

    /*************************************************************************
     * Освобождение контекста проверки                                        *
     *************************************************************************/
    TS_VERIFY_CTX_free(ctx);
free_ts_req:

    /*************************************************************************
     * Освобождение TS_REQ структуры                                          *
     *************************************************************************/
    TS_REQ_free(request);
free_cms:

    /*************************************************************************
     * Освобождение CMS структуры                                             *
     *************************************************************************/
    CMS_ContentInfo_free(cms);
free_in_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
unregister_engine:

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    r = ENGINE_finish(rtEngine);
    CHECK_RELEASE("  ENGINE_finish", r == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_unload_engine();
    CHECK_RELEASE("  rt_eng_unload_engine", r == 1, errorCode);
exit:
    OPENSSL_cleanup();
    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}

X509_STORE* create_cert_store() {
    BIO* inBio;               // Описатель потока ввода
    X509* caCert;             // Описатель сертификата удостоверяющего центра
    X509_STORE* store = NULL; // Описатель хранилища сертификатов

    int r; // Код возврата

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    inBio = BIO_new_file("test_trusted_ca.cer", "rb");
    CHECK("    BIO_new_file", inBio != NULL, exit);

    /*************************************************************************
     * Чтение сертификата удостоверяющего центра                              *
     *************************************************************************/
    caCert = PEM_read_bio_X509(inBio, NULL, NULL, NULL);
    CHECK("    PEM_read_bio_X509", caCert != NULL, free_in_bio);

    /*************************************************************************
     * Создание хранилища сертификатов                                        *
     *************************************************************************/
    store = X509_STORE_new();
    CHECK("    X509_STORE_new", store != NULL, free_cert);

    /*************************************************************************
     * Добавление сертификата удостоверяющего центра в хранилище              *
     *************************************************************************/
    r = X509_STORE_add_cert(store, caCert);
    CHECK("    X509_STORE_add_cert", r == 1, free_store);

    goto free_cert;
free_store:

    /*************************************************************************
     * Освобождение хранилища сертификатов                                    *
     *************************************************************************/
    X509_STORE_free(store);
    store = NULL;
free_cert:

    /*************************************************************************
     * Освобождение сертификата                                               *
     *************************************************************************/
    X509_free(caCert);
free_in_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
exit:
    return store;
}

STACK_OF(X509) * get_tsa_cert() {
    BIO* inBio;                   // Описатель потока ввода
    STACK_OF(X509)* certs = NULL; // Описатель контейнера сертификатов
    X509* tsaCert;                // Описатель сертификата

    int r; // Код возврата

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    inBio = BIO_new_file("test_tsa_cert.cer", "rb");
    CHECK("    BIO_new_file", inBio != NULL, exit);

    /*************************************************************************
     * Чтение сертификата                                                     *
     *************************************************************************/
    tsaCert = PEM_read_bio_X509(inBio, NULL, NULL, NULL);
    CHECK("    PEM_read_bio_X509", tsaCert != NULL, free_in_bio);

    /*************************************************************************
     * Создание контейнера сертификатов                                       *
     *************************************************************************/
    certs = sk_X509_new_null();
    CHECK("    sk_X509_new_null", certs != NULL, free_cert);

    /*************************************************************************
     * Добавление сертификата в контейнер                                     *
     *************************************************************************/
    r = sk_X509_push(certs, tsaCert);
    CHECK("    sk_X509_push", r == 1, free_sk_certs);

    goto free_in_bio;
free_sk_certs:

    /*************************************************************************
     * Освобождение контейнера сертификатов                                   *
     *************************************************************************/
    sk_X509_free(certs);
    certs = NULL;
free_cert:

    /*************************************************************************
     * Освобождение сертификата                                               *
     *************************************************************************/
    X509_free(tsaCert);
free_in_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
exit:
    return certs;
}

int add_tst_to_cms(CMS_ContentInfo* cms, TS_RESP* response) {
    PKCS7* token;                       // Метка времени
    unsigned char* tokenBuf = NULL;     // Закодированная метка времени
    int tokenBufLen;                    // Длина закодированной метки времени
    STACK_OF(CMS_SignerInfo) * signers; // Описатель контейнера подписантов
    CMS_SignerInfo* signerInfo;         // Описатель структуры подписанта
    int num;                            // Количество подписантов в CMS подписи

    int r = 0; // Код возврата

    /*************************************************************************
     * Получение метки времени из ответа сервера                              *
     *************************************************************************/
    token = TS_RESP_get_token(response);
    CHECK("    TS_RESP_get_token", token != NULL, exit);

    /*************************************************************************
     * DER кодирование метки времени                                          *
     *************************************************************************/
    tokenBufLen = i2d_PKCS7(token, &tokenBuf);
    CHECK("    i2d_PKCS7", tokenBufLen >= 0, exit);

    /*************************************************************************
     * Получение подписантов                                                  *
     *************************************************************************/
    signers = CMS_get0_SignerInfos(cms);
    CHECK("    CMS_get0_SignerInfos", signers != NULL, free_tokenbuf);

    /*************************************************************************
     * Проверка количества подписантов                                        *
     *************************************************************************/
    num = sk_CMS_SignerInfo_num(signers);
    CHECK("    sk_CMS_SignerInfo_num", num > 0, free_tokenbuf);

    /*************************************************************************
     * Получение первого подписанта                                           *
     *************************************************************************/
    signerInfo = sk_CMS_SignerInfo_value(signers, 0);
    CHECK("    sk_CMS_SignerInfo_value", signerInfo != NULL, free_tokenbuf);

    /*************************************************************************
     * Добавление метки времени в CMS в качестве неподписанного атрибута      *
     *************************************************************************/
    r = CMS_unsigned_add1_attr_by_NID(signerInfo, NID_id_smime_aa_timeStampToken, V_ASN1_SEQUENCE, tokenBuf,
                                      tokenBufLen);
    CHECK("    CMS_unsigned_add1_attr_by_NID", r == 1, free_tokenbuf);

free_tokenbuf:

    /*************************************************************************
     * Освобождение закодированной метки времени                              *
     *************************************************************************/
    OPENSSL_free(tokenBuf);
exit:
    return r;
}
