/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/***********************************************************************
 * Для работы данного примера необходим сертификат, выписанный на       *
 * ключевой паре, которая будет использована в примере. О том, как      *
 * выписать сертификат на имеющуюся ключевую пару, можно узнать из      *
 * описания к использованию rtengine через инструменты командной строки *
 * OpenSSL настоящего пакета SDK (openssl/samples/tool/README.md)       *
 *                                                                      *
 * Пример собирается только с openssl 3.0                               *
 * ---------------------------------------------------------------------*
 * Для работы примера с ECDSA и RSA ключами нужно внести соответствующие *
 * изменения в Common.h и TokenPreparation.c                             *
 *************************************************************************/

#include <assert.h>

#include <Common.h>

#include <openssl/cms.h>
#include <openssl/ts.h>
#include <openssl/x509.h>

ASN1_OCTET_STRING* get_signature(CMS_ContentInfo* cms);
TS_REQ* create_ts_request(ASN1_OCTET_STRING* signature);
X509_STORE* create_cert_store();
STACK_OF(X509) * get_tsa_cert();
int add_tst_to_cms(CMS_ContentInfo* cms, TS_RESP* response);

int main(void) {
    EVP_PKEY* key;      // Описатель ключевой пары
    ENGINE* rtEngine;   // rtengine
    BIO* inBio;         // Описатель потока ввода
    BIO* inBioResponse; // Описатель потока ввода для ответа от сервера
    BIO* outBioRequest; // Описатель потока вывода для запроса метки времени
    BIO* outBioCadesT;  // Описатель потока вывода для результрующей CMS

    X509* cert;                   // Описатель сертификата
    CMS_ContentInfo* cms;         // Описатель CMS структуры
    ASN1_OCTET_STRING* signature; // Подпись
    TS_REQ* request;              // Описатель структуры запроса метки времени
    TS_RESP* response; // Описатель структуры ответа сервера с меткой времени
    TS_VERIFY_CTX* ctx; // Описатель контекста проверки ответа сервера с меткой времени

    int r;             // Код возврата
    int errorCode = 1; // Флаг ошибки

    printf("Sample has started.\n\n");
    /*************************************************************************
     * Инициализация OPENSSL_crypto                                           *
     *************************************************************************/
    r = OPENSSL_init_crypto(OPENSSL_INIT_NO_LOAD_CONFIG | OPENSSL_INIT_NO_ATEXIT, NULL);
    CHECK("  OPENSSL_init_crypto", r, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_load_engine();
    CHECK("  rt_eng_load_engine", r == 1, exit);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    r = ENGINE_init(rtEngine);
    CHECK("  ENGINE_init", r == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK("  ENGINE_set_default", r == 1, finalize_engine);

    /*************************************************************************
     * Получение ключевой пары                                                *
     *************************************************************************/
    printf("  get_key_pair...\n");
    key = get_key_pair();
    CHECK("  get_key_pair", key != NULL, unregister_engine);

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    inBio = BIO_new_file(TEST_CERT_NAME, "r");
    CHECK("  BIO_new_file", inBio != NULL, free_key);

    /*************************************************************************
     * Чтение сертификата                                                     *
     *************************************************************************/
    cert = PEM_read_bio_X509(inBio, NULL, NULL, NULL);
    CHECK("  PEM_read_bio_X509", cert != NULL, free_in_bio);

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    BIO_free_all(inBio);
    inBio = BIO_new_file("test_data", "r");
    CHECK("  BIO_new_file", inBio != NULL, free_cert);

    /*************************************************************************
     * Создание и подпись CAdES-BES структуры                                 *
     *************************************************************************/
    cms = CMS_sign(cert, key, NULL, inBio, CMS_BINARY | CMS_NOSMIMECAP | CMS_CADES);
    CHECK("  CMS_sign", cms != NULL, free_cert);

    /*************************************************************************
     * Чтение подписи из CMS структуры                                        *
     *************************************************************************/
    signature = get_signature(cms);
    CHECK("  get_signature", signature != NULL, free_cms);

    /*************************************************************************
     * Создание запроса метки времени                                         *
     *************************************************************************/
    request = create_ts_request(signature);
    CHECK("  create_ts_request", request != NULL, free_cms);

    /*************************************************************************
     * Открытие поточного вывода в файл                                       *
     *************************************************************************/
    outBioRequest = BIO_new_file("requestCadesT.tsq", "wb");
    CHECK("  BIO_new_file", outBioRequest != NULL, free_ts_req);

    /*************************************************************************
     * Запись запроса метки времени                                           *
     *************************************************************************/
    r = i2d_TS_REQ_bio(outBioRequest, request);
    CHECK("  i2d_TS_REQ_bio", r == 1, free_out_bio);
    BIO_flush(outBioRequest);

    /*************************************************************************
     * Ожидание создания ответа на запрос метки времени                       *
     *************************************************************************/
    printf(
        "\n\nPlease create reply on the time stamp request (requestCadesT.tsq)\n"
        "(see item 13 in the openssl/samples/tool/README.md (responseCadesT.tsr is out parameter))\n"
        "and press Enter to continue.\n");
    fflush(stdout);
    getchar();

    /*************************************************************************
     * Создание контекста проверки метки времени из запроса                   *
     *************************************************************************/
    ctx = TS_REQ_to_TS_VERIFY_CTX(request, NULL);
    CHECK("  TS_REQ_to_TS_VERIFY_CTX", ctx != NULL, free_out_bio);

    /*************************************************************************
     * Выставление флага проверки подписи метки                               *
     *************************************************************************/
    TS_VERIFY_CTX_add_flags(ctx, TS_VFY_SIGNATURE);

    /*************************************************************************
     * Создание хранилища сертификатов и добавление его в контекст проверки   *
     *************************************************************************/
    r = TS_VERIFY_CTX_set_store(ctx, create_cert_store()) != NULL;
    CHECK("  TS_VERIFY_CTX_set_store", r == 1, free_ctx);

    /*************************************************************************
     * Добавление сертификата TSA в контекст проверки                         *
     *************************************************************************/
    r = TS_VERIFY_CTS_set_certs(ctx, get_tsa_cert()) != NULL;
    CHECK("  TS_VERIFY_CTS_set_certs", r == 1, free_ctx);

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    inBioResponse = BIO_new_file("responseCadesT.tsr", "rb");
    CHECK("  BIO_new_file", inBioResponse != NULL, free_ctx);

    /*************************************************************************
     * Чтение ответа сервера с меткой времени                                 *
     *************************************************************************/
    response = d2i_TS_RESP_bio(inBioResponse, NULL);
    CHECK("  d2i_TS_RESP_bio", response != NULL, free_in_bio_response);

    /*************************************************************************
     * Проверка ответа от сервера                                             *
     *************************************************************************/
    r = TS_RESP_verify_response(ctx, response);
    CHECK("  TS_RESP_verify_response", r == 1, free_ts_resp);

    /*************************************************************************
     * Добавление метки времени в качестве неподписанного атрибута CMS        *
     *************************************************************************/
    r = add_tst_to_cms(cms, response);
    CHECK("  add_tst_to_cms", r == 1, free_ts_resp);

    /*************************************************************************
     * Открытие поточного вывода в файл                                       *
     *************************************************************************/
    outBioCadesT = BIO_new_file("cadesT.pem", "wb");
    CHECK("  BIO_new_file", outBioCadesT != NULL, free_ts_resp);

    /*************************************************************************
     * Запись CMS структуры                                                   *
     *************************************************************************/
    r = PEM_write_bio_CMS(outBioCadesT, cms);
    CHECK("  PEM_write_bio_CMS", r > 0, free_out_bio_result);

    /*************************************************************************
     * Установка признака успешного завершения программы                      *
     *************************************************************************/
    errorCode = 0;
free_out_bio_result:

    /*************************************************************************
     * Закрытие потока вывода для результирующей CMS                          *
     *************************************************************************/
    BIO_free_all(outBioCadesT);
free_ts_resp:

    /*************************************************************************
     * Освобождение TS_RESP структуры                                         *
     *************************************************************************/
    TS_RESP_free(response);
free_in_bio_response:

    /*************************************************************************
     * Закрытие потока ввода для ответа от сервера                            *
     *************************************************************************/
    BIO_free_all(inBioResponse);
free_ctx:

    /*************************************************************************
     * Освобождение контекста проверки                                        *
     *************************************************************************/
    TS_VERIFY_CTX_free(ctx);
free_out_bio:

    /*************************************************************************
     * Закрытие потока вывода                                                 *
     *************************************************************************/
    BIO_free_all(outBioRequest);
free_ts_req:

    /*************************************************************************
     * Освобождение TS_REQ структуры                                          *
     *************************************************************************/
    TS_REQ_free(request);
free_cms:

    /*************************************************************************
     * Освобождение CMS структуры                                             *
     *************************************************************************/
    CMS_ContentInfo_free(cms);
free_cert:

    /*************************************************************************
     * Освобождние сертификата                                                *
     *************************************************************************/
    X509_free(cert);
free_in_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
free_key:

    /*************************************************************************
     * Освобождение описателя ключевой пары                                   *
     *************************************************************************/
    printf("  free_key_pair...\n");
    r = free_key_pair(key);
    CHECK_RELEASE("  free_key_pair", r == 0, errorCode);
unregister_engine:

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    r = ENGINE_finish(rtEngine);
    CHECK_RELEASE("  ENGINE_finish", r == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_unload_engine();
    CHECK_RELEASE("  rt_eng_unload_engine", r == 1, errorCode);
exit:
    OPENSSL_cleanup();
    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}

ASN1_OCTET_STRING* get_signature(CMS_ContentInfo* cms) {
    ASN1_OCTET_STRING* signature = NULL; // Описатель подписи
    CMS_SignerInfo* signerInfo;          // Описатель структуры подписанта
    STACK_OF(CMS_SignerInfo) * signers;  // Описатель контейнера подписантов
    int num;                             // Количество подписантов в CMS подписи

    /*************************************************************************
     * Получение подписантов                                                  *
     *************************************************************************/
    signers = CMS_get0_SignerInfos(cms);
    CHECK("    CMS_get0_SignerInfos", signers != NULL, exit);

    /*************************************************************************
     * Проверка количества подписантов                                        *
     *************************************************************************/
    num = sk_CMS_SignerInfo_num(signers);
    CHECK("    sk_CMS_SignerInfo_num", num > 0, exit);

    /*************************************************************************
     * Получение первого подписанта                                           *
     *************************************************************************/
    signerInfo = sk_CMS_SignerInfo_value(signers, 0);
    CHECK("    sk_CMS_SignerInfo_value", signerInfo != NULL, exit);

    /*************************************************************************
     * Получение подписи первого подписанта                                   *
     *************************************************************************/
    signature = CMS_SignerInfo_get0_signature(signerInfo);
    CHECK("    CMS_SignerInfo_get0_signature", signature != NULL, exit);
exit:
    return signature;
}

TS_REQ* create_ts_request(ASN1_OCTET_STRING* signature) {
    TS_MSG_IMPRINT* msgImprint; // Описатель структуры TS_MSG_IMPRINT
    X509_ALGOR* algo;           // Описатель структуры алгоритма хэширования
    TS_REQ* req = NULL;         // Описатель структуры запроса
    const EVP_MD* md = NULL;    // Описатель структуры EVP_MD
    unsigned char* digest;      // Хэш от подписи
    int digestLen;              // Длина хэша
    uint64_t nonce = 0;         // Случайное 64-битное число
    ASN1_INTEGER* asnNonce; // Описатель струткуры ASN1_INTEGER, представляющей случайное число

    int r; // Код возврата

    /*************************************************************************
     * Вычисление хэша от подписи                                             *
     *************************************************************************/
    md = EVP_get_digestbyname("md_gost12_256");
    CHECK("    EVP_get_digestbyname", md != NULL, exit);

    digestLen = EVP_MD_size(md);
    digest = OPENSSL_malloc(digestLen);
    CHECK("    OPENSSL_malloc", digest != NULL, exit);

    r = EVP_Digest(ASN1_STRING_get0_data(signature), ASN1_STRING_length(signature), digest, NULL, md, NULL);
    CHECK("    EVP_Digest", r == 1, free_digest);

    /*************************************************************************
     * Создание структуры TS_MSG_IMPRINT                                      *
     *************************************************************************/
    msgImprint = TS_MSG_IMPRINT_new();
    CHECK("    TS_MSG_IMPRINT_new", msgImprint != NULL, free_digest);

    algo = X509_ALGOR_new();
    CHECK("    X509_ALGOR_new", algo != NULL, free_msg_imprint);

    X509_ALGOR_set_md(algo, md);

    r = TS_MSG_IMPRINT_set_algo(msgImprint, algo);
    CHECK("    TS_MSG_IMPRINT_set_algo", r == 1, free_algo);

    r = TS_MSG_IMPRINT_set_msg(msgImprint, digest, digestLen);
    CHECK("    TS_MSG_IMPRINT_set_msg", r == 1, free_algo);

    /*************************************************************************
     * Создание nonce                                                         *
     *************************************************************************/
    r = RAND_bytes((unsigned char*)&nonce, sizeof(nonce));
    CHECK("    RAND_bytes", r == 1, free_algo);

    asnNonce = ASN1_INTEGER_new();
    CHECK("    ASN1_INTEGER_new", asnNonce != NULL, free_algo);

    r = ASN1_INTEGER_set_uint64(asnNonce, nonce);
    CHECK("    ASN1_INTEGER_set_uint64", r == 1, free_nonce);

    /*************************************************************************
     * Создание структуры TS_REQ                                              *
     *************************************************************************/
    req = TS_REQ_new();
    CHECK("    TS_REQ_new", req != NULL, free_nonce);

    r = TS_REQ_set_version(req, 1);
    CHECK("    TS_REQ_set_version", r == 1, free_req);

    r = TS_REQ_set_msg_imprint(req, msgImprint);
    CHECK("    TS_REQ_set_msg_imprint", r == 1, free_req);

    r = TS_REQ_set_nonce(req, asnNonce);
    CHECK("    TS_REQ_set_nonce", r == 1, free_req);

    goto free_nonce;
free_req:

    /*************************************************************************
     * Освобождение структуры запроса                                         *
     *************************************************************************/
    TS_REQ_free(req);
    req = NULL;
free_nonce:

    /*************************************************************************
     * Освобождение nonce                                                     *
     *************************************************************************/
    ASN1_INTEGER_free(asnNonce);
free_algo:

    /*************************************************************************
     * Освобождение структуры алгоритма хэширования                           *
     *************************************************************************/
    X509_ALGOR_free(algo);
free_msg_imprint:

    /*************************************************************************
     * Освобождение структуры TS_MSG_IMPRINT                                  *
     *************************************************************************/
    TS_MSG_IMPRINT_free(msgImprint);
free_digest:

    /*************************************************************************
     * Освобождение хэша                                                      *
     *************************************************************************/
    OPENSSL_free(digest);
exit:
    return req;
}

X509_STORE* create_cert_store() {
    BIO* inBio;               // Описатель потока ввода
    X509* caCert;             // Описатель сертификата удостоверяющего центра
    X509_STORE* store = NULL; // Описатель хранилища сертификатов

    int r; // Код возврата

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    inBio = BIO_new_file("test_trusted_ca.cer", "rb");
    CHECK("    BIO_new_file", inBio != NULL, exit);

    /*************************************************************************
     * Чтение сертификата удостоверяющего центра                              *
     *************************************************************************/
    caCert = PEM_read_bio_X509(inBio, NULL, NULL, NULL);
    CHECK("    PEM_read_bio_X509", caCert != NULL, free_in_bio);

    /*************************************************************************
     * Создание хранилища сертификатов                                        *
     *************************************************************************/
    store = X509_STORE_new();
    CHECK("    X509_STORE_new", store != NULL, free_cert);

    /*************************************************************************
     * Добавление сертификата удостоверяющего центра в хранилище              *
     *************************************************************************/
    r = X509_STORE_add_cert(store, caCert);
    CHECK("    X509_STORE_add_cert", r == 1, free_store);

    goto free_cert;
free_store:

    /*************************************************************************
     * Освобождение хранилища сертификатов                                    *
     *************************************************************************/
    X509_STORE_free(store);
    store = NULL;
free_cert:

    /*************************************************************************
     * Освобождение сертификата                                               *
     *************************************************************************/
    X509_free(caCert);
free_in_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
exit:
    return store;
}

STACK_OF(X509) * get_tsa_cert() {
    BIO* inBio;                   // Описатель потока ввода
    STACK_OF(X509)* certs = NULL; // Описатель контейнера сертификатов
    X509* tsaCert;                // Описатель сертификата

    int r; // Код возврата

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    inBio = BIO_new_file("test_tsa_cert.cer", "rb");
    CHECK("    BIO_new_file", inBio != NULL, exit);

    /*************************************************************************
     * Чтение сертификата                                                     *
     *************************************************************************/
    tsaCert = PEM_read_bio_X509(inBio, NULL, NULL, NULL);
    CHECK("    PEM_read_bio_X509", tsaCert != NULL, free_in_bio);

    /*************************************************************************
     * Создание контейнера сертификатов                                       *
     *************************************************************************/
    certs = sk_X509_new_null();
    CHECK("    sk_X509_new_null", certs != NULL, free_cert);

    /*************************************************************************
     * Добавление сертификата в контейнер                                     *
     *************************************************************************/
    r = sk_X509_push(certs, tsaCert);
    CHECK("    sk_X509_push", r == 1, free_sk_certs);

    goto free_in_bio;
free_sk_certs:

    /*************************************************************************
     * Освобождение контейнера сертификатов                                   *
     *************************************************************************/
    sk_X509_free(certs);
    certs = NULL;
free_cert:

    /*************************************************************************
     * Освобождение сертификата                                               *
     *************************************************************************/
    X509_free(tsaCert);
free_in_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
exit:
    return certs;
}

int add_tst_to_cms(CMS_ContentInfo* cms, TS_RESP* response) {
    PKCS7* token;                       // Метка времени
    unsigned char* tokenBuf = NULL;     // Закодированная метка времени
    int tokenBufLen;                    // Длина закодированной метки времени
    STACK_OF(CMS_SignerInfo) * signers; // Описатель контейнера подписантов
    CMS_SignerInfo* signerInfo;         // Описатель структуры подписанта
    int num;                            // Количество подписантов в CMS подписи

    int r = 0; // Код возврата

    /*************************************************************************
     * Получение метки времени из ответа сервера                              *
     *************************************************************************/
    token = TS_RESP_get_token(response);
    CHECK("    TS_RESP_get_token", token != NULL, exit);

    /*************************************************************************
     * DER кодирование метки времени                                          *
     *************************************************************************/
    tokenBufLen = i2d_PKCS7(token, &tokenBuf);
    CHECK("    i2d_PKCS7", tokenBufLen >= 0, exit);

    /*************************************************************************
     * Получение подписантов                                                  *
     *************************************************************************/
    signers = CMS_get0_SignerInfos(cms);
    CHECK("    CMS_get0_SignerInfos", signers != NULL, free_tokenbuf);

    /*************************************************************************
     * Проверка количества подписантов                                        *
     *************************************************************************/
    num = sk_CMS_SignerInfo_num(signers);
    CHECK("    sk_CMS_SignerInfo_num", num > 0, free_tokenbuf);

    /*************************************************************************
     * Получение первого подписанта                                           *
     *************************************************************************/
    signerInfo = sk_CMS_SignerInfo_value(signers, 0);
    CHECK("    sk_CMS_SignerInfo_value", signerInfo != NULL, free_tokenbuf);

    /*************************************************************************
     * Добавление метки времени в CMS в качестве неподписанного атрибута      *
     *************************************************************************/
    r = CMS_unsigned_add1_attr_by_NID(signerInfo, NID_id_smime_aa_timeStampToken, V_ASN1_SEQUENCE, tokenBuf,
                                      tokenBufLen);
    CHECK("    CMS_unsigned_add1_attr_by_NID", r == 1, free_tokenbuf);

free_tokenbuf:

    /*************************************************************************
     * Освобождение закодированной метки времени                              *
     *************************************************************************/
    OPENSSL_free(tokenBuf);
exit:
    return r;
}
