/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/*************************************************************************
 * Этот пример использует CMS, созданный примером SignCMS.                *
 *************************************************************************/

#include <assert.h>

#include <Common.h>

#include <openssl/cms.h>
#include <openssl/ts.h>

ASN1_OCTET_STRING* get_signature(CMS_ContentInfo* cms);
TS_REQ* create_ts_request(ASN1_OCTET_STRING* signature);

int main(void) {
    ENGINE* rtEngine; // rtengine
    BIO* inBio;       // Описатель потока ввода
    BIO* outBio;      // Описатель потока вывода

    CMS_ContentInfo* cms;         // Описатель CMS структуры
    ASN1_OCTET_STRING* signature; // Подпись
    TS_REQ* request;              // Описатель структуры запроса метки времени

    int r;             // Код возврата
    int errorCode = 1; // Флаг ошибки

    printf("Sample has started.\n\n");
    /*************************************************************************
     * Инициализация OPENSSL_crypto                                           *
     *************************************************************************/
    r = OPENSSL_init_crypto(OPENSSL_INIT_NO_LOAD_CONFIG | OPENSSL_INIT_NO_ATEXIT, NULL);
    CHECK("  OPENSSL_init_crypto", r, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_load_engine();
    CHECK("  rt_eng_load_engine", r == 1, exit);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    r = ENGINE_init(rtEngine);
    CHECK("  ENGINE_init", r == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK("  ENGINE_set_default", r == 1, finalize_engine);

    /*************************************************************************
     * Открытие потока ввода                                                  *
     *************************************************************************/
    inBio = BIO_new_file("cms_signed.pem", "rb");
    CHECK("  BIO_new_file", inBio != NULL, unregister_engine);

    /*************************************************************************
     * Чтение CMS структуры                                                   *
     *************************************************************************/
    cms = PEM_read_bio_CMS(inBio, NULL, NULL, NULL);
    CHECK("  PEM_read_bio_CMS", cms != NULL, free_in_bio);

    /*************************************************************************
     * Чтение подписи из CMS структуры                                        *
     *************************************************************************/
    signature = get_signature(cms);
    CHECK("  get_signature", signature != NULL, free_cms);

    /*************************************************************************
     * Создание запроса метки времени                                         *
     *************************************************************************/
    request = create_ts_request(signature);
    CHECK("  create_ts_request", request != NULL, free_cms);

    /*************************************************************************
     * Открытие поточного вывода из файла                                     *
     *************************************************************************/
    outBio = BIO_new_file("request.tsq", "wb");
    CHECK("  BIO_new_file", outBio != NULL, free_ts_req);

    /*************************************************************************
     * Запись запроса метки времени                                           *
     *************************************************************************/
    r = i2d_TS_REQ_bio(outBio, request);
    CHECK("  i2d_TS_REQ_bio", r == 1, free_out_bio);

    /*************************************************************************
     * Установка признака успешного завершения программы                      *
     *************************************************************************/
    errorCode = 0;
free_out_bio:

    /*************************************************************************
     * Закрытие потока вывода                                                 *
     *************************************************************************/
    BIO_free_all(outBio);
free_ts_req:

    /*************************************************************************
     * Освобождение TS_REQ структуры                                          *
     *************************************************************************/
    TS_REQ_free(request);
free_cms:

    /*************************************************************************
     * Освобождение CMS структуры                                             *
     *************************************************************************/
    CMS_ContentInfo_free(cms);
free_in_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
unregister_engine:

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    r = ENGINE_finish(rtEngine);
    CHECK_RELEASE("  ENGINE_finish", r == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_unload_engine();
    CHECK_RELEASE("  rt_eng_unload_engine", r == 1, errorCode);
exit:
    OPENSSL_cleanup();
    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}

ASN1_OCTET_STRING* get_signature(CMS_ContentInfo* cms) {
    ASN1_OCTET_STRING* signature = NULL; // Описатель подписи
    CMS_SignerInfo* signerInfo;          // Описатель структуры подписанта
    STACK_OF(CMS_SignerInfo) * signers;  // Описатель контейнера подписантов
    int num;                             // Количество подписантов в CMS подписи

    /*************************************************************************
     * Получение подписантов                                                  *
     *************************************************************************/
    signers = CMS_get0_SignerInfos(cms);
    CHECK("    CMS_get0_SignerInfos", signers != NULL, exit);

    /*************************************************************************
     * Проверка количества подписантов                                        *
     *************************************************************************/
    num = sk_CMS_SignerInfo_num(signers);
    CHECK("    sk_CMS_SignerInfo_num", num > 0, exit);

    /*************************************************************************
     * Получение первого подписанта                                           *
     *************************************************************************/
    signerInfo = sk_CMS_SignerInfo_value(signers, 0);
    CHECK("    sk_CMS_SignerInfo_value", signerInfo != NULL, exit);

    /*************************************************************************
     * Получение подписи первого подписанта                                   *
     *************************************************************************/
    signature = CMS_SignerInfo_get0_signature(signerInfo);
    CHECK("    CMS_SignerInfo_get0_signature", signature != NULL, exit);
exit:
    return signature;
}

TS_REQ* create_ts_request(ASN1_OCTET_STRING* signature) {
    TS_MSG_IMPRINT* msgImprint; // Описатель структуры TS_MSG_IMPRINT
    X509_ALGOR* algo;           // Описатель структуры алгоритма хэширования
    TS_REQ* req = NULL;         // Описатель структуры запроса
    const EVP_MD* md = NULL;    // Описатель структуры EVP_MD
    unsigned char* digest;      // Хэш от подписи
    int digestLen;              // Длина хэша
    uint64_t nonce = 0;         // Случайное 64-битное число
    ASN1_INTEGER* asnNonce; // Описатель струткуры ASN1_INTEGER, представляющей случайное число

    int r; // Код возврата

    /*************************************************************************
     * Вычисление хэша от подписи                                             *
     *************************************************************************/
    md = EVP_get_digestbyname("md_gost12_256");
    CHECK("    EVP_get_digestbyname", md != NULL, exit);

    digestLen = EVP_MD_size(md);
    digest = OPENSSL_malloc(digestLen);
    CHECK("    OPENSSL_malloc", digest != NULL, exit);

    r = EVP_Digest(ASN1_STRING_get0_data(signature), ASN1_STRING_length(signature), digest, NULL, md, NULL);
    CHECK("    EVP_Digest", r == 1, free_digest);

    /*************************************************************************
     * Создание структуры TS_MSG_IMPRINT                                      *
     *************************************************************************/
    msgImprint = TS_MSG_IMPRINT_new();
    CHECK("    TS_MSG_IMPRINT_new", msgImprint != NULL, free_digest);

    algo = X509_ALGOR_new();
    CHECK("    X509_ALGOR_new", algo != NULL, free_msg_imprint);

    X509_ALGOR_set_md(algo, md);

    r = TS_MSG_IMPRINT_set_algo(msgImprint, algo);
    CHECK("    TS_MSG_IMPRINT_set_algo", r == 1, free_algo);

    r = TS_MSG_IMPRINT_set_msg(msgImprint, digest, digestLen);
    CHECK("    TS_MSG_IMPRINT_set_msg", r == 1, free_algo);

    /*************************************************************************
     * Создание nonce                                                         *
     *************************************************************************/
    r = RAND_bytes((unsigned char*)&nonce, sizeof(nonce));
    CHECK("    RAND_bytes", r == 1, free_algo);

    asnNonce = ASN1_INTEGER_new();
    CHECK("    ASN1_INTEGER_new", asnNonce != NULL, free_algo);

    r = ASN1_INTEGER_set_uint64(asnNonce, nonce);
    CHECK("    ASN1_INTEGER_set_uint64", r == 1, free_nonce);

    /*************************************************************************
     * Создание структуры TS_REQ                                              *
     *************************************************************************/
    req = TS_REQ_new();
    CHECK("    TS_REQ_new", req != NULL, free_nonce);

    r = TS_REQ_set_version(req, 1);
    CHECK("    TS_REQ_set_version", r == 1, free_req);

    r = TS_REQ_set_msg_imprint(req, msgImprint);
    CHECK("    TS_REQ_set_msg_imprint", r == 1, free_req);

    r = TS_REQ_set_nonce(req, asnNonce);
    CHECK("    TS_REQ_set_nonce", r == 1, free_req);

    goto free_nonce;
free_req:

    /*************************************************************************
     * Освобождение структуры запроса                                         *
     *************************************************************************/
    TS_REQ_free(req);
    req = NULL;
free_nonce:

    /*************************************************************************
     * Освобождение nonce                                                     *
     *************************************************************************/
    ASN1_INTEGER_free(asnNonce);
free_algo:

    /*************************************************************************
     * Освобождение структуры алгоритма хэширования                           *
     *************************************************************************/
    X509_ALGOR_free(algo);
free_msg_imprint:

    /*************************************************************************
     * Освобождение структуры TS_MSG_IMPRINT                                  *
     *************************************************************************/
    TS_MSG_IMPRINT_free(msgImprint);
free_digest:

    /*************************************************************************
     * Освобождение хэша                                                      *
     *************************************************************************/
    OPENSSL_free(digest);
exit:
    return req;
}
