/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/************************************************************************
 * Для работы данного примера необходим сертификат, выписанный на        *
 * ключевой паре, которая будет использована в примере. О том, как       *
 * выписать сертификат на имеющуюся ключевую пару, можно узнать из       *
 * описания к использованию rtengine через инструменты командной строки  *
 * OpenSSL настоящего пакета SDK (openssl/samples/tool/README.md)        *
 * ----------------------------------------------------------------------*
 * Для работы примера с ECDSA и RSA ключами нужно внести соответствующие *
 * изменения в Common.h и TokenPreparation.c                             *
 ************************************************************************/

#include <assert.h>

#include <Common.h>

#include <openssl/cms.h>

int main(void) {
    ENGINE* rtEngine; // rtengine
    BIO* inBio;       // Описатель потока ввода
    BIO* outBio;      // Описатель потока вывода

    STACK_OF(X509) * certs;   // Описатель контейнера сертификатов
    X509* cert;               // Описатель сертификата
    CMS_ContentInfo* cms;     // Описатель CMS структуры
    const EVP_CIPHER* cipher; // Описатель алгоритма шифрования

    int r;             // Код возврата
    int errorCode = 1; // Флаг ошибки

    printf("Sample has started.\n\n");
    /*************************************************************************
     * Инициализация OPENSSL_crypto                                           *
     *************************************************************************/
    r = OPENSSL_init_crypto(OPENSSL_INIT_NO_LOAD_CONFIG | OPENSSL_INIT_NO_ATEXIT, NULL);
    CHECK("  OPENSSL_init_crypto", r, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_load_engine();
    CHECK("  rt_eng_load_engine", r == 1, exit);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    r = ENGINE_init(rtEngine);
    CHECK("  ENGINE_init", r == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK("  ENGINE_set_default", r == 1, finalize_engine);

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    inBio = BIO_new_file(TEST_CERT_NAME, "r");
    CHECK("  BIO_new_file", inBio != NULL, unregister_engine);

    /*************************************************************************
     * Чтение сертификата из файла                                            *
     *************************************************************************/
    cert = PEM_read_bio_X509(inBio, NULL, NULL, NULL);
    CHECK("  PEM_read_bio_X509", cert != NULL, free_in_bio);

    /*************************************************************************
     * Создание контейнера сертификатов                                       *
     *************************************************************************/
    certs = sk_X509_new_null();
    CHECK("  sk_X509_new_null", certs != NULL, free_cert);

    /*************************************************************************
     * Добавление сертификата в контейнер                                     *
     *************************************************************************/
    r = sk_X509_push(certs, cert);
    CHECK("  sk_X509_push", r == 1, free_sk_certs);

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    BIO_free_all(inBio);
    inBio = BIO_new_file("test_data", "r");
    CHECK("  BIO_new_file", inBio != NULL, free_sk_certs);

    /*************************************************************************
     * Получение алгоритма шифрования                                         *
     *************************************************************************/
    /*************************************************************************
     * Для работы с ECDSA ключами следует заменить                            *
     * EVP_get_cipherbynid(rt_eng_nid_gost28147_cfb) на EVP_aes_128_cbc()     *
     *************************************************************************/
    cipher = EVP_get_cipherbynid(rt_eng_nid_gost28147_cfb);
    CHECK("  EVP_get_cipherbynid", cipher != NULL, free_sk_certs);

    /*************************************************************************
     * Создание CMS структуры с шифротекстом                                  *
     *************************************************************************/
    cms = CMS_encrypt(certs, inBio, cipher, CMS_KEY_PARAM);
    CHECK("  CMS_encrypt", cms != NULL, free_sk_certs);

    /*************************************************************************
     * Открытие поточного вывода в файл                                       *
     *************************************************************************/
    outBio = BIO_new_file("cms_encrypted.pem", "w");
    CHECK("  BIO_new_file", outBio != NULL, free_cms);

    /*************************************************************************
     * Запись CMS структуры в файл                                            *
     *************************************************************************/
    r = PEM_write_bio_CMS(outBio, cms);
    CHECK("  PEM_write_bio_CMS", r == 1, free_out_bio);

    /*************************************************************************
     * Установка признака успешного завершения программы                      *
     *************************************************************************/
    errorCode = 0;
free_out_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(outBio);
free_cms:

    /*************************************************************************
     * Освобождение CMS структуры                                             *
     *************************************************************************/
    CMS_ContentInfo_free(cms);
free_sk_certs:

    /*************************************************************************
     * Освобождение контейнера с сертификатами                                *
     *************************************************************************/
    sk_X509_free(certs);
free_cert:

    /*************************************************************************
     * Освобождение сертификата                                               *
     *************************************************************************/
    X509_free(cert);
free_in_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
unregister_engine:

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    r = ENGINE_finish(rtEngine);
    CHECK_RELEASE("  ENGINE_finish", r == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_unload_engine();
    CHECK_RELEASE("  rt_eng_unload_engine", r == 1, errorCode);
exit:
    OPENSSL_cleanup();
    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}
