/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

#include <assert.h>

#include <Common.h>

int main(void) {
    EVP_PKEY* key;     // Описатель ключевой пары
    ENGINE* rtEngine;  // rtengine
    EVP_PKEY_CTX* ctx; // Контекст подписи
    FILE* outputFile;  // Описатель потока вывода
    uint8_t hash[] = {
        0x00, 0x90, 0x31, 0xCF, 0x95, 0x7A, 0x59, 0xE5, 0xD2, 0xBF, 0x2C, 0xDB, 0xB5, 0x83, 0x4D, 0x03,
        0x17, 0x5D, 0x25, 0x2A, 0xFD, 0x72, 0x1E, 0x01, 0x02, 0x60, 0x88, 0x92, 0x9A, 0x9B, 0x2A, 0xA9
    }; // Хеш для подписи

    size_t siglen;            // Длина подписи
    unsigned char* signature; // Буфер с подписью

    size_t numberOfBytes; // Число записанных байт
    int r;                // Код возврата
    int errorCode = 1;    // Флаг ошибки

    printf("Sample has started.\n\n");
    /*************************************************************************
     * Инициализация OPENSSL_crypto                                           *
     *************************************************************************/
    r = OPENSSL_init_crypto(OPENSSL_INIT_NO_LOAD_CONFIG | OPENSSL_INIT_NO_ATEXIT, NULL);
    CHECK("  OPENSSL_init_crypto", r, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_load_engine();
    CHECK("  rt_eng_load_engine", r == 1, exit);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    r = ENGINE_init(rtEngine);
    CHECK("  ENGINE_init", r == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK("  ENGINE_set_default", r == 1, finalize_engine);

    /*************************************************************************
     * Получение ключевой пары                                                *
     *************************************************************************/
    printf("  get_key_pair...\n");
    key = get_key_pair();
    CHECK("  get_key_pair", key != NULL, unregister_engine);

    /*************************************************************************
     * Создание контекста подписи                                             *
     *************************************************************************/
    ctx = EVP_PKEY_CTX_new(key, rtEngine);
    CHECK("  EVP_PKEY_CTX_new", ctx != NULL, free_key);

    /*************************************************************************
     * Инициализация контекста подписи                                        *
     *************************************************************************/
    r = EVP_PKEY_sign_init(ctx);
    CHECK("  EVP_PKEY_sign_init", r == 1, free_context);

    /*************************************************************************
     * Получение размера подписи                                              *
     *************************************************************************/
    r = EVP_PKEY_sign(ctx, NULL, &siglen, NULL, 0);
    CHECK("  EVP_PKEY_sign", r == 1, free_context);

    /*************************************************************************
     * Выделение памяти для подписи                                           *
     *************************************************************************/
    signature = OPENSSL_malloc(siglen);
    CHECK("  OPENSSL_malloc", signature != NULL, free_context);

    /*************************************************************************
     * Подписание                                                             *
     *************************************************************************/
    r = EVP_PKEY_sign(ctx, signature, &siglen, hash, sizeof(hash));
    CHECK("  EVP_PKEY_sign", r == 1, free_signature);

    /*************************************************************************
     * Открытие поточного вывода                                              *
     *************************************************************************/
    outputFile = fopen("signature", "wb");
    CHECK("  fopen", outputFile != NULL, free_signature);

    /*************************************************************************
     * Запись подписи в файл                                                  *
     *************************************************************************/
    numberOfBytes = fwrite(signature, 1, siglen, outputFile);
    CHECK("  fwrite", numberOfBytes == siglen, close_file);

    /*************************************************************************
     * Установка признака успешного завершения программы                      *
     *************************************************************************/
    errorCode = 0;
close_file:

    /*************************************************************************
     * Закрытие потока вывода                                                 *
     *************************************************************************/
    r = fclose(outputFile);
    CHECK_RELEASE("  fclose", r == 0, errorCode);
free_signature:

    /*************************************************************************
     * Освобождение буфера с подписью                                         *
     *************************************************************************/
    OPENSSL_free(signature);
free_context:

    /*************************************************************************
     * Освобождение контекста                                                 *
     *************************************************************************/
    EVP_PKEY_CTX_free(ctx);
free_key:

    /*************************************************************************
     * Освобождение описателя ключевой пары                                   *
     *************************************************************************/
    printf("  free_key_pair...\n");
    r = free_key_pair(key);
    CHECK_RELEASE("  free_key_pair", r == 0, errorCode);
unregister_engine:

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    r = ENGINE_finish(rtEngine);
    CHECK_RELEASE("  ENGINE_finish", r == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_unload_engine();
    CHECK_RELEASE("  rt_eng_unload_engine", r == 1, errorCode);
exit:
    OPENSSL_cleanup();
    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}
