/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/*************************************************************************
 * Данный пример подготавливает Рутокен к выполнению примеров,            *
 * использующих ключевую пару с Рутокен. Перед запуском этого примера     *
 * необходимо отформатировать токен или другим способом удалить все       *
 * ключевые пары с него (например, используя KeyDeleting)                 *
 * Выполняются следующие действия:                                        *
 * - генерируется ключевая пара ГОСТ 34.10-2012 (256/512 бит) на Рутокен; *
 * - для данной ключевой пары выписывается сертификат при помощи          *
 *   тестового удостверяющего центра, который используется в примерах     *
 *   работы с OpenSSL tool                                                *
 *************************************************************************/
#include <assert.h>

#include <Common.h>

#include <openssl/x509.h>
#include <openssl/x509v3.h>

/************************************************************************
 * Описание типов объектов                                               *
 ************************************************************************/
CK_OBJECT_CLASS publicKeyObject = CKO_PUBLIC_KEY;
CK_OBJECT_CLASS privateKeyObject = CKO_PRIVATE_KEY;

/************************************************************************
 * Описание типа ключа ГОСТ                                              *
 ************************************************************************/
/*************************************************************************
 * Для использования ГОСТ ключа длиной 512 бит необходимо заменить        *
 * CKK_GOSTR3410 на CKK_GOSTR3410_512                                     *
 *************************************************************************/
CK_KEY_TYPE keyTypeGostR3410 = CKK_GOSTR3410;

/************************************************************************
 * Описание типа ключа ECDSA                                             *
 ************************************************************************/
CK_KEY_TYPE keyTypeEcdsa = CKK_EC;

/************************************************************************
 * Описание типа ключа RSA                                              *
 ************************************************************************/
CK_KEY_TYPE keyTypeRsa = CKK_RSA;

/*************************************************************************
 * Длина модуля ключа RSA в битах                                        *
 * Допустимые значения 512, 1024, 2048, 4096.                            *
 * Используйте значения, поддерживаемые токеном                          *
 **************************************************************************/
CK_ULONG rsaModulusBits = 1024;

/************************************************************************
 * Набор параметров КриптоПро A алгоритма ГОСТ Р 34.10-2012              *
 ************************************************************************/

/* Набор параметров для ключа длиной 256 */
CK_BYTE parametersGostR3410_2012_256[] = { 0x06, 0x07, 0x2a, 0x85, 0x03, 0x02, 0x02, 0x23, 0x01 };
/* Набор параметров для ключа длиной 512 */
CK_BYTE parametersGostR3410_2012_512[] = { 0x06, 0x09, 0x2a, 0x85, 0x03, 0x07, 0x01, 0x02, 0x01, 0x02, 0x01 };

/************************************************************************
 * Набор параметров КриптоПро алгоритма ГОСТ Р 34.11-2012                *
 ************************************************************************/

/* Набор параметров для ключа длиной 256 */
CK_BYTE parametersGostR3411_2012_256[] = { 0x06, 0x08, 0x2a, 0x85, 0x03, 0x07, 0x01, 0x01, 0x02, 0x02 };
/* Набор параметров для ключа длиной 512 */
CK_BYTE parametersGostR3411_2012_512[] = { 0x06, 0x08, 0x2a, 0x85, 0x03, 0x07, 0x01, 0x01, 0x02, 0x03 };

/************************************************************************
 * Набор параметров ключа ECDSA для кривой secp256k1                     *
 ************************************************************************/
CK_BYTE secp256k1Oid[] = { 0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x0A };

/************************************************************************
 * Механизм генерации ключевой пары ГОСТ Р 34.10-2012-256                *
 ************************************************************************/
/************************************************************************
 * Для использования ключа длиной 512 бит необходимо заменить            *
 * CKM_GOSTR3410_KEY_PAIR_GEN на CKM_GOSTR3410_512_KEY_PAIR_GEN          *
 ************************************************************************/
CK_MECHANISM gostR3410KeyPairGenMech = { CKM_GOSTR3410_KEY_PAIR_GEN, NULL_PTR, 0 };

/************************************************************************
 * Механизм генерации ключевой пары ECDSA                                *
 ************************************************************************/
CK_MECHANISM ecdsaKeyPairGenMech = { CKM_EC_KEY_PAIR_GEN, NULL_PTR, 0 };

/***********************************************************************
 * Механизм генерации ключевой пары RSA                                *
 ***********************************************************************/
CK_MECHANISM rsaKeyPairGenMech = { CKM_RSA_PKCS_KEY_PAIR_GEN, NULL_PTR, 0 };

/************************************************************************
 * Вспомогательные переменные                                            *
 ************************************************************************/
CK_BBOOL attributeTrue = CK_TRUE;
CK_BBOOL attributeFalse = CK_FALSE;

/*************************************************************************
 * Шаблон для генерации открытого ключа ГОСТ Р 34.10-2012                 *
 *************************************************************************/
/*************************************************************************
 * Для использования ключа длиной 512 бит необходимо заменить             *
 * parametersGostR3410_2012_256 на parametersGostR3410_2012_512 и         *
 * parametersGostR3411_2012_256 на parametersGostR3411_2012_512           *
 *************************************************************************/
CK_ATTRIBUTE gostPublicKeyTemplate[] = {
    { CKA_CLASS, &publicKeyObject, sizeof(publicKeyObject) }, // Класс - открытый ключ
    { CKA_ID, &g_keyPairIdGost2012RtEngine,
      sizeof(g_keyPairIdGost2012RtEngine) -
          1 }, // Идентификатор ключевой пары (должен совпадать у открытого и закрытого ключей)
    { CKA_KEY_TYPE, &keyTypeGostR3410, sizeof(keyTypeGostR3410) }, // Тип ключа - ГОСТ Р 34.10-2012
    { CKA_TOKEN, &attributeTrue, sizeof(attributeTrue) }, // Ключ является объектом токена
    { CKA_PRIVATE, &attributeFalse, sizeof(attributeFalse) }, // Ключ доступен без аутентификации на токене
    { CKA_GOSTR3410_PARAMS, parametersGostR3410_2012_256,
      sizeof(parametersGostR3410_2012_256) }, // Параметры алгоритма ГОСТ Р 34.10-2012
    { CKA_GOSTR3411_PARAMS, parametersGostR3411_2012_256,
      sizeof(parametersGostR3411_2012_256) } // Параметры алгоритма ГОСТ Р 34.11-2012
};

/*************************************************************************
 * Шаблон для генерации закрытого ключа ГОСТ Р 34.10-2012                 *
 *************************************************************************/
/*************************************************************************
 * Для использования ключа длиной 512 бит необходимо заменить             *
 * parametersGostR3410_2012_256 на parametersGostR3410_2012_512 и         *
 * parametersGostR3411_2012_256 на parametersGostR3411_2012_512           *
 *************************************************************************/
CK_ATTRIBUTE gostPrivateKeyTemplate[] = {
    { CKA_CLASS, &privateKeyObject, sizeof(privateKeyObject) }, // Класс - закрытый ключ
    { CKA_ID, &g_keyPairIdGost2012RtEngine,
      sizeof(g_keyPairIdGost2012RtEngine) -
          1 }, // Идентификатор ключевой пары (должен совпадать у открытого и закрытого ключей)
    { CKA_KEY_TYPE, &keyTypeGostR3410, sizeof(keyTypeGostR3410) }, // Тип ключа - ГОСТ Р 34.10-2012
    { CKA_TOKEN, &attributeTrue, sizeof(attributeTrue) }, // Ключ является объектом токена
    { CKA_PRIVATE, &attributeTrue, sizeof(attributeTrue) }, // Ключ доступен только после аутентификации на токене
    { CKA_DERIVE, &attributeTrue, sizeof(attributeTrue) }, // Ключ поддерживает выработку общих ключей (VKO)
    { CKA_GOSTR3410_PARAMS, parametersGostR3410_2012_256,
      sizeof(parametersGostR3410_2012_256) }, // Параметры алгоритма ГОСТ Р 34.10-2012
    { CKA_GOSTR3411_PARAMS, parametersGostR3411_2012_256,
      sizeof(parametersGostR3411_2012_256) } // Параметры алгоритма ГОСТ Р 34.11-2012
};

/*************************************************************************
 * Шаблон для генерации открытого ключа ECDSA                             *
 *************************************************************************/
CK_ATTRIBUTE ecdsaPublicKeyTemplate[] = { { CKA_CLASS, &publicKeyObject, sizeof(publicKeyObject) },
                                          { CKA_ID, g_keyPairIdEcdsaRtEngine, sizeof(g_keyPairIdEcdsaRtEngine) - 1 },
                                          { CKA_TOKEN, &attributeTrue, sizeof(attributeTrue) },
                                          { CKA_PRIVATE, &attributeFalse, sizeof(attributeFalse) },
                                          { CKA_KEY_TYPE, &keyTypeEcdsa, sizeof(keyTypeEcdsa) },
                                          { CKA_EC_PARAMS, secp256k1Oid, sizeof(secp256k1Oid) } };

/*************************************************************************
 * Шаблон для генерации закрытого ключа ECDSA                             *
 *************************************************************************/
CK_ATTRIBUTE ecdsaPrivateKeyTemplate[] = {
    { CKA_CLASS, &privateKeyObject, sizeof(privateKeyObject) },
    { CKA_ID, g_keyPairIdEcdsaRtEngine, sizeof(g_keyPairIdEcdsaRtEngine) - 1 },
    { CKA_TOKEN, &attributeTrue, sizeof(attributeTrue) },
    { CKA_PRIVATE, &attributeTrue, sizeof(attributeTrue) },
    { CKA_KEY_TYPE, &keyTypeEcdsa, sizeof(keyTypeEcdsa) },
    { CKA_DERIVE, &attributeTrue, sizeof(attributeTrue) },
};

/************************************************************************
 * Шаблон для генерации открытого ключа RSA                               *
 * (ключевая пара для подписи и шифрования)                               *
 * Для изменения длины модуля измените значение атрибута                  *
 * CKA_MODULUS_BITS. Например, для генерации пары RSA-4096 значение       *
 * этого атрибута должно быть 4096.                                       *
 *************************************************************************/
CK_ATTRIBUTE rsaPublicKeyTemplate[] = {
    { CKA_CLASS, &publicKeyObject, sizeof(publicKeyObject) }, // Класс - открытый ключ
    { CKA_ID, &g_keyPairIdRsaRtEngine,
      sizeof(g_keyPairIdRsaRtEngine) -
          1 }, // Идентификатор ключевой пары (должен совпадать у открытого и закрытого ключей)
    { CKA_KEY_TYPE, &keyTypeRsa, sizeof(keyTypeRsa) },    // Тип ключа - RSA
    { CKA_TOKEN, &attributeTrue, sizeof(attributeTrue) }, // Ключ является объектом токена
    { CKA_ENCRYPT, &attributeTrue, sizeof(attributeTrue) }, // Ключ предназначен для зашифрования
    { CKA_PRIVATE, &attributeFalse, sizeof(attributeFalse) }, // Ключ доступен без аутентификации на токене
    { CKA_MODULUS_BITS, &rsaModulusBits, sizeof(rsaModulusBits) } // Длина модуля ключа
};

/*************************************************************************
 * Шаблон для генерации закрытого ключа RSA                               *
 * (Ключевая пара для подписи и шифрования)                               *
 *************************************************************************/
CK_ATTRIBUTE rsaPrivateKeyTemplate[] = {
    { CKA_CLASS, &privateKeyObject, sizeof(privateKeyObject) }, // Класс - закрытый ключ
    { CKA_ID, &g_keyPairIdRsaRtEngine,
      sizeof(g_keyPairIdRsaRtEngine) -
          1 }, // Идентификатор ключевой пары (должен совпадать у открытого и закрытого ключей)
    { CKA_KEY_TYPE, &keyTypeRsa, sizeof(keyTypeRsa) }, // Тип ключа - RSA
    { CKA_DECRYPT, &attributeTrue, sizeof(attributeTrue) }, // Ключ предназначен для расшифрования
    { CKA_TOKEN, &attributeTrue, sizeof(attributeTrue) }, // Ключ является объектом токена
    { CKA_PRIVATE, &attributeTrue, sizeof(attributeTrue) } // Ключ доступен только после аутентификации на токене
};

ASN1_INTEGER* generateSerialNumber() {
    ASN1_INTEGER* sn;
    BIGNUM* bigNum;
    int r;

    sn = ASN1_INTEGER_new();
    CHECK("      ASN1_INTEGER_new", sn != NULL, exit);

    bigNum = BN_new();
    CHECK("      BN_new", bigNum != NULL, free_sn);

    r = BN_rand(bigNum, 8 * 16, BN_RAND_TOP_ANY, BN_RAND_BOTTOM_ANY);
    CHECK("      BN_rand", r == 1, free_bignum);
    CHECK("      BN_to_ASN1_INTEGER", BN_to_ASN1_INTEGER(bigNum, sn) != NULL, free_bignum);
    BN_free(bigNum);
    return sn;
free_bignum:
    BN_free(bigNum);
free_sn:
    ASN1_INTEGER_free(sn);
exit:
    return NULL;
}

EVP_PKEY* make_hardware_key_pair(CK_OBJECT_HANDLE_PTR privateKeyHandlePtr, CK_OBJECT_HANDLE_PTR publicKeyHandlePtr) {
    EVP_PKEY* key = NULL;                      // Описатель ключевой пары
    rt_eng_p11_session* wrappedSession;        // Описатель обернутой сессии
    CK_UTF8CHAR userPin[] = { "12345678" };    // PIN-код пользователя Рутокен
    CK_ULONG userPinLen = sizeof(userPin) - 1; // Длина PIN-кода пользователя Рутокен
    CK_OBJECT_HANDLE privateKeyHandle;         // Описатель закрытого ключа
    CK_OBJECT_HANDLE publicKeyHandle;          // Описатель открытого ключа
    CK_ATTRIBUTE_PTR publicKeyTemplate;        // Шаблон для генерации открытого ключа
    CK_ATTRIBUTE_PTR privateKeyTemplate;       // Шаблон для генерации закрытого ключа
    CK_ULONG pubKeyTmplSize;  // Размер шаблона для генерации открытого ключа
    CK_ULONG privKeyTmplSize; // Размер шаблона для генерации закрытого ключа
    CK_MECHANISM generationMechanism; // Механизм генерации ключевой пары
    CK_RV rv;                         // Код возврата

    /************************************************************************
     * Для генерации ключевой пары ECDSA необходимо заменить                *
     * gostPublicKeyTemplate на ecdsaPublicKeyTemplate,                     *
     * gostPrivateKeyTemplate на ecdsaPrivateKeyTemplate и                  *
     * gostR3410KeyPairGenMech на ecdsaKeyPairGenMech                       *
     * ---------------------------------------------------------------------*
     * Для генерации ключевой пары RSA необходимо заменить                  *
     * gostPublicKeyTemplate на rsaPublicKeyTemplate,                       *
     * gostPrivateKeyTemplate на rsaPrivateKeyTemplate и                    *
     * gostR3410KeyPairGenMech на rsaKeyPairGenMech                         *
     ************************************************************************/
    publicKeyTemplate = gostPublicKeyTemplate;
    privateKeyTemplate = gostPrivateKeyTemplate;
    pubKeyTmplSize = sizeof(gostPublicKeyTemplate) / sizeof(gostPublicKeyTemplate[0]);
    privKeyTmplSize = sizeof(gostPrivateKeyTemplate) / sizeof(gostPrivateKeyTemplate[0]);
    generationMechanism = gostR3410KeyPairGenMech;

    /*************************************************************************
     * Инициализация библиотеки и открытие новой сессии в первом слоте        *
     *************************************************************************/
    printf("    initializeToken...\n");
    g_session = initializeToken(&g_functionList, &g_pkcsModule);
    CHECK("    initializeToken", g_session != CK_INVALID_HANDLE, exit);

    /*************************************************************************
     * Оборачивание сессии для использования ее в rtEngine                    *
     *************************************************************************/
    wrappedSession = rt_eng_p11_session_wrap(g_functionList, g_session, 0, NULL);
    CHECK("    rt_eng_p11_session_wrap", wrappedSession != NULL, finalize_token);

    /*************************************************************************
     * Аутентификация пользователя                                            *
     *************************************************************************/
    rv = g_functionList->C_Login(g_session, CKU_USER, userPin, userPinLen);
    CHECK("    C_Login", rv == CKR_OK, free_wrapped_session);

    /*************************************************************************
     * Сгенерировать ключевую пару                                            *
     *************************************************************************/
    rv = g_functionList->C_GenerateKeyPair(g_session, &generationMechanism, publicKeyTemplate, pubKeyTmplSize,
                                           privateKeyTemplate, privKeyTmplSize, &publicKeyHandle, &privateKeyHandle);
    CHECK("    C_GenerateKeyPair", rv == CKR_OK, logout);

    /*************************************************************************
     * Создание описателя ключевой пары в формате OpenSSL из описателей       *
     * открытого и закрытого ключа в формате PKCS#11                          *
     *************************************************************************/
    key = rt_eng_p11_key_pair_wrap(wrappedSession, privateKeyHandle, publicKeyHandle);
    CHECK("    rt_eng_p11_key_pair_wrap", key != NULL, logout);

    if (privateKeyHandlePtr) {
        *privateKeyHandlePtr = privateKeyHandle;
    }
    if (publicKeyHandlePtr) {
        *publicKeyHandlePtr = publicKeyHandle;
    }
    /*************************************************************************
     * Освобождение обернутой сессии                                          *
     *************************************************************************/
    rt_eng_p11_session_free(wrappedSession);
    return key;

    /*************************************************************************
     * Сброс прав доступа                                                     *
     *************************************************************************/
logout:
    rv = g_functionList->C_Logout(g_session);
    CHECK("    C_Logout", rv == CKR_OK, free_wrapped_session);

    /*************************************************************************
     * Освобождение обернутой сессии                                          *
     *************************************************************************/
free_wrapped_session:
    rt_eng_p11_session_free(wrappedSession);

    /*************************************************************************
     * Закрытие сессии и финализация библиотеки                               *
     *************************************************************************/
finalize_token:
    finalizeToken();
exit:
    return NULL;
}

int free_hardware_key_pair(EVP_PKEY* key, int* errorCode, CK_OBJECT_HANDLE privateKeyHandle,
                           CK_OBJECT_HANDLE publicKeyHandle) {
    int r;

    r = rt_eng_p11_key_pair_invalidate(key);
    CHECK_RELEASE("  rt_eng_p11_key_pair_invalidate", r == 1, *errorCode);

    if (*errorCode) {
        /*************************************************************************
         * Удаление закрытого ключа с Рутокен                                     *
         *************************************************************************/
        r = g_functionList->C_DestroyObject(g_session, privateKeyHandle);
        CHECK_RELEASE("  C_DestroyObject(private key)", r == CKR_OK, *errorCode);

        /*************************************************************************
         * Удаление открытого ключа с Рутокен                                     *
         *************************************************************************/
        r = g_functionList->C_DestroyObject(g_session, publicKeyHandle);
        CHECK_RELEASE("  C_DestroyObject(public key)", r == CKR_OK, *errorCode);
    }
    EVP_PKEY_free(key);

    /*************************************************************************
     * Сброс прав доступа                                                     *
     *************************************************************************/
    r = g_functionList->C_Logout(g_session);
    CHECK_RELEASE("  C_Logout", r == CKR_OK, *errorCode);

    printf("  finalizeToken...\n");
    r = finalizeToken();
    CHECK_RELEASE("  finalizeToken", r == 0, *errorCode);

    return r;
}

X509* create_cert(EVP_PKEY* key, EVP_PKEY* caKey, X509* caCert) {
    X509* cert;                          // Описатель сертификата
    X509_NAME* subject;                  // Описатель субъекта
    X509_EXTENSION* keyUsageExt;         // Описатель расширения 'key usage'
    X509_EXTENSION* extendedKeyUsageExt; // Описатель расширения 'extended key usage'
    X509_EXTENSION* subjectSignTool;     // Описатель расширения 'subject sign tool'
    ASN1_INTEGER* sn;                    // Описатель серийного номера сертификата
    int r;                               // Код возврата
    int errorCode = 1;                   // Флаг ошибки

    cert = X509_new();
    CHECK("    X509_new", cert != NULL, exit);

    /*************************************************************************
     * Получение имени субъекта                                               *
     *************************************************************************/
    subject = X509_get_subject_name(cert);
    CHECK("    X509_get_subject_name", subject != NULL, free_cert);

    /*************************************************************************
     * Установка значения для поля 'Country'                                  *
     *************************************************************************/
    r = X509_NAME_add_entry_by_txt(subject, "C", MBSTRING_ASC, (unsigned char*)"RU", -1, -1, 0);
    CHECK("    X509_NAME_add_entry_by_txt", r == 1, free_cert);

    /*************************************************************************
     * Установка значения для поля 'Common name'                              *
     *************************************************************************/
    r = X509_NAME_add_entry_by_txt(subject, "CN", MBSTRING_UTF8, (unsigned char*)"Иванов", -1, -1, 0);
    CHECK("    X509_NAME_add_entry_by_txt", r == 1, free_cert);

    /*************************************************************************
     * Установка значения для поля 'E-mail'                                   *
     *************************************************************************/
    r = X509_NAME_add_entry_by_txt(subject, "emailAddress", MBSTRING_ASC, (unsigned char*)"ivanov@mail.ru", -1, -1, 0);
    CHECK("    X509_NAME_add_entry_by_txt", r == 1, free_cert);

    /*************************************************************************
     * Установка значения для поля 'State'                                    *
     *************************************************************************/
    r = X509_NAME_add_entry_by_txt(subject, "ST", MBSTRING_UTF8, (unsigned char*)"Москва", -1, -1, 0);
    CHECK("    X509_NAME_add_entry_by_txt", r == 1, free_cert);

    /*************************************************************************
     * Создание расширения 'Subject sign tool'                                *
     *************************************************************************/
    subjectSignTool =
        X509V3_EXT_nconf_nid(NULL, NULL, NID_subjectSignTool, "ASN1:FORMAT:UTF8,UTF8String:СКЗИ \"Рутокен ЭЦП 2.0\"");
    CHECK("    X509V3_EXT_nconf_nid", subjectSignTool != NULL, free_cert);

    /*************************************************************************
     * Создание расширения 'Key usage'                                        *
     *************************************************************************/
    keyUsageExt = X509V3_EXT_nconf_nid(NULL, NULL, NID_key_usage,
                                       "digitalSignature,nonRepudiation,keyEncipherment,dataEncipherment");
    CHECK("    X509V3_EXT_nconf_nid", keyUsageExt != NULL, free_subjectSignTool);

    /*************************************************************************
     * Создание расширения 'Extended key usage'                               *
     *************************************************************************/
    extendedKeyUsageExt = X509V3_EXT_nconf_nid(NULL, NULL, NID_ext_key_usage, "emailProtection");
    CHECK("    X509V3_EXT_nconf_nid", extendedKeyUsageExt != NULL, free_keyUsage);

    /*************************************************************************
     * Добавление расширения 'Subject sign tool' в сертификат                 *
     *************************************************************************/
    r = X509_add_ext(cert, subjectSignTool, -1);
    CHECK("    X509_add_ext", r == 1, free_extKeyUsage);

    /*************************************************************************
     * Добавление расширения 'Key usage' в сертификат                         *
     *************************************************************************/
    r = X509_add_ext(cert, keyUsageExt, -1);
    CHECK("    X509_add_ext", r == 1, free_extKeyUsage);

    /*************************************************************************
     * Добавление расширения 'Extended key usage' в сертификат                *
     *************************************************************************/
    r = X509_add_ext(cert, extendedKeyUsageExt, -1);
    CHECK("    X509_add_ext", r == 1, free_extKeyUsage);

    /*************************************************************************
     * Установка открытого ключа                                              *
     *************************************************************************/
    r = X509_set_pubkey(cert, key);
    CHECK("    X509_set_pubkey", r == 1, free_extKeyUsage);

    /*************************************************************************
     * Установка значения поля имени УЦ, выдавшего сертификат                 *
     *************************************************************************/
    r = X509_set_issuer_name(cert, X509_get_subject_name(caCert));
    CHECK("    X509_set_issuer_name", r == 1, free_extKeyUsage);

    /*************************************************************************
     * Установка серийного номера сертификата                                 *
     *************************************************************************/
    printf("    generateSerialNumber...\n");
    sn = generateSerialNumber();
    CHECK("    generateSerialNumber", sn != NULL, free_extKeyUsage);
    r = X509_set_serialNumber(cert, sn);
    CHECK("    X509_set_serialNumber", r == 1, free_sn);

    /*************************************************************************
     * Установка периода действия сертификата                                 *
     *************************************************************************/
    CHECK("    X509_gmtime_adj", X509_gmtime_adj(X509_getm_notBefore(cert), -24 * 3600) != NULL, free_sn);
    CHECK("    X509_gmtime_adj", X509_gmtime_adj(X509_getm_notAfter(cert), 365 * 24 * 3600) != NULL, free_sn);

    /*************************************************************************
     * Подпись сертификата удостoверяющим центром                             *
     *************************************************************************/
    /*************************************************************************
     * Для использования ключа длиной 512 бит необходимо заменить             *
     * NID_id_GostR3410_2012_256 на NID_id_GostR3410_2012_512                 *
     * Для использования ECDSA ключа необходимо заменить                      *
     * NID_id_GostR3410_2012_256 на NID_ecdsa_with_SHA1                       *
     * Для использования RSA ключа необходимо заменить                        *
     * NID_id_GostR3410_2012_256 на NID_sha1WithRSAEncryption                 *
     *************************************************************************/
    r = X509_sign(cert, caKey, EVP_get_digestbynid(NID_id_GostR3410_2012_256));
    CHECK("    X509_sign", r > 1, free_sn);

    /*************************************************************************
     * Установка признака успешного завершения программы                      *
     *************************************************************************/
    errorCode = 0;

    /*************************************************************************
     * Освобождение памяти, выделенной под серийный номер сертификата         *
     *************************************************************************/
free_sn:
    ASN1_INTEGER_free(sn);

    /*************************************************************************
     * Освобождение расширения                                                *
     *************************************************************************/
free_extKeyUsage:
    X509_EXTENSION_free(extendedKeyUsageExt);

    /*************************************************************************
     * Освобождение расширения                                                *
     *************************************************************************/
free_keyUsage:
    X509_EXTENSION_free(keyUsageExt);

    /*************************************************************************
     * Освобождение расширения                                                *
     *************************************************************************/
free_subjectSignTool:
    X509_EXTENSION_free(subjectSignTool);

    if (!errorCode) {
        goto exit;
    }

    /*************************************************************************
     * Освобождение сертификата                                               *
     *************************************************************************/
free_cert:
    X509_free(cert);
    cert = NULL;
exit:
    return cert;
}

int main(void) {
    ENGINE* rtEngine; // rtengine
    EVP_PKEY* key;    // Описатель ключевой пары
    EVP_PKEY* caKey;  // Описатель закрытого ключа удостверяющего центра
    X509* caCert;     // Описатель сертификата удостверяющего центра

    X509* cert;     // Описатель сертификата
    BIO* inCertBio; // Описатель потока ввода сертификата УЦ
    BIO* inKeyBio;  // Описатель потока ввода закрытого ключа УЦ
    BIO* outBio;    // Описатель потока вывода

    CK_OBJECT_HANDLE privateKeyHandle; // Описатель закрытого ключа в формате PKCS#11
    CK_OBJECT_HANDLE publicKeyHandle;  // Описатель открытого ключа в формате PKCS#11

    int r;             // Код возврата
    int errorCode = 1; // Флаг ошибки

    printf("Sample has started.\n\n");
    /*************************************************************************
     * Инициализация OPENSSL_crypto                                           *
     *************************************************************************/
    r = OPENSSL_init_crypto(OPENSSL_INIT_NO_LOAD_CONFIG | OPENSSL_INIT_NO_ATEXIT, NULL);
    CHECK("  OPENSSL_init_crypto", r, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_load_engine();
    CHECK("  rt_eng_load_engine", r == 1, exit);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    r = ENGINE_init(rtEngine);
    CHECK("  ENGINE_init", r == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK("  ENGINE_set_default", r == 1, finalize_engine);

    /*************************************************************************
     * Открытие потока ввода из файла                                         *
     *************************************************************************/
    inCertBio = BIO_new_file("test_trusted_ca.cer", "r");
    CHECK("  BIO_new_file", inCertBio != NULL, unregister_engine);

    /*************************************************************************
     * Чтение сертификата удостверяющего центра из файла                      *
     *************************************************************************/
    caCert = PEM_read_bio_X509(inCertBio, NULL, NULL, NULL);
    CHECK("  PEM_read_bio_X509", caCert != NULL, free_in_cert_bio);

    /*************************************************************************
     * Открытие потока ввода из файла                                     *
     *************************************************************************/
    inKeyBio = BIO_new_file("test_ca_key.pem", "r");
    CHECK("  BIO_new_file", inKeyBio != NULL, free_ca_cert);

    /*************************************************************************
     * Чтение сертификата удостверяющего центра из файла                      *
     *************************************************************************/
    caKey = PEM_read_bio_PrivateKey(inKeyBio, NULL, NULL, NULL);
    CHECK("  PEM_read_bio_PrivateKey", caCert != NULL, free_in_key_bio);

    /*************************************************************************
     * Генерация ключевой пары на Рутокен                                     *
     *************************************************************************/
    printf("  make_hardware_key_pair...\n");
    key = make_hardware_key_pair(&privateKeyHandle, &publicKeyHandle);
    CHECK("  make_hardware_key_pair", key != NULL, free_ca_key);

    /*************************************************************************
     * Создание сертификата                                                   *
     *************************************************************************/
    printf("  create_cert...\n");
    cert = create_cert(key, caKey, caCert);
    CHECK("  create_cert", cert != NULL, free_key);

    /*************************************************************************
     * Открытие потока вывода в файл                                          *
     *************************************************************************/
    outBio = BIO_new_file(TEST_CERT_NAME, "w");
    CHECK("  BIO_new_file", outBio != NULL, free_cert);

    /*************************************************************************
     * Запись сертификата                                                     *
     *************************************************************************/
    r = PEM_write_bio_X509(outBio, cert);
    CHECK("  PEM_write_bio_X509", r == 1, free_out_bio);

    /*************************************************************************
     * Установка признака успешного завершения программы                      *
     *************************************************************************/
    errorCode = 0;

    /*************************************************************************
     * Закрытие потока вывода                                                 *
     *************************************************************************/
free_out_bio:
    BIO_free_all(outBio);

    /*************************************************************************
     * Освобождение сертификата                                               *
     *************************************************************************/
free_cert:
    X509_free(cert);

    /*************************************************************************
     * Освобождение описателя ключевой пары, удаление ключевой пары с         *
     * Рутокен в случае ошибки                                                *
     *************************************************************************/
free_key:
    /*************************************************************************
     * Инвалидация описателя ключевой пары                                    *
     *************************************************************************/
    free_hardware_key_pair(key, &errorCode, privateKeyHandle, publicKeyHandle);

    /*************************************************************************
     * Освобождение закрытого ключа УЦ                                        *
     *************************************************************************/
free_ca_key:
    EVP_PKEY_free(caKey);

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
free_in_key_bio:
    BIO_free_all(inKeyBio);

    /*************************************************************************
     * Освобождение сертификата УЦ                                            *
     *************************************************************************/
free_ca_cert:
    X509_free(caCert);

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
free_in_cert_bio:
    BIO_free_all(inCertBio);

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
unregister_engine:
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    r = ENGINE_finish(rtEngine);
    CHECK_RELEASE("  ENGINE_finish", r == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_unload_engine();
    CHECK_RELEASE("  rt_eng_unload_engine", r == 1, errorCode);
exit:
    OPENSSL_cleanup();
    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}
