/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

#include <assert.h>

#include <Common.h>

int main(void) {
    EVP_PKEY* key;     // Описатель открытого ключа
    ENGINE* rtEngine;  // rtengine
    EVP_PKEY_CTX* ctx; // Контекст подписи
    FILE* inputFile;   // Описатель потока ввода
    uint8_t hash[] = {
        0x00, 0x90, 0x31, 0xCF, 0x95, 0x7A, 0x59, 0xE5, 0xD2, 0xBF, 0x2C, 0xDB, 0xB5, 0x83, 0x4D, 0x03,
        0x17, 0x5D, 0x25, 0x2A, 0xFD, 0x72, 0x1E, 0x01, 0x02, 0x60, 0x88, 0x92, 0x9A, 0x9B, 0x2A, 0xA9
    }; // Хеш для подписи

    size_t siglen;            // Длина подписи
    unsigned char* signature; // Буфер с подписью

    size_t numberOfBytes; // Число прочитанных байт
    int r;                // Код возврата
    int errorCode = 1;    // Флаг ошибки

    printf("Sample has started.\n\n");
    /*************************************************************************
     * Инициализация OPENSSL_crypto                                           *
     *************************************************************************/
    r = OPENSSL_init_crypto(OPENSSL_INIT_NO_LOAD_CONFIG | OPENSSL_INIT_NO_ATEXIT, NULL);
    CHECK("  OPENSSL_init_crypto", r, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_load_engine();
    CHECK("  rt_eng_load_engine", r == 1, exit);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    r = ENGINE_init(rtEngine);
    CHECK("  ENGINE_init", r == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK("  ENGINE_set_default", r == 1, finalize_engine);

    /*************************************************************************
     * Получение открытого ключа                                              *
     *************************************************************************/
    printf("  get_public_key...\n");
    key = get_public_key();
    CHECK("  get_public_key", key != NULL, unregister_engine);

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    inputFile = fopen("signature", "rb");
    CHECK("  fopen", inputFile != NULL, free_key);

    /*************************************************************************
     * Определение размера файла подписи                                      *
     *************************************************************************/
    r = fseek(inputFile, 0, SEEK_END);
    CHECK("  fseek", r == 0, close_file);
    siglen = ftell(inputFile);
    CHECK("  ftell", siglen > 0, close_file);
    r = fseek(inputFile, 0, SEEK_SET);
    CHECK("  fseek", r == 0, close_file);

    /*************************************************************************
     * Выделение памяти для подписи                                           *
     *************************************************************************/
    signature = malloc(siglen);
    CHECK("  malloc", signature != NULL, close_file);

    /*************************************************************************
     * Чтение подписи из файла                                                *
     *************************************************************************/
    numberOfBytes = fread(signature, 1, siglen, inputFile);
    CHECK("  fread", numberOfBytes == siglen, free_signature);

    /*************************************************************************
     * Создание контекста проверки подписи                                    *
     *************************************************************************/
    ctx = EVP_PKEY_CTX_new(key, rtEngine);
    CHECK("  EVP_PKEY_CTX_new", ctx != NULL, free_signature);

    /*************************************************************************
     * Инициализация контекста проверки подписи                               *
     *************************************************************************/
    r = EVP_PKEY_verify_init(ctx);
    CHECK("  EVP_PKEY_verify_init", r == 1, free_context);

    /*************************************************************************
     * Проверка подписи                                                       *
     *************************************************************************/
    r = EVP_PKEY_verify(ctx, signature, siglen, hash, sizeof(hash));
    CHECK("  EVP_PKEY_verify", r >= 0, free_context);

    if (r == 1) {
        printf("  Signature is correct!\n");
        errorCode = 0;
    } else {
        printf("  Signature is wrong!\n");
        errorCode = 2;
    }

free_context:

    /*************************************************************************
     * Освобождение контекста                                                 *
     *************************************************************************/
    EVP_PKEY_CTX_free(ctx);
free_signature:

    /*************************************************************************
     * Освобождение буфера с подписью                                         *
     *************************************************************************/
    free(signature);
close_file:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    r = fclose(inputFile);
    CHECK_RELEASE("  fclose", r == 0, errorCode);
free_key:

    /*************************************************************************
     * Освобождение описателя открытого ключа                                 *
     *************************************************************************/
    printf("  free_public_key...\n");
    free_public_key(key);
    CHECK_RELEASE("  free_public_key", r == 0, errorCode);
unregister_engine:

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    r = ENGINE_finish(rtEngine);
    CHECK_RELEASE("  ENGINE_finish", r == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_unload_engine();
    CHECK_RELEASE("  rt_eng_unload_engine", r == 1, errorCode);
exit:
    OPENSSL_cleanup();
    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}
