/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/*************************************************************************
 * Этот пример верифицирует подписанный файл signed_data.xml, созданный в *
 * результате работы примеров XMLDSigEnveloped или XMLDSigDetached        *
 * Версии Software и Hardware этого примера идентичны, разделение         *
 * сделано для удобства использования                                     *
 *************************************************************************/

#include <assert.h>

#include <commonXmldsig.h>

int main() {
    ENGINE* rtEngine;             // rtengine
    xmlDocPtr doc;                // Описатель XML документа
    xmlNodePtr node;              // Описатель XML ноды для подписи
    xmlSecKeysMngrPtr keyManager; // Менеджер ключей
    xmlSecDSigCtxPtr dsigCtx;     // Описатель контекста для верификации
    int rv;                       // Код возврата
    int errorCode = 1;            // Флаг ошибки

    printf("Sample has started.\n\n");

    /*************************************************************************
     * Инициализация xmlsec                                                   *
     *************************************************************************/
    rv = xmlsec_init();
    CHECK("  xmlsec_init", rv == 0, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    rv = rt_eng_load_engine();
    CHECK(" rt_eng_load_engine", rv == 1, finalize_xmlsec);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    rv = ENGINE_init(rtEngine);
    CHECK(" ENGINE_init", rv == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    rv = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK(" ENGINE_set_default", rv == 1, finalize_engine);

    /*************************************************************************
     * Считывание файла в стукртуру XML документа                             *
     *************************************************************************/
    doc = xmlParseFile("signed_data.xml");
    CHECK("  xmlParseFile", doc != NULL, unregister_engine);

    /*************************************************************************
     * Поиск XML ноды с подписью                                              *
     *************************************************************************/
    node = xmlSecFindNode(xmlDocGetRootElement(doc), xmlSecNodeSignature, xmlSecDSigNs);
    CHECK("  xmlSecFindNode", node != NULL, free_xmlDoc);

    /*************************************************************************
     * Создание менеджера ключей                                              *
     *************************************************************************/
    keyManager = xmlSecKeysMngrCreate();
    CHECK("  xmlSecKeysMngrCreate", keyManager != NULL, free_xmlDoc);

    /*************************************************************************
     * Инициализация менеджера ключей                                         *
     *************************************************************************/
    rv = xmlSecCryptoAppDefaultKeysMngrInit(keyManager);
    CHECK("  xmlSecCryptoAppDefaultKeysMngrInit", rv >= 0, free_keyManager);

    /*************************************************************************
     * Добавление root сертификата в хранилище ключей                         *
     *************************************************************************/
    rv = xmlSecCryptoAppKeysMngrCertLoad(keyManager, "test_trusted_ca.cer", xmlSecKeyDataFormatPem,
                                         xmlSecKeyDataTypeTrusted);
    CHECK("  xmlSecCryptoAppKeysMngrCertLoad", rv >= 0, free_keyManager);

    /*************************************************************************
     * Создание контекста для верификации подписи                             *
     *************************************************************************/
    dsigCtx = xmlSecDSigCtxCreate(keyManager);
    CHECK("  xmlSecDSigCtxCreate", dsigCtx != NULL, free_xmlDoc);

    /*************************************************************************
     * Верификация                                                            *
     *************************************************************************/
    errorCode = 0;
    xmlSecDSigCtxVerify(dsigCtx, node);
    CHECK_RELEASE("  xmlSecDSigCtxVerify", dsigCtx->status == xmlSecDSigStatusSucceeded, errorCode);

    /*************************************************************************
     * Удаление контекста                                                     *
     *************************************************************************/
    xmlSecDSigCtxDestroy(dsigCtx);
free_keyManager:

    /*************************************************************************
     * Удаление менеджера ключей                                              *
     *************************************************************************/
    xmlSecKeysMngrDestroy(keyManager);
free_xmlDoc:

    /*************************************************************************
     * Удаление xmlDoc                                                        *
     *************************************************************************/
    xmlFreeDoc(doc);
unregister_engine:

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    rv = ENGINE_finish(rtEngine);
    CHECK_RELEASE(" ENGINE_finish", rv == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    rv = rt_eng_unload_engine();
    CHECK_RELEASE(" rt_eng_unload_engine", rv == 1, errorCode);
finalize_xmlsec:

    /*************************************************************************
     * Финализация xmlsec                                                     *
     *************************************************************************/
    xmlsec_final();
exit:

    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}
