/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/*************************************************************************
 * Этот пример подписывает файл test_data.xml (любой xml файл с указанным *
 * названием) в режиме Enveloped и создает в out подписанный документ     *
 * в формате XAdES-BES с названием signed_data.xml. Для хардварной версии *
 * примера используется ключевая пара и файл hardware_test_cert.cer,      *
 * созданные примером TokenPreparation                                    *
 *************************************************************************/

#include <commonXmldsig.h>

/*************************************************************************
 * Поддерживаемые виды каноникализации описаны в commonXmldsig.h          *
 * Для использования другого вида каноникализации следует заменить        *
 * XMLSEC_CANON_EXCL_C14N на желаемый                                     *
 *************************************************************************/
#define XMLSEC_CANON XMLSEC_CANON_EXCL_C14N // Определение каноникализации
#define XML_DATE_TIME_LEN 19

xmlNodePtr xmlsec_create_xades_cert_digest_node(xmlNsPtr ns, xmlNodePtr certNode) {
    xmlNodePtr certDigestNode;   // Нода для хэш-суммы сертификата
    xmlNodePtr digestMethodNode; // Нода для алгоритма получения хэш-суммы
    xmlNodePtr digestValueNode;  // Нода для значение хэш-суммы

    xmlAttrPtr attrProp; // Указатель для проверки валидности операций присвоения свойств

    BIO* inBio;                 // Буфер для данных сертификата
    X509* cert;                 // Сертификат, полученный из буфера
    unsigned char* digestValue; // Значение хэш-суммы

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Создание ноды для информации о хэш-сумме сертификата                   *
     *************************************************************************/
    certDigestNode = xmlNewChild(certNode, NULL, (const xmlChar*)"CertDigest", NULL);
    CHECK("     xmlNewChild", certDigestNode != NULL, exit);

    /*************************************************************************
     * Создание ноды для алгоритма получения хэш-суммы                        *
     *************************************************************************/
    digestMethodNode = xmlNewChild(certDigestNode, ns, (const xmlChar*)"DigestMethod", NULL);
    CHECK("     xmlNewChild", digestMethodNode != NULL, exit);

    /*************************************************************************
     * Высталение алгоритма получения хэш-суммы.                              *
     * Если требуется ГОСТ Р34.11-2012-512, то замените gostr34112012-256 на  *
     * gostr34112012-512                                                      *
     *************************************************************************/
    attrProp = xmlSetProp(digestMethodNode, (const xmlChar*)"Algorithm",
                          (const xmlChar*)"urn:ietf:params:xml:ns:cpxmlsec:algorithms:gostr34112012-256");
    CHECK("     xmlSetProp", attrProp != NULL, exit);

    /*************************************************************************
     * Чтение данных из файла и занесение их в буфер                          *
     *************************************************************************/
    inBio = BIO_new_file(TEST_CERT_NAME, "r");
    CHECK("     BIO_new_file", inBio != NULL, exit);

    /*************************************************************************
     * Чтение буфера и получение сертификата из него                          *
     *************************************************************************/
    cert = PEM_read_bio_X509(inBio, NULL, 0, NULL);
    CHECK("     PEM_read_bio_X509", cert != NULL, free_bio);

    /*************************************************************************
     * Получение хэш-суммы сертификата.                                       *
     * Если требуется ГОСТ Р34.11-2012-512, то замените md_gost12_256 на      *
     * md_gost12_512                                                          *
     *************************************************************************/
    digestValue = xmlsec_digest_cert(cert, "md_gost12_256");
    CHECK("     xmlsec_digest_cert", digestValue != NULL, free_x509);

    /*************************************************************************
     * Создание ноды под значение полученной хэш-суммы и присвоение           *
     * результата вычисления хэш-суммы                                        *
     *************************************************************************/
    digestValueNode = xmlNewChild(certDigestNode, ns, (const xmlChar*)"DigestValue", (const xmlChar*)digestValue);
    CHECK("     xmlNewChild", digestValueNode != NULL, free_digest);

    errorCode = 0;

free_digest:
    OPENSSL_free(digestValue);
free_x509:
    X509_free(cert);
free_bio:
    BIO_free_all(inBio);
exit:
    return errorCode ? NULL : certDigestNode;
}

xmlNodePtr xmlsec_create_xades_issuer_serial_v2_node(xmlNodePtr certNode) {
    xmlNodePtr issuerSerialV2Node; // Нода для общих данных сертификата

    BIO* inBio;      // Буфер для данных сертификата
    X509* cert;      // Сертификат, полученный из буфера
    char* serialStr; // Строка с серийным номером

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Создание ноды для общих данных сертификата                             *
     *************************************************************************/
    issuerSerialV2Node = xmlNewChild(certNode, NULL, (const xmlChar*)"IssuerSerialV2", NULL);
    CHECK("      xmlNewChild", issuerSerialV2Node != NULL, exit);

    /*************************************************************************
     * Чтение данных из файла и занесение их в буфер                          *
     *************************************************************************/
    inBio = BIO_new_file(TEST_CERT_NAME, "r");
    CHECK("      BIO_new_file", inBio != NULL, exit);

    /*************************************************************************
     * Чтение буфера и получение сертификата из него                          *
     *************************************************************************/
    cert = PEM_read_bio_X509(inBio, NULL, 0, NULL);
    CHECK("      PEM_read_bio_X509", cert != NULL, free_bio);

    /*************************************************************************
     * Получение серийного номера из сертификата                              *
     *************************************************************************/
    serialStr = xmlsec_get_encoded_issuer_serial_v2(cert);
    CHECK("      xmlsec_get_encoded_issuer_serial_v2", serialStr != NULL, free_x509);

    /*************************************************************************
     * Установка значения имени эмитента                                      *
     *************************************************************************/
    xmlNodeSetContent(issuerSerialV2Node, (const xmlChar*)serialStr);

    errorCode = 0;

    OPENSSL_free(serialStr);
free_x509:
    X509_free(cert);
free_bio:
    BIO_free_all(inBio);
exit:
    return errorCode ? NULL : issuerSerialV2Node;
}

xmlNodePtr xmlsec_create_xades_cert_node(xmlDocPtr doc, xmlNodePtr signingCertNode) {
    xmlNodePtr certNode; // Нода для сертификата
    xmlNodePtr rvNode; // Указатель для проверки валидности операций с нодами
    xmlNsPtr dsNs;     // Пространство имен

    /*************************************************************************
     * Получение пространства имен из корня документа                         *
     *************************************************************************/
    dsNs = xmlGetNsList(doc, xmlDocGetRootElement(doc))[0];
    CHECK("    xmlSearchNs", dsNs != NULL, exit);

    /*************************************************************************
     * Создание ноды под сертификат                                           *
     *************************************************************************/
    certNode = xmlNewChild(signingCertNode, NULL, (const xmlChar*)"Cert", NULL);
    CHECK("    xmlNewChild", certNode != NULL, exit);

    /*************************************************************************
     * Создание ноды с хэш-суммой сертификата                                 *
     *************************************************************************/
    rvNode = xmlsec_create_xades_cert_digest_node(dsNs, certNode);
    CHECK("    xmlsec_create_xades_cert_digest_node", rvNode != NULL, exit);

    /*************************************************************************
     * Создание ноды с именем эмитента и серийного номера                     *
     *************************************************************************/
    rvNode = xmlsec_create_xades_issuer_serial_v2_node(certNode);
    CHECK("    xmlsec_create_xades_issuer_serial_v2_node", rvNode != NULL, exit);

    return certNode;
exit:
    return NULL;
}

xmlNodePtr xmlsec_create_xades_signing_time_node(xmlNodePtr signedSigPropNode) {
    xmlNodePtr signingTimeNode;             // Нода для времени подписи
    char timeString[XML_DATE_TIME_LEN + 1]; // Строка со временем

    /*************************************************************************
     * Получение текущего времени и копирование его в строку со временем      *
     *************************************************************************/
    const time_t currentTime = time(NULL);
    strftime(timeString, sizeof(timeString), "%FT%T", localtime(&currentTime));

    /*************************************************************************
     * Создание ноды со временем и присвоение ей значения времени             *
     *************************************************************************/
    signingTimeNode = xmlNewChild(signedSigPropNode, NULL, (const xmlChar*)"SigningTime", (const xmlChar*)timeString);
    CHECK("    xmlNewChild", signingTimeNode != NULL, exit);

    return signingTimeNode;
exit:
    return NULL;
}

xmlNodePtr xmlsec_create_xades_bes_signature_node(xmlDocPtr doc, xmlSecTransformId c14nMethodId,
                                                  xmlSecTransformId signMethodId) {
    xmlNodePtr signatureNode; // Нода для подписи
    xmlNodePtr refNode;       // Нода для ссылки на документ
    xmlNodePtr refXadesNode;  // Нода для ссылки на SignedProperties
    xmlNodePtr rvNode; // Указатель для проверки валидности операций с нодами
    xmlNodePtr objNode;   // Нода для объекта подписи
    xmlNodePtr qPropNode; // Нода для квалифицирующих свойств объекта подписи
    xmlNodePtr signedPropNode;    // Нода для подписыванных свойств
    xmlNodePtr signedSigPropNode; // Нода для подписанных свойств подписи
    xmlNodePtr signingTimeNode;   // Нода для временной метки времени подписи
    xmlNodePtr signingCertNode;   // Нода для сертификата подписи
    xmlNodePtr certNode;          // Нода для сертификата

    xmlAttrPtr attrProp; // Указатель для проверки валидности операций присвоения свойств

    xmlNsPtr xadesNs;    // Нода для пространства имен
    xmlNsPtr xades141Ns; // Нода для простанства имен версии 1.4.1

    /*************************************************************************
     * Создание пустой ноды для подписи                                       *
     *************************************************************************/
    signatureNode = xmlsec_create_signature_node(doc, c14nMethodId, signMethodId);
    CHECK("   xmlsec_create_signature_node", signatureNode != NULL, exit);

    attrProp = xmlSetProp(signatureNode, (const xmlChar*)"Id", (const xmlChar*)"xmldsig-id");
    CHECK("   xmlSetProp", attrProp != NULL, exit);

    /*************************************************************************
     * Создание <dsig:Reference/>.                                            *
     * Для ключа длинной 512 бит следует использовать                         *
     * xmlSecOpenSSLTransformGostR3411_2012_512Id и самостоятельно            *
     * сгенерировать на токене ключ длинной 512 бит с ID, указанным в         *
     * g_keyPairIdGost2012RtEngine в Common.h.                                *
     * Также необходимо изменить функцию xmlsec_create_xades_cert_digest_node *
     * в commonXmldsig.h как указано в комментариях в её определении          *
     *************************************************************************/
    refNode =
        xmlSecTmplSignatureAddReference(signatureNode, xmlSecOpenSSLTransformGostR3411_2012_256Id, NULL, NULL, NULL);
    CHECK("   xmlSecTmplSignatureAddReference", refNode != NULL, exit);

    refXadesNode = xmlSecTmplSignatureAddReference(signatureNode, xmlSecOpenSSLTransformGostR3411_2012_256Id, NULL,
                                                   (const xmlChar*)"#xmldsig-id-signedprops",
                                                   (const xmlChar*)"http://uri.etsi.org/01903#SignedProperties");
    CHECK("   xmlSecTmplSignatureAddReference", refXadesNode != NULL, exit);

    /*************************************************************************
     * Создание <dsig:Transform/>                                             *
     *************************************************************************/
    rvNode = xmlSecTmplReferenceAddTransform(refNode, xmlSecTransformEnvelopedId);
    CHECK("   xmlSecTmplReferenceAddTransform", rvNode != NULL, exit);

    /*************************************************************************
     * Создание <dsig:Object/>                                                *
     *************************************************************************/
    objNode = xmlSecTmplSignatureAddObject(signatureNode, NULL, NULL, NULL);
    CHECK("   xmlSecTmplSignatureAddObject", objNode != NULL, exit);

    /*************************************************************************
     * Создание <xades:QualifyingProperties>                                 *
     *************************************************************************/
    qPropNode = xmlNewChild(objNode, NULL, (const xmlChar*)"QualifyingProperties", NULL);
    CHECK("   xmlNewChild", qPropNode != NULL, exit);

    attrProp = xmlSetProp(qPropNode, (const xmlChar*)"Target", (const xmlChar*)"#xmldsig-id");
    CHECK("   xmlSetProp", attrProp != NULL, exit);

    /*************************************************************************
     * Задание простанства имен. В конце блока явно вызывается повторное      *
     * выставление простанства имен, т.к. каждый новый вызов xmlNewNs()       *
     * затирает выставленные ранее простанства имен                           *
     *************************************************************************/
    xadesNs = xmlNewNs(qPropNode, (const xmlChar*)"http://uri.etsi.org/01903/v1.3.2#", (const xmlChar*)"xades");
    CHECK("   xmlNewNs", xadesNs != NULL, exit);

    xades141Ns = xmlNewNs(qPropNode, (const xmlChar*)"http://uri.etsi.org/01903/v1.4.1#", (const xmlChar*)"xades141");
    CHECK("   xmlNewNs", xades141Ns != NULL, exit);

    xmlSetNs(qPropNode, xadesNs);

    /*************************************************************************
     * Создание <xades:SignedProperties>                                      *
     *************************************************************************/
    signedPropNode = xmlNewChild(qPropNode, NULL, (const xmlChar*)"SignedProperties", NULL);
    CHECK("   xmlNewChild", signedPropNode != NULL, exit);

    attrProp = xmlSetProp(signedPropNode, (const xmlChar*)"Id", (const xmlChar*)"xmldsig-id-signedprops");
    CHECK("   xmlSetProp", attrProp != NULL, exit);

    /*************************************************************************
     * Создание <xades:SignedSignatureProperties>                             *
     *************************************************************************/
    signedSigPropNode = xmlNewChild(signedPropNode, NULL, (const xmlChar*)"SignedSignatureProperties", NULL);
    CHECK("   xmlNewChild", signedSigPropNode != NULL, exit);

    /*************************************************************************
     * Создание <xades:SigningTime>                                           *
     *************************************************************************/
    signingTimeNode = xmlsec_create_xades_signing_time_node(signedSigPropNode);
    CHECK("   xmlsec_create_xades_signing_time_node", signingTimeNode != NULL, exit);

    /*************************************************************************
     * Создание <xades:SigningCertificate>                                    *
     *************************************************************************/
    signingCertNode = xmlNewChild(signedSigPropNode, NULL, (const xmlChar*)"SigningCertificate", NULL);
    CHECK("   xmlNewChild", signingCertNode != NULL, exit);

    /*************************************************************************
     * Создание <xades:Cert>                                                  *
     *************************************************************************/
    certNode = xmlsec_create_xades_cert_node(doc, signingCertNode);
    CHECK("   xmlsec_create_xades_cert_node", certNode != NULL, exit);

    return signatureNode;

exit:
    return NULL;
}

int main() {
    ENGINE* rtEngine;       // rtengine
    FILE* outFile;          // Описатель потока вывода
    xmlSecKeyPtr keyPair;   // Описатель ключа
    xmlDocPtr doc;          // Описатель XML документа
    xmlNodePtr node;        // Описатель XML ноды для подписи
    unsigned char* outBuff; // Буфер для хранения подписанного файла
    int outSize;            // Размер подписанного файла
    int rv;                 // Код возврата
    int errorCode = 1;      // Флаг ошибки

    printf("Sample has started.\n\n");

    /*************************************************************************
     * Инициализация xmlsec                                                   *
     *************************************************************************/
    rv = xmlsec_init();
    CHECK("  xmlsec_init", rv == 0, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    rv = rt_eng_load_engine();
    CHECK(" rt_eng_load_engine", rv == 1, finalize_xmlsec);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    rv = ENGINE_init(rtEngine);
    CHECK(" ENGINE_init", rv == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    rv = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK(" ENGINE_set_default", rv == 1, finalize_engine);

    /*************************************************************************
     * Считывание файла в структуру XML документа                             *
     * Если не используется пролог в XML документе, то можно явно указать     *
     * необходимую кодировку во втором параметре функции xmlReadFile,         *
     * например: doc = xmlReadFile("test_data.xml", "UTF-8", 0);              *
     *************************************************************************/
    doc = xmlReadFile("test_data.xml", NULL, 0);
    CHECK("  xmlReadFile", doc != NULL, unregister_engine);

    /*************************************************************************
     * Получение ключевой пары                                                *
     *************************************************************************/
    keyPair = xmlsec_get_key_pair();
    CHECK("  xmlsec_get_key_pair", keyPair != NULL, free_xmlDoc);

    /*************************************************************************
     * Добавление сетрификата к ключу                                         *
     *************************************************************************/
    rv = xmlSecCryptoAppKeyCertLoad(keyPair, TEST_CERT_NAME, xmlSecKeyDataFormatCertPem);
    if (rv != 0) {
        xmlSecKeyDestroy(keyPair);
    }
    CHECK("  xmlSecCryptoAppKeyCertLoadMemory", rv == 0, free_xmlDoc);

    /*************************************************************************
     * Создание в XML документе ноды для подписи                              *
     * Для ключа длинной 512 бит следует использовать                         *
     * xmlSecOpenSSLTransformGostR3410_2012GostR3411_2012_512Id и             *
     * самостоятельно сгенерировать на токене ключ длинной 512 бит с ID,      *
     * указанным в g_keyPairIdGost2012RtEngine в Common.h                     *
     *************************************************************************/
    printf("  xmlsec_create_xades_bes_signature_node...\n");
    node = xmlsec_create_xades_bes_signature_node(doc, XMLSEC_CANON,
                                                  xmlSecOpenSSLTransformGostR3410_2012GostR3411_2012_256Id);
    if (node == NULL) {
        xmlSecKeyDestroy(keyPair);
    }
    CHECK("  xmlsec_create_xades_bes_signature_node", node != NULL, free_xmlDoc);

    /*************************************************************************
     * Подпись                                                                *
     *************************************************************************/
    rv = xmlsec_sign(doc, keyPair);
    if (rv != 0) {
        xmlSecKeyDestroy(keyPair);
    }
    CHECK("  xmlsec_sign", rv == 0, free_xmlDoc);

    /*************************************************************************
     * Открытие поточного вывода                                              *
     *************************************************************************/
    outFile = fopen("signed_data.xml", "wb");
    CHECK("  fopen", outFile != NULL, free_xmlDoc);

    /*************************************************************************
     * Помещение подписанных данных в буфер для вывода                        *
     *************************************************************************/
    xmlDocDumpMemory(doc, &outBuff, &outSize);
    CHECK("  xmlDocDumpMemory", outBuff != NULL && outSize > 0, close_outFile);

    /*************************************************************************
     * Запись подписанных данных в поток вывода                               *
     *************************************************************************/
    rv = fprintf(outFile, "%s", outBuff);
    CHECK("  fprintf", rv == outSize, free_outBuff);

    errorCode = 0;
free_outBuff:

    /*************************************************************************
     * Освобождение памяти, выделенной под буфер для записи                   *
     *************************************************************************/
    xmlFree(outBuff);
close_outFile:

    /*************************************************************************
     * Закрытие поточного вывода                                              *
     *************************************************************************/
    fclose(outFile);

/*************************************************************************
 * Деинициализация библиотеки pkcs11ecp                                   *
 *************************************************************************/
#ifdef HARDWARE_KEYS
    logout(NULL_PTR);
    rv = finalizeToken();
    CHECK_RELEASE("    finalizeToken", rv == 0, errorCode);
#endif
free_xmlDoc:

    /*************************************************************************
     * Удаление xmlDoc                                                        *
     *************************************************************************/
    xmlFreeDoc(doc);
unregister_engine:

    /*************************************************************************
     * Выгрузка rtengine из OpenSSL                                           *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    rv = ENGINE_finish(rtEngine);
    CHECK_RELEASE(" ENGINE_finish", rv == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    rv = rt_eng_unload_engine();
    CHECK_RELEASE(" rt_eng_unload_engine", rv == 1, errorCode);
finalize_xmlsec:

    /*************************************************************************
     * Деинициализация xmlsec                                                 *
     *************************************************************************/
    xmlsec_final();
exit:

    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}
