/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
 *------------------------------------------------------------------------*
 * Пример использования функций расширения компании "Актив"               *
 * стандарта PKCS#11:                                                     *
 *  - установление соединения с Rutoken в первом доступном слоте;         *
 *  - запрос на принудительную смену ПИН-кода;                            *
 *  - демонстрация недоступности функций, требующих аутентификации        *
 *  - смена ПИН-кода;                                                     *
 *  - демонстрация восстановления доступности функций, требующих          *
 *    аутентификации.                                                     *
 *------------------------------------------------------------------------*
 * Данный пример является самодостаточным.                                *
 *************************************************************************/
#include <Common.h>

/*************************************************************************
 * Шаблон для генерации открытого ключа ГОСТ Р 34.10-2012(256)            *
 *************************************************************************/
CK_ATTRIBUTE publicKeyTemplate[] = {
    { CKA_CLASS, &publicKeyObject, sizeof(publicKeyObject) }, // Класс - открытый ключ
    { CKA_LABEL, &publicKeyLabelGost2012_256_1, sizeof(publicKeyLabelGost2012_256_1) - 1 }, // Метка ключа
    { CKA_ID, &keyPairIdGost2012_256_1,
      sizeof(keyPairIdGost2012_256_1) -
          1 }, // Идентификатор ключевой пары #1 (должен совпадать у открытого и закрытого ключей)
    { CKA_KEY_TYPE, &keyTypeGostR3410_2012_256,
      sizeof(keyTypeGostR3410_2012_256) },                // Тип ключа - ГОСТ Р 34.10-2012(256)
    { CKA_TOKEN, &attributeTrue, sizeof(attributeTrue) }, // Ключ является объектом токена
    { CKA_PRIVATE, &attributeFalse, sizeof(attributeFalse) }, // Ключ доступен без аутентификации на токене
    { CKA_GOSTR3410_PARAMS, parametersGostR3410_2012_256,
      sizeof(parametersGostR3410_2012_256) }, // Параметры алгоритма ГОСТ Р 34.10-2012(256)
    { CKA_GOSTR3411_PARAMS, parametersGostR3411_2012_256,
      sizeof(parametersGostR3411_2012_256) } // Параметры алгоритма ГОСТ Р 34.11-2012(256)
};

/*************************************************************************
 * Шаблон для генерации закрытого ключа ГОСТ Р 34.10-2012(256)            *
 *************************************************************************/
CK_ATTRIBUTE privateKeyTemplate[] = {
    { CKA_CLASS, &privateKeyObject, sizeof(privateKeyObject) }, // Класс - закрытый ключ
    { CKA_LABEL, &privateKeyLabelGost2012_256_1, sizeof(privateKeyLabelGost2012_256_1) - 1 }, // Метка ключа
    { CKA_ID, &keyPairIdGost2012_256_1,
      sizeof(keyPairIdGost2012_256_1) -
          1 }, // Идентификатор ключевой пары #1 (должен совпадать у открытого и закрытого ключей)
    { CKA_KEY_TYPE, &keyTypeGostR3410_2012_256,
      sizeof(keyTypeGostR3410_2012_256) },                // Тип ключа - ГОСТ Р 34.10-2012(256)
    { CKA_TOKEN, &attributeTrue, sizeof(attributeTrue) }, // Ключ является объектом токена
    { CKA_PRIVATE, &attributeTrue, sizeof(attributeTrue) }, // Ключ доступен только после аутентификации на токене
    { CKA_DERIVE, &attributeTrue, sizeof(attributeTrue) }, // Ключ поддерживает выработку ключей обмена (VKO)
    { CKA_GOSTR3410_PARAMS, parametersGostR3410_2012_256,
      sizeof(parametersGostR3410_2012_256) }, // Параметры алгоритма ГОСТ Р 34.10-2012(256)
    { CKA_GOSTR3411_PARAMS, parametersGostR3411_2012_256,
      sizeof(parametersGostR3411_2012_256) } // Параметры алгоритма ГОСТ Р 34.11-2012(256)
};

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_FUNCTION_LIST_EXTENDED_PTR functionListEx; // Указатель на список функций расширения PKCS#11, хранящийся в
                                                  // структуре CK_FUNCTION_LIST_EXTENDED
    CK_C_EX_GetFunctionListExtended getFunctionListEx; // Указатель на функцию C_EX_GetFunctionListExtended

    CK_SLOT_ID_PTR slots; // Массив идентификаторов слотов
    CK_ULONG slotCount;   // Количество идентификаторов слотов в массиве

    CK_OBJECT_HANDLE publicKey;  // Хэндл открытого ключа ГОСТ Р 34.10-2012(256)
    CK_OBJECT_HANDLE privateKey; // Хэндл закрытого ключа ГОСТ Р 34.10-2012(256)

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на              *
     * расширенные функции                                                    *
     *************************************************************************/
    getFunctionListEx = (CK_C_EX_GetFunctionListExtended)GetProcAddress(module, "C_EX_GetFunctionListExtended");
    CHECK(" GetProcAddress", getFunctionListEx != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на расширенные функции                *
     *************************************************************************/
    rv = getFunctionListEx(&functionListEx);
    CHECK_AND_LOG(" Get function list extended", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    printf(" Slots available: %d\n", (int)slotCount);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);

    printf("Initialization has been completed successfully.\n\n");

    /*************************************************************************
     * Открыть RW сессию в первом доступном слоте                             *
     *************************************************************************/
    printf("Authentication...\n");

    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Выполнить аутентификацию Администратора                                *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_SO, SO_PIN, SO_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Authentication has been completed successfully.\n\n");

    /*************************************************************************
     * Запросить смену пароля пользователя                                    *
     *************************************************************************/
    printf("Forcing user pin change...\n");
    rv = functionListEx->C_EX_TokenManage(session, MODE_FORCE_USER_TO_CHANGE_PIN, &userTypeUser);
    CHECK_AND_LOG(" C_EX_TokenManage", rv == CKR_OK, rvToStr(rv), logout);
    printf("User pin set to be changed.\n\n");

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
    printf("Switching user...\n");
    rv = functionList->C_Logout(session);
    CHECK_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), close_session);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
    rv = functionList->C_CloseSession(session);
    CHECK_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Открыть RW сессию в первом доступном слоте                             *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Выполнить аутентификацию Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Authentication has been completed successfully.\n\n");

    /*************************************************************************
     * Попытаться использовать функцию, требующую аутентификацию (создание    *
     * пары ключей ГОСТ Р 34.10-2012(256))                                    *
     *************************************************************************/
    printf("Checking features...\n");
    rv = functionList->C_GenerateKeyPair(session, &gostR3410_2012_256KeyPairGenMech, publicKeyTemplate,
                                         arraysize(publicKeyTemplate), privateKeyTemplate,
                                         arraysize(privateKeyTemplate), &publicKey, &privateKey);
    // Возвращаемое значение: CKR_PIN_EXPIRED
    CHECK_AND_LOG(" C_GenerateKeyPair is not available (CKR_PIN_EXPIRED)", rv == CKR_PIN_EXPIRED, rvToStr(rv), logout);

    /*************************************************************************
     * Сменить ПИН-код пользователя                                           *
     *************************************************************************/
    rv = functionList->C_SetPIN(session, USER_PIN, USER_PIN_LEN, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_SetPIN", rv == CKR_OK, rvToStr(rv), close_session);

    /*************************************************************************
     * Использовать функции, требующие аутентификацию (создание и удаление    *
     * пары ключей ГОСТ Р 34.10-2012(256))                                    *
     *************************************************************************/
    rv = functionList->C_GenerateKeyPair(session, &gostR3410_2012_256KeyPairGenMech, publicKeyTemplate,
                                         arraysize(publicKeyTemplate), privateKeyTemplate,
                                         arraysize(privateKeyTemplate), &publicKey, &privateKey);
    CHECK_AND_LOG(" C_GenerateKeyPair is available", rv == CKR_OK, rvToStr(rv), logout);

    rv = functionList->C_DestroyObject(session, publicKey);
    CHECK_AND_LOG(" C_DestroyObject (publicKey)", rv == CKR_OK, rvToStr(rv), logout);

    rv = functionList->C_DestroyObject(session, privateKey);
    CHECK_AND_LOG(" C_DestroyObject (privateKey)", rv == CKR_OK, rvToStr(rv), logout);

    printf("Features checked.\n");

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    printf("\nFinalizing... \n");

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, выделенную под слоты                                  *
     *************************************************************************/
free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}