/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
 *------------------------------------------------------------------------*
 * Пример использования функций расширения компании "Актив"               *
 * стандарта PKCS#11:                                                     *
 *  - установление соединения с Rutoken в первом доступном слоте;         *
 *  - проверка необходимости смены ПИН-кода                               *
 *  - поиск объекта HW фичи политики ПИН-кодов                            *
 *  - чтение и вывод атрибутов политики ПИН-кодов                         *
 *------------------------------------------------------------------------*
 * Данный пример является самодостаточным.                                *
 *************************************************************************/
#include <Common.h>

#define SHIFT_SIZE 50

/*************************************************************************
 * Шаблон для поиска объекта с бинарными данными                          *
 *************************************************************************/
CK_ATTRIBUTE hardwareObjectTemplate[] = {
    { CKA_CLASS, &hardwareFeatureObject, sizeof(hardwareFeatureObject) }, // Класс - Hardware Feature Object
    { CKA_HW_FEATURE_TYPE, &hardwareFeaturePinPolicy,
      sizeof(hardwareFeaturePinPolicy) }, // Описание типа объекта, атрибуты которого будут выведены
    { CKA_VENDOR_USER_TYPE, &userTypeUser, sizeof(userTypeUser) } // Тип пользователя
};

/*************************************************************************
 * Шаблон для получения бинарных данных из объекта                        *
 *************************************************************************/
CK_ATTRIBUTE attributeValueTemplate[] = { { CKA_VENDOR_USER_TYPE, NULL_PTR, 0 },
                                          { CKA_VENDOR_PIN_POLICY_STATE, NULL_PTR, 0 },
                                          { CKA_MODIFIABLE, NULL_PTR, 0 },
                                          { CKA_VENDOR_SUPPORTED_PIN_POLICIES, NULL_PTR, 0 },
                                          { CKA_VENDOR_PIN_POLICY_MIN_LENGTH, NULL_PTR, 0 },
                                          { CKA_VENDOR_PIN_POLICY_HISTORY_DEPTH, NULL_PTR, 0 },
                                          { CKA_VENDOR_PIN_POLICY_ALLOW_DEFAULT_PIN_USAGE, NULL_PTR, 0 },
                                          { CKA_VENDOR_PIN_POLICY_DIGIT_REQUIRED, NULL_PTR, 0 },
                                          { CKA_VENDOR_PIN_POLICY_UPPERCASE_REQUIRED, NULL_PTR, 0 },
                                          { CKA_VENDOR_PIN_POLICY_LOWERCASE_REQUIRED, NULL_PTR, 0 },
                                          { CKA_VENDOR_PIN_POLICY_SPEC_CHAR_REQUIRED, NULL_PTR, 0 },
                                          { CKA_VENDOR_PIN_POLICY_DIFF_CHARS_REQUIRED, NULL_PTR, 0 } };

/*************************************************************************
 * Выводит в консоль значение атрибута в человекочитаемой форме, если     *
 * невозможно распознать значение - выводит его в шестнадцатеричной форме *
 *************************************************************************/
void printLabel(size_t* pos, const char* label);
void printShift(size_t* pos, const size_t shiftSize);
void printValue(size_t* pos, const size_t shiftSize, const char* val);
void printLabelAndValue(const size_t shiftSize, const char* label, const char* val);

void printAttribute(const CK_ATTRIBUTE attribute, // Атрибут, значение которого выводится
                    const size_t shiftSize // Кол-во знаков, на которое сдвигается новая строка
) {
    size_t pos = 0;

    switch (attribute.type) {
    case CKA_MODIFIABLE:
        printLabel(&pos, "CKA_MODIFIABLE");
        break;
    case CKA_VENDOR_PIN_POLICY_ALLOW_DEFAULT_PIN_USAGE:
        printLabel(&pos, "CKA_VENDOR_PIN_POLICY_ALLOW_DEFAULT_PIN_USAGE");
        break;
    case CKA_VENDOR_PIN_POLICY_DIGIT_REQUIRED:
        printLabel(&pos, "CKA_VENDOR_PIN_POLICY_DIGIT_REQUIRED");
        break;
    case CKA_VENDOR_PIN_POLICY_UPPERCASE_REQUIRED:
        printLabel(&pos, "CKA_VENDOR_PIN_POLICY_UPPERCASE_REQUIRED");
        break;
    case CKA_VENDOR_PIN_POLICY_LOWERCASE_REQUIRED:
        printLabel(&pos, "CKA_VENDOR_PIN_POLICY_LOWERCASE_REQUIRED");
        break;
    case CKA_VENDOR_PIN_POLICY_SPEC_CHAR_REQUIRED:
        printLabel(&pos, "CKA_VENDOR_PIN_POLICY_SPEC_CHAR_REQUIRED");
        break;
    case CKA_VENDOR_PIN_POLICY_DIFF_CHARS_REQUIRED:
        printLabel(&pos, "CKA_VENDOR_PIN_POLICY_DIFF_CHARS_REQUIRED");
        break;
    case CKA_VENDOR_PIN_POLICY_MIN_LENGTH:
        printLabel(&pos, "CKA_VENDOR_PIN_POLICY_MIN_LENGTH");
        break;
    case CKA_VENDOR_PIN_POLICY_HISTORY_DEPTH:
        printLabel(&pos, "CKA_VENDOR_PIN_POLICY_HISTORY_DEPTH");
        break;
    case CKA_VENDOR_USER_TYPE:
        printLabel(&pos, "CKA_VENDOR_USER_TYPE");
        break;
    case CKA_VENDOR_PIN_POLICY_STATE:
        printLabel(&pos, "CKA_VENDOR_PIN_POLICY_STATE");
        break;
    case CKA_VENDOR_SUPPORTED_PIN_POLICIES:
        printLabel(&pos, "CKA_VENDOR_SUPPORTED_PIN_POLICIES");
        break;
    default:
        return;
    }

    if (attribute.ulValueLen == 0) {
        printValue(&pos, shiftSize, "VOID VALUE");
        printf("\n");
        return;
    }

    switch (attribute.type) {
    /*************************************************************************
     * Семейство типов атрибута с бинарными значениями                        *
     *************************************************************************/
    case CKA_MODIFIABLE:
    case CKA_VENDOR_PIN_POLICY_ALLOW_DEFAULT_PIN_USAGE:
    case CKA_VENDOR_PIN_POLICY_DIGIT_REQUIRED:
    case CKA_VENDOR_PIN_POLICY_UPPERCASE_REQUIRED:
    case CKA_VENDOR_PIN_POLICY_LOWERCASE_REQUIRED:
    case CKA_VENDOR_PIN_POLICY_SPEC_CHAR_REQUIRED:
    case CKA_VENDOR_PIN_POLICY_DIFF_CHARS_REQUIRED:
        switch (*(CK_BBOOL*)attribute.pValue) {
        case CK_TRUE:
            printValue(&pos, shiftSize, "CK_TRUE");
            return;
        case CK_FALSE:
            printValue(&pos, shiftSize, "CK_FALSE");
            return;
        }
        break;

    /*************************************************************************
     * Семейство типов атрибута с численными значениями                       *
     *************************************************************************/
    case CKA_VENDOR_PIN_POLICY_MIN_LENGTH:
    case CKA_VENDOR_PIN_POLICY_HISTORY_DEPTH:
        printShift(&pos, shiftSize);
        printf("%d\n", (int)(*(CK_BYTE*)attribute.pValue));
        return;

    /*************************************************************************
     * Семейство типов атрибута со множественными значениями                  *
     *************************************************************************/
    case CKA_VENDOR_USER_TYPE:
        switch (*(CK_USER_TYPE*)attribute.pValue) {
        case CKU_SO:
            printValue(&pos, shiftSize, "CKU_SO");
            return;
        case CKU_USER:
            printValue(&pos, shiftSize, "CKU_USER");
            return;
        case CKU_CONTEXT_SPECIFIC:
            printValue(&pos, shiftSize, "CKU_CONTEXT_SPECIFIC");
            return;
        }
        break;
    case CKA_VENDOR_PIN_POLICY_STATE:
        switch (*(CK_ULONG*)attribute.pValue) {
        case PIN_POLICY_STATE_WELL_DEFINED:
            printValue(&pos, shiftSize, "PIN_POLICY_STATE_WELL_DEFINED");
            return;
        case PIN_POLICY_STATE_HAS_UNKNOWN_ATTRIBUTES:
            printValue(&pos, shiftSize, "PIN_POLICY_STATE_HAS_UNKNOWN_ATTRIBUTES");
            return;
        case PIN_POLICY_STATE_HAS_UNKNOWN_NONDEFAULT_ATTRIBUTES:
            printValue(&pos, shiftSize, "PIN_POLICY_STATE_HAS_UNKNOWN_NONDEFAULT_ATTRIBUTES");
            return;
        }
        break;
    case CKA_VENDOR_SUPPORTED_PIN_POLICIES:
        for (size_t i = 0; i < attribute.ulValueLen / sizeof(CK_ULONG); ++i) {
            switch (((CK_ULONG_PTR)attribute.pValue)[i]) {
            case CKA_MODIFIABLE:
                printValue(&pos, shiftSize, "CKA_MODIFIABLE");
                break;
            case CKA_VENDOR_PIN_POLICY_MIN_LENGTH:
                printValue(&pos, shiftSize, "CKA_VENDOR_PIN_POLICY_MIN_LENGTH");
                break;
            case CKA_VENDOR_PIN_POLICY_HISTORY_DEPTH:
                printValue(&pos, shiftSize, "CKA_VENDOR_PIN_POLICY_HISTORY_DEPTH");
                break;
            case CKA_VENDOR_PIN_POLICY_ALLOW_DEFAULT_PIN_USAGE:
                printValue(&pos, shiftSize, "CKA_VENDOR_PIN_POLICY_ALLOW_DEFAULT_PIN_USAGE");
                break;
            case CKA_VENDOR_PIN_POLICY_DIGIT_REQUIRED:
                printValue(&pos, shiftSize, "CKA_VENDOR_PIN_POLICY_DIGIT_REQUIRED");
                break;
            case CKA_VENDOR_PIN_POLICY_UPPERCASE_REQUIRED:
                printValue(&pos, shiftSize, "CKA_VENDOR_PIN_POLICY_UPPERCASE_REQUIRED");
                break;
            case CKA_VENDOR_PIN_POLICY_LOWERCASE_REQUIRED:
                printValue(&pos, shiftSize, "CKA_VENDOR_PIN_POLICY_LOWERCASE_REQUIRED");
                break;
            case CKA_VENDOR_PIN_POLICY_SPEC_CHAR_REQUIRED:
                printValue(&pos, shiftSize, "CKA_VENDOR_PIN_POLICY_SPEC_CHAR_REQUIRED");
                break;
            case CKA_VENDOR_PIN_POLICY_DIFF_CHARS_REQUIRED:
                printValue(&pos, shiftSize, "CKA_VENDOR_PIN_POLICY_DIFF_CHARS_REQUIRED");
                break;
            }
        }
        printf("\n");
        return;
    }

    printHexBuffer(attribute.pValue, attribute.ulValueLen, 0, 0, 0);
    printf("\n");
    return;
}

CK_RV checkPinChangeForced(CK_VOID_PTR userType, size_t shiftSize, CK_FUNCTION_LIST_EXTENDED_PTR functionListEx,
                           CK_SLOT_ID slot) {
    CK_RV rv = functionListEx->C_EX_SlotManage(slot, MODE_GET_PIN_SET_TO_BE_CHANGED, userType);
    CHECK_AND_LOG("  C_EX_SlotManage", rv == CKR_OK || rv == CKR_PIN_EXPIRED, rvToStr(rv), return_error);
    printLabelAndValue(shiftSize, "Reason", rv == CKR_OK ? "default pin (change optional)" : "forced to change");
    return CKR_OK;
return_error:
    return rv;
}

CK_RV checkPinChangeRequest(const char* login, CK_FLAGS tokenInfoFlags, CK_FLAGS checkingFlag, CK_VOID_PTR userType,
                            size_t shiftSize, CK_FUNCTION_LIST_EXTENDED_PTR functionListEx, CK_SLOT_ID slot) {
    printLabelAndValue(shiftSize, "Login", login);
    int isPinChangeRequested = tokenInfoFlags & checkingFlag;
    printLabelAndValue(shiftSize, "Pin change requested", isPinChangeRequested ? "true" : "false");
    return isPinChangeRequested ? checkPinChangeForced(userType, shiftSize, functionListEx, slot) : CKR_OK;
}

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_FUNCTION_LIST_EXTENDED_PTR functionListEx; // Указатель на список функций расширения PKCS#11, хранящийся в
                                                  // структуре CK_FUNCTION_LIST_EXTENDED
    CK_C_EX_GetFunctionListExtended getFunctionListEx; // Указатель на функцию C_EX_GetFunctionListExtended

    CK_SLOT_ID_PTR slots;         // Массив идентификаторов слотов
    CK_ULONG slotCount;           // Количество идентификаторов слотов в массиве
    CK_ULONG objectCount;         // Количество хэндлов объектов в массиве
    CK_OBJECT_HANDLE_PTR objects; // Массив найденных объектов

    CK_TOKEN_INFO tokenInfo; // Структура данных типа CK_TOKEN_INFO с информацией о токене

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11

    int r; // Код возврата для функций, возвращающих int

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/

    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на              *
     * расширенные функции                                                    *
     *************************************************************************/
    getFunctionListEx = (CK_C_EX_GetFunctionListExtended)GetProcAddress(module, "C_EX_GetFunctionListExtended");
    CHECK(" GetProcAddress", getFunctionListEx != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на расширенные функции                *
     *************************************************************************/
    rv = getFunctionListEx(&functionListEx);
    CHECK_AND_LOG(" Get function list extended", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    printf(" Slots available: %d\n", (int)slotCount);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);

    printf("Initialization has been completed successfully.\n\n");

    /*************************************************************************
     * Открыть RW сессию в первом доступном слоте                             *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Проверить необходимость смены пароля                                   *
     *************************************************************************/
    rv = functionList->C_GetTokenInfo(slots[0], &tokenInfo);
    CHECK_AND_LOG(" C_GetTokenInfo", rv == CKR_OK, rvToStr(rv), free_slots);

    rv = checkPinChangeRequest("User", tokenInfo.flags, CKF_USER_PIN_TO_BE_CHANGED, &userTypeUser, SHIFT_SIZE,
                               functionListEx, slots[0]);
    CHECK("Pin change request check", rv == CKR_OK, free_slots);

    rv = checkPinChangeRequest("SO", tokenInfo.flags, CKF_SO_PIN_TO_BE_CHANGED, &userTypeSO, SHIFT_SIZE, functionListEx,
                               slots[0]);
    CHECK("Pin change request check", rv == CKR_OK, free_slots);

    /*************************************************************************
     * Выполнить аутентификацию Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG("\n\n C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Authentication has been completed successfully.\n");

    /*************************************************************************
     * Найти объект на токене                                                 *
     *************************************************************************/
    printf("\nFinding hardware feature object...\n");

    r = findObjects(functionList, session, hardwareObjectTemplate, arraysize(hardwareObjectTemplate), &objects,
                    &objectCount);
    CHECK(" FindObjects", r == 0, logout);
    CHECK(" Check hardware feature object is found", objectCount != 0, logout);
    CHECK(" Check hardware feature object is single", objectCount == 1, logout);

    /*************************************************************************
     * Получить размер каждого атрибута найденного объекта                    *
     *************************************************************************/
    printf("\nGetting data from object...\n");

    printf(" Getting the size of each attribute from attributeValueTemplate...\n");
    rv = functionList->C_GetAttributeValue(session, objects[0], attributeValueTemplate,
                                           arraysize(attributeValueTemplate));
    CHECK_AND_LOG("  C_GetAttributeValue", rv == CKR_OK, rvToStr(rv), free_objects);

    /*************************************************************************
     * Выделить память под каждый из атрибутов                                *
     *************************************************************************/
    CK_ULONG i;

    printf(" Allocating memory for attributes...\n");
    for (i = 0; i < arraysize(attributeValueTemplate); ++i) {
        attributeValueTemplate[i].pValue = (CK_VOID_PTR)malloc(attributeValueTemplate[i].ulValueLen);
        CHECK_AND_LOG_ONLY_ERROR(attributeValueTemplate[i].pValue != NULL, "Allocating memory for attributes failed",
                                 free_attributeValues);
    }

    /*************************************************************************
     * Получить значения атрибутов                                            *
     *************************************************************************/
    printf(" Getting the values of attributes...\n");
    rv = functionList->C_GetAttributeValue(session, objects[0], attributeValueTemplate,
                                           arraysize(attributeValueTemplate));
    CHECK_AND_LOG("  C_GetAttributeValue", rv == CKR_OK, rvToStr(rv), free_attributeValues);
    printf("\n");

    /*************************************************************************
     * Вывести значения атрибутов                                             *
     *************************************************************************/
    for (i = 0; i < arraysize(attributeValueTemplate); ++i) {
        printAttribute(attributeValueTemplate[i], SHIFT_SIZE);
    }
    printf("\n");

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    printf("\nFinalizing... \n");

    /*************************************************************************
     * Очистить память, выделенную под атрибуты объекта
     *************************************************************************/
free_attributeValues:
    for (i = 0; i < arraysize(attributeValueTemplate); ++i) {
        free(attributeValueTemplate[i].pValue);
    }

    /*************************************************************************
     * Очистить память, выделенную под объекты                                *
     *************************************************************************/
free_objects:
    free(objects);

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, выделенную под слоты                                  *
     *************************************************************************/
free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}

void printLabel(size_t* pos, const char* label) {
    printf("\n%s:", label);
    *pos = strlen(label) + 1;
}

void printShift(size_t* pos, const size_t shiftSize) {
    for (; *pos < shiftSize; ++*pos) {
        printf(" ");
    }
}

void printValue(size_t* pos, const size_t shiftSize, const char* val) {
    printShift(pos, shiftSize);
    printf("%s\n", val);
    *pos = 0;
}

void printLabelAndValue(const size_t shiftSize, const char* label, const char* val) {
    size_t pos;
    printLabel(&pos, label);
    printValue(&pos, shiftSize, val);
}
