/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен ЭЦП при помощи библиотеки PKCS#11 на языке C   *
 *------------------------------------------------------------------------*
 * Использование команды проверки подписанных данных ключевой парой       *
 * ГОСТ Р 34.10 в формате PKCS#7:                                         *
 *  - установление соединения с Рутокен ЭЦП в первом доступном слоте;     *
 *  - выполнение аутентификации Пользователя;                             *
 *  - проверка подписанных данных в формате PKCS#7;                       *
 *  - сброс прав доступа Пользователя и закрытие соединения с Рутокен.    *
 *                                                                        *
 *------------------------------------------------------------------------*
 * Пример использует объекты, созданные в памяти примерами:               *
 *                                                                        *
 * Для алгоритма ГОСТ Р 34.10-2001:                                       *
 * ImportCertificate-GOST34.10-2001 и SignPKCS7-GOST34.10-2001.           *
 *                                                                        *
 * Для алгоритма ГОСТ Р 34.10-2012 с длиной закрытого ключа 256 бит:      *
 * ImportCertificate-GOST34.10-2012-256 и SignPKCS7-GOST34.10-2012-256.   *
 *                                                                        *
 * Для алгоритма ГОСТ Р 34.10-2012 с длиной закрытого ключа 512 бит:      *
 * ImportCertificate-GOST34.10-2012-512 и SignPKCS7-GOST34.10-2012-512.   *
 *                                                                        *
 * Также необходимо предоставить сертификат УЦ, в котором был выписан     *
 * сертификат в der кодировке. Файл, содержащий сертификат, необходимо    *
 * назвать "CA_cert.cer" и положить в папку, содержащую исполняемый файл  *
 * примера.                                                               *
 *************************************************************************/

#include <Common.h>

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_FUNCTION_LIST_EXTENDED_PTR functionListEx; // Указатель на список функций расширения PKCS#11, хранящийся в
                                                  // структуре CK_FUNCTION_LIST_EXTENDED
    CK_C_EX_GetFunctionListExtended getFunctionListEx; // Указатель на функцию C_EX_GetFunctionListExtended

    FILE* inputCms;    // Описатель потока ввода CMS
    FILE* inputCaCert; // Описатель потока ввода сертификата УЦ

    CK_SLOT_ID_PTR slots; // Указатель на массив идентификаторов слотов
    CK_ULONG slotCount;   // Количество идентификаторов слотов в массиве

    CK_BYTE_PTR cms;  // Указатель на CMS
    CK_ULONG cmsSize; // Размер CMS, в байтах

    CK_VENDOR_BUFFER rootCertByteArrays[1]; // Массив буферов, содержащих сертификаты в виде массива байт
    CK_ULONG rootCertByteArraysCount = 1; // Количество буферов, содержащих сертификаты в виде массива байт

    CK_VENDOR_X509_STORE store; // структура, содержащая указатели на необходимые для проверки подписи сертификаты

    CK_VENDOR_BUFFER_PTR
    signerCertificates; // Указатель на массив буферов, содержащих сертификаты, использованные при подписи
    CK_ULONG signerCertificatesCount; // Количество элементов в массиве буферов, содержащих сертификаты, использованные
                                      // при подписи

    CK_BYTE_PTR signedData; // Указатель на подписанные данные, которые вернула C_EX_PKCS7Verify
    CK_ULONG signedDataSize; // Размер подписанных данных, которые вернула C_EX_PKCS7Verify

    CK_ULONG i; // Счетчик

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
    int r;    // Код возврата для функций, возвращающих int

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     * расширения стандарта PKCS#11                                           *
     *************************************************************************/
    getFunctionListEx = (CK_C_EX_GetFunctionListExtended)GetProcAddress(module, "C_EX_GetFunctionListExtended");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции расширения стандарта       *
     *************************************************************************/
    rv = getFunctionListEx(&functionListEx);
    CHECK_AND_LOG(" Get function list extended", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
    printf(" Slots available: %d\n", (int)slotCount);

    /*************************************************************************
     * Открыть сессию в первом доступном слоте                                *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Выполнить аутентификацию Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Initialization has been completed successfully.\n");

    /*************************************************************************
     * Чтение CMS                                                             *
     *************************************************************************/
    printf("\nReading CMS...\n");

    /*************************************************************************
     * Открыть поточный ввод CMS из файла                                     *
     *************************************************************************/
    inputCms = fopen("cms_data", "rb");
    CHECK_AND_LOG(" fopen", inputCms != NULL, "\"cms_data\" doesn't exist", logout);

    /*************************************************************************
     * Определить размер файла                                                *
     *************************************************************************/
    r = fseek(inputCms, 0, SEEK_END);
    CHECK(" fseek", r == 0, close_inputCms);
    cmsSize = ftell(inputCms);
    CHECK(" ftell", cmsSize > 0, close_inputCms);
    r = fseek(inputCms, 0, SEEK_SET);
    CHECK(" fseek", r == 0, close_inputCms);

    /*************************************************************************
     * Выделить память для CMS                                                *
     *************************************************************************/
    cms = (CK_BYTE_PTR)malloc(cmsSize);
    CHECK(" malloc", cms != NULL, close_inputCms);

    /*************************************************************************
     * Прочитать CMS                                                          *
     *************************************************************************/
    r = (int)fread(cms, 1, (int)cmsSize, inputCms);
    CHECK(" fread", r == (int)cmsSize, free_cms);

    /*************************************************************************
     * Чтение сертификата УЦ                                                  *
     *************************************************************************/
    printf("\nReading CA certificate...\n");

    /*************************************************************************
     * Открыть поточный ввод сертификата УЦ из файла                          *
     *************************************************************************/
    inputCaCert = fopen("CA_cert.cer", "rb");
    CHECK_AND_LOG(" fopen", inputCaCert != NULL, "\"CA_cert.cer\" doesn't exist", free_cms);

    /*************************************************************************
     * Определить размер файла                                                *
     *************************************************************************/
    r = fseek(inputCaCert, 0, SEEK_END);
    CHECK(" fseek", r == 0, close_input_ca_cert);
    rootCertByteArrays[0].ulSize = ftell(inputCaCert);
    CHECK(" ftell", rootCertByteArrays[0].ulSize > 0, close_input_ca_cert);
    r = fseek(inputCaCert, 0, SEEK_SET);
    CHECK(" fseek", r == 0, close_input_ca_cert);

    /*************************************************************************
     * Выделить память для сертификата УЦ                                     *
     *************************************************************************/
    rootCertByteArrays[0].pData = (CK_BYTE_PTR)malloc(rootCertByteArrays[0].ulSize);
    CHECK(" malloc", rootCertByteArrays[0].pData != NULL, close_input_ca_cert);

    /*************************************************************************
     * Прочитать сертификат УЦ                                                *
     *************************************************************************/
    r = (int)fread(rootCertByteArrays[0].pData, 1, (int)rootCertByteArrays[0].ulSize, inputCaCert);
    CHECK(" fread", r == (int)rootCertByteArrays[0].ulSize, free_certificate);

    /*************************************************************************
     * Заполнить структуру CK_VENDOR_X509_STORE                               *
     *************************************************************************/
    store.pTrustedCertificates = rootCertByteArrays; // указатель на массив доверенных сертификатов
    store.ulTrustedCertificateCount = rootCertByteArraysCount; // количество доверенных сертификатов в массиве
    store.pCertificates = NULL_PTR; // указатель на массив, содержащий сертификаты для проверки подписи
    store.ulCertificateCount = 0; // количество сертификатов в цепочке сертификатов
    store.pCrls = NULL_PTR; // указатель на массив списков отзыва сертификатов
    store.ulCrlCount = 0; // количество списков отзыва сертификатов в массиве

    /*************************************************************************
     * Проверка подписи                                                       *
     *************************************************************************/
    printf("\nVerifying...\n");

    /*************************************************************************
     * Инициализировать операцию проверки подписи                             *
     *************************************************************************/
    rv = functionListEx->C_EX_PKCS7VerifyInit(session, cms, cmsSize, &store, OPTIONAL_CRL_CHECK, 0);
    CHECK_AND_LOG(" C_EX_PKCS7VerifyInit", rv == CKR_OK, rvToStr(rv), free_certificate);

    /*************************************************************************
     * Проверить подпись                                                      *
     *************************************************************************/
    rv = functionListEx->C_EX_PKCS7Verify(session, &signedData, &signedDataSize, &signerCertificates,
                                          &signerCertificatesCount);
    CHECK_AND_LOG(" C_EX_PKCS7Verify", rv == CKR_OK, rvToStr(rv), free_certificate);

    /*************************************************************************
     * Распечатать буфер, содержащий подписанные данные                       *
     *************************************************************************/
    printf(" Signed data is:\n");
    printHex(signedData, signedDataSize);

    /*************************************************************************
     * Распечатать буферы, содержащие сертификаты, использованные при подписи *
     *************************************************************************/
    for (i = 0; i < signerCertificatesCount; ++i) {
        printf(" %lu signer certificate's data is:\n", i + 1);
        printHex(signerCertificates[i].pData, signerCertificates[i].ulSize);
    }

    printf("Data has been verified successfully.\n");

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    /*************************************************************************
     * Выполнить действия для завершения работы с библиотекой PKCS#11         *
     *************************************************************************/
    printf("\nFinalizing... \n");

    /*************************************************************************
     * Очистить память, содержащую сертификаты, использованные при подписи    *
     *************************************************************************/
    for (i = 0; i < signerCertificatesCount; ++i) {
        rv = functionListEx->C_EX_FreeBuffer((CK_BYTE_PTR)signerCertificates[i].pData);
        CHECK_RELEASE_AND_LOG(" C_EX_FreeBuffer", rv == CKR_OK, rvToStr(rv), errorCode);
    }

    rv = functionListEx->C_EX_FreeBuffer((CK_BYTE_PTR)signerCertificates);
    CHECK_RELEASE_AND_LOG(" C_EX_FreeBuffer", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, содержащую подписанные данные                         *
     *************************************************************************/
    rv = functionListEx->C_EX_FreeBuffer(signedData);
    CHECK_RELEASE_AND_LOG(" C_EX_FreeBuffer", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, содержащую массив байт с сертификатом                 *
     *************************************************************************/
free_certificate:
    free(rootCertByteArrays[0].pData);

    /*************************************************************************
     * Закрыть поток ввода сертификата УЦ                                     *
     *************************************************************************/
close_input_ca_cert:
    r = fclose(inputCaCert);
    CHECK_RELEASE(" fclose", r == 0, errorCode);

    /*************************************************************************
     * Освободить память, выделенную для CMS                                  *
     *************************************************************************/
free_cms:
    free(cms);

    /*************************************************************************
     * Закрыть поток ввода CMS                                                *
     *************************************************************************/
close_inputCms:
    r = fclose(inputCms);
    CHECK_RELEASE(" fclose", r == 0, errorCode);

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память из-под слотов                                          *
     *************************************************************************/
free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}