/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
 *------------------------------------------------------------------------*
 * Использование команд шифрования/расшифрования на ключе ГОСТ 34.12-2018 *
 * с длиной блока 128 бит (Кузнечик):                                     *
 *  - установление соединения с Рутокен в первом доступном слоте;         *
 *  - выполнение аутентификации Пользователя;                             *
 *  - шифрование сообщения на демонстрационном ключе (одним блоком);      *
 *  - расшифрование зашифрованнного сообщения на демонстрационном ключе;  *
 *  - сброс прав доступа Пользователя на Рутокен и закрытие соединения    *
 *    с Рутокен.                                                          *
 *------------------------------------------------------------------------*
 * Пример использует объекты, созданные в памяти Рутокен примером         *
 * CreateForEncDec.                                                       *
 *************************************************************************/

#include <Common.h>

/*************************************************************************
 * Шаблон для поиска симметричного ключа Кузнечик                         *
 *************************************************************************/
CK_ATTRIBUTE secKeyTemplate[] = {
    { CKA_ID, &secretKeyKuznechikId, sizeof(secretKeyKuznechikId) - 1 }, // Идентификатор ключа
    { CKA_CLASS, &secretKeyObject, sizeof(secretKeyObject) },            // Класс - секретный ключ
    { CKA_KEY_TYPE, &keyTypeKuznechik, sizeof(keyTypeKuznechik) },       // Тип ключа - Кузнечик
    { CKA_ENCRYPT, &attributeTrue, sizeof(attributeTrue) }, // Ключ предназначен для зашифрования
    { CKA_DECRYPT, &attributeTrue, sizeof(attributeTrue) }, // Ключ предназначен для расшифрования
};

/*************************************************************************
 * Шаблон для поиска симметричного ключа Магма                            *
 *************************************************************************/
// CK_ATTRIBUTE secKeyTemplate[] =
//{
//	{ CKA_ID, &secretKeyMagmaId, sizeof(secretKeyMagmaId) - 1 }, // Идентификатор ключа
//	{ CKA_CLASS, &secretKeyObject, sizeof(secretKeyObject) },    // Класс - секретный ключ
//	{ CKA_KEY_TYPE, &keyTypeMagma, sizeof(keyTypeMagma) },       // Тип ключа - Магма
//	{ CKA_ENCRYPT, &attributeTrue, sizeof(attributeTrue) },      // Ключ предназначен для зашифрования
//	{ CKA_DECRYPT, &attributeTrue, sizeof(attributeTrue) },      // Ключ предназначен для расшифрования
// };

/*************************************************************************
 * Шаблон для поиска симметричного ключа ГОСТ 28147-89                    *
 *************************************************************************/
// CK_ATTRIBUTE secKeyTemplate[] =
//{
//	{ CKA_ID, &secretKeyId, sizeof(secretKeyId) - 1 },             // Идентификатор ключа
//	{ CKA_CLASS, &secretKeyObject, sizeof(secretKeyObject) },      // Класс - секретный ключ
//	{ CKA_KEY_TYPE, &keyTypeGost28147, sizeof(keyTypeGost28147) }, // Тип ключа - ГОСТ 28147-89
// };

/*************************************************************************
 * Данные для шифрования                                                  *
 *************************************************************************/
CK_BYTE data[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                   0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x00 };

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_SLOT_ID_PTR slots; // Массив идентификаторов слотов
    CK_ULONG slotCount;   // Количество идентификаторов слотов в массиве

    CK_MECHANISM_TYPE_PTR mechanisms; // Массив поддерживаемых механизмов
    CK_ULONG mechanismCount;          // Количество поддерживаемых механизмов

    CK_OBJECT_HANDLE_PTR encKeys; // Массив хэндлов ключей для зашифрования
    CK_ULONG encKeysCount;        // Количество хэндлов объектов в массиве

    CK_BYTE_PTR encrypted; // Указатель на временный буфер для зашифрованных данных
    CK_ULONG encryptedSize; // Размер временного буфера в байтах

    CK_OBJECT_HANDLE_PTR decKeys; // Массив хэндлов ключей для расшифрования
    CK_ULONG decKeysCount;        // Количество хэндлов объектов в массиве

    CK_BYTE_PTR decrypted; // Указатель на временный буфер для расшифрованных данных
    CK_ULONG decryptedSize; // Размер временного буфера в байтах

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
    int r;    // Код возврата для функций, возвращающих int

    CK_ULONG i; // Вспомогательная переменная-счетчик в циклах

    int isEcbSupported = 0; // Флаг для проверки поддержки токеном механизма

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
    printf(" Slots available: %d\n", (int)slotCount);

    /*************************************************************************
     * Получить список поддерживаемых токеном механизмов                      *
     *************************************************************************/
    rv = functionList->C_GetMechanismList(slots[0], NULL_PTR, &mechanismCount);
    CHECK_AND_LOG(" C_GetMechanismList (number of mechanisms)", rv == CKR_OK, rvToStr(rv), free_slots);

    CHECK_AND_LOG(" Checking mechanisms available", mechanismCount > 0, " No mechanisms available", free_slots);

    mechanisms = (CK_MECHANISM_TYPE_PTR)malloc(mechanismCount * sizeof(CK_MECHANISM_TYPE));
    CHECK(" Memory allocation for mechanisms", mechanisms != NULL_PTR, free_slots);

    rv = functionList->C_GetMechanismList(slots[0], mechanisms, &mechanismCount);
    CHECK_AND_LOG(" C_GetMechanismList", rv == CKR_OK, rvToStr(rv), free_mechanisms);

    /*************************************************************************
     * Определение поддерживаемых токеном механизмов                          *
     *************************************************************************/
    for (i = 0; i < mechanismCount; ++i) {
        if (mechanisms[i] == CKM_KUZNECHIK_ECB) {
            isEcbSupported = 1;
            break;
        }

        // if (mechanisms[i] == CKM_MAGMA_ECB) {
        //	isEcbSupported = 1;
        //	break;
        // }

        // if (mechanisms[i] == CKM_GOST28147_ECB) {
        //	isEcbSupported = 1;
        //	break;
        // }
    }

    CHECK_AND_LOG(" Checking CKM_KUZNECHIK_ECB support", isEcbSupported, "CKM_KUZNECHIK_ECB isn`t supported!\n",
                  free_mechanisms);

    /*************************************************************************
     * Открыть сессию в первом доступном слоте                               *
     ************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_mechanisms);

    /*************************************************************************
     * Выполнить аутентификацию Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Initialization has been completed successfully.\n");

    /*************************************************************************
     * Зашифровать данные по алгоритму Кузнечик                               *
     *************************************************************************/
    printf("\nEncrypting...\n");

    /*************************************************************************
     * Получить массив хэндлов секретных ключей                               *
     *************************************************************************/
    printf(" Getting secret key...\n");
    r = findObjects(functionList, session, secKeyTemplate, arraysize(secKeyTemplate), &encKeys, &encKeysCount);
    CHECK(" findObjects", r == 0, logout);

    CHECK_AND_LOG(" Checking number of keys found", encKeysCount > 0, "No objects found\n", logout);

    /*************************************************************************
     * Инициализировать операцию шифрования по алгоритму Кузнечик             *
     *************************************************************************/
    rv = functionList->C_EncryptInit(session, &kuznechikEncDecEcbMech, encKeys[0]);
    CHECK_AND_LOG(" C_EncryptInit", rv == CKR_OK, rvToStr(rv), free_encKeys);

    /*************************************************************************
     * Инициализировать операцию шифрования по алгоритму Магма                *
     *************************************************************************/
    // rv = functionList->C_EncryptInit(session, &magmaEncDecEcbMech, encKeys[0]);
    // CHECK_AND_LOG(" C_EncryptInit", rv == CKR_OK, rvToStr(rv), free_encKeys);

    /*************************************************************************
     * Инициализировать операцию шифрования по алгоритму ГОСТ 28147-89        *
     *************************************************************************/
    // rv = functionList->C_EncryptInit(session, &gost28147EncDecEcbMech, encKeys[0]);
    // CHECK_AND_LOG(" C_EncryptInit", rv == CKR_OK, rvToStr(rv), free_encKeys);

    /*************************************************************************
     * Зашифровать данные (при шифровании с использованием механизма          *
     * CKM_KUZNECHIK_ECB размер данных должен быть кратен 16, при шифровании  *
     * с использованием CKM_MAGMA_ECB или CKM_GOST28147_ECB размер данных     *
     * должен быть кратен 8)                                                  *
     *************************************************************************/
    encryptedSize = sizeof(data);

    encrypted = (CK_BYTE_PTR)malloc(encryptedSize * sizeof(CK_BYTE));
    CHECK("  Memory allocation for encrypted data", encrypted != NULL_PTR, free_encKeys);

    rv = functionList->C_Encrypt(session, data, sizeof(data), encrypted, &encryptedSize);
    CHECK_AND_LOG(" C_Encrypt", rv == CKR_OK, rvToStr(rv), free_encrypted);

    /*************************************************************************
     * Распечатать буфер, содержащий зашифрованные данные                     *
     *************************************************************************/
    printf(" Encrypted buffer is:\n");
    printHex(encrypted, encryptedSize);

    printf("Encryption has been completed successfully.\n");

    /*************************************************************************
     * Расшифровать данные по алгоритму Кузнечик                              *
     *************************************************************************/
    printf("\nDecrypting data...\n");

    /*************************************************************************
     * Получить массив хэндлов секретных ключей                               *
     *************************************************************************/
    printf(" Getting secret key...\n");
    r = findObjects(functionList, session, secKeyTemplate, arraysize(secKeyTemplate), &decKeys, &decKeysCount);
    CHECK(" findObjects", r == 0, free_encrypted);

    CHECK_AND_LOG(" Checking number of keys found", decKeysCount > 0, "No objects found\n", free_encKeys);

    /**************************************************************************
     * Инициализировать операцию расшифрования по алгоритму Кузнечик           *
     **************************************************************************/
    rv = functionList->C_DecryptInit(session, &kuznechikEncDecEcbMech, decKeys[0]);
    CHECK_AND_LOG(" C_DecryptInit", rv == CKR_OK, rvToStr(rv), free_decKeys);

    /*************************************************************************
     * Инициализировать операцию расшифрования по алгоритму Магма             *
     *************************************************************************/
    // rv = functionList->C_DecryptInit(session, &magmaEncDecEcbMech, decKeys[0]);
    // CHECK_AND_LOG(" C_DecryptInit", rv == CKR_OK, rvToStr(rv), free_decKeys);

    /*************************************************************************
     * Инициализировать операцию расшифрования по алгоритму ГОСТ 28147-89     *
     *************************************************************************/
    // rv = functionList->C_DecryptInit(session, &gost28147EncDecEcbMech, decKeys[0]);
    // CHECK_AND_LOG(" C_DecryptInit", rv == CKR_OK, rvToStr(rv), free_decKeys);

    /*************************************************************************
     * Расшифровать данные                                                    *
     *************************************************************************/
    decryptedSize = encryptedSize;

    decrypted = (CK_BYTE_PTR)malloc(decryptedSize * sizeof(CK_BYTE));
    CHECK("  Memory allocation for decrypted data", decrypted != NULL_PTR, free_decKeys);

    rv = functionList->C_Decrypt(session, encrypted, encryptedSize, decrypted, &decryptedSize);
    CHECK_AND_LOG(" C_Decrypt", rv == CKR_OK, rvToStr(rv), free_decrypted);

    /*************************************************************************
     * Распечатать буфер, содержащий расшифрованные данные                    *
     *************************************************************************/
    printf(" Decrypted buffer is:\n");
    printHex(decrypted, decryptedSize);

    printf("Decryption has been completed successfully.\n");

    /*************************************************************************
     * Сравнить исходные данные с расшифрованными                             *
     *************************************************************************/
    r = (decryptedSize == sizeof(data)) && (memcmp(data, decrypted, decryptedSize) == 0);
    CHECK("Compare decrypted and plain texts", r, free_decrypted);

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    /*************************************************************************
     * Выполнить действия для завершения работы с библиотекой PKCS#11         *
     *************************************************************************/
    printf("\nFinalizing... \n");

    /*************************************************************************
     * Очистить память, выделенную под расшифрованные и зашифрованные данные, *
     * объекты ключей                                                         *
     *************************************************************************/
free_decrypted:
    free(decrypted);

free_decKeys:
    free(decKeys);

free_encrypted:
    free(encrypted);

free_encKeys:
    free(encKeys);

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, выделенную под механизмы и слоты                      *
     *************************************************************************/
free_mechanisms:
    free(mechanisms);

free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}
