/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
 *------------------------------------------------------------------------*
 * Использование команд шифрования/расшифрования на ключе ГОСТ 28147-89:  *
 *  - установление соединения с Рутокен в первом доступном слоте;         *
 *  - выполнение аутентификации Пользователя;                             *
 *  - шифрование сообщения на импортированном ключе (одним блоком);       *
 *  - расшифрование зашифрованнного сообщения на импортированном ключе;   *
 *  - сброс прав доступа Пользователя на Рутокен и закрытие соединения    *
 *    с Рутокен.                                                          *
 *------------------------------------------------------------------------*
 * Данный пример является самодостаточным.                                *
 *************************************************************************/

#include <Common.h>

/*************************************************************************
 * Ключ ГОСТ 28147-89 для импорта                                         *
 *************************************************************************/
CK_BYTE keyValue[GOST_28147_KEY_SIZE] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A,
                                          0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
                                          0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F };

/*************************************************************************
 * Шаблон для импорта симметричного ключа ГОСТ 28147-89                   *
 *************************************************************************/
CK_ATTRIBUTE importKeyTemplate[] = {
    { CKA_CLASS, &secretKeyObject, sizeof(secretKeyObject) },      // Класс - секретный ключ
    { CKA_LABEL, &secretKeyLabel, sizeof(secretKeyLabel) - 1 },    // Метка ключа
    { CKA_ID, &secretKeyId, sizeof(secretKeyId) - 1 },             // Идентификатор ключа
    { CKA_KEY_TYPE, &keyTypeGost28147, sizeof(keyTypeGost28147) }, // Тип ключа - ГОСТ 28147-89
    { CKA_ENCRYPT, &attributeTrue, sizeof(attributeTrue) }, // Ключ предназначен для зашифрования
    { CKA_DECRYPT, &attributeTrue, sizeof(attributeTrue) }, // Ключ предназначен для расшифрования
    { CKA_TOKEN, &attributeFalse, sizeof(attributeFalse) }, // Ключ является объектом токена
    { CKA_PRIVATE, &attributeTrue, sizeof(attributeTrue) }, // Ключ недоступен без аутентификации на токене
    { CKA_GOST28147_PARAMS, parametersGost28147, sizeof(parametersGost28147) }, // Параметры алгоритма из стандарта
    { CKA_VALUE, keyValue, sizeof(keyValue) } // значение секретного ключа
};

/*************************************************************************
 * Константа для Crypto Pro Key Meshing                                   *
 *************************************************************************/
CK_BYTE constForKeyMeshing[] = { 0x69, 0x00, 0x72, 0x22, 0x64, 0xC9, 0x04, 0x23, 0x8D, 0x3A, 0xDB,
                                 0x96, 0x46, 0xE9, 0x2A, 0xC4, 0x18, 0xFE, 0xAC, 0x94, 0x00, 0xED,
                                 0x07, 0x12, 0xC0, 0x86, 0xDC, 0xC2, 0xEF, 0x4C, 0xA9, 0x2B };

/*************************************************************************
 * Шаблон для поиска симметричного ключа ГОСТ 28147-89                    *
 *************************************************************************/
CK_ATTRIBUTE secKeyTemplate[] = {
    { CKA_ID, &secretKeyId, sizeof(secretKeyId) - 1 },             // Идентификатор ключа
    { CKA_CLASS, &secretKeyObject, sizeof(secretKeyObject) },      // Класс - секретный ключ
    { CKA_KEY_TYPE, &keyTypeGost28147, sizeof(keyTypeGost28147) }, // Тип ключа - ГОСТ 28147-89
};

/*************************************************************************
 * Данные для шифрования                                                  *
 *************************************************************************/
CK_BYTE data[2421];

/*************************************************************************
 * XOR двух блоков                                                        *
 *************************************************************************/
void Xor(CK_BYTE_PTR a, CK_BYTE_PTR b, CK_ULONG blockSize, CK_BYTE_PTR result) {
    CK_ULONG i;
    for (i = 0; i < blockSize; ++i) {
        result[i] = a[i] ^ b[i];
    }
}

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_SLOT_ID_PTR slots; // Массив идентификаторов слотов
    CK_ULONG slotCount;   // Количество идентификаторов слотов в массиве

    CK_OBJECT_HANDLE encKey; // Хэндл ключа для зашифрования

    CK_BYTE_PTR dataWithPadding; // Указатель на временный буфер для данных, подготовленных к шифрованию
    CK_ULONG dataWithPaddingSize; // Размер временного буфера в байтах

    CK_BYTE_PTR encrypted; // Указатель на временный буфер для зашифрованных данных
    CK_ULONG encryptedSize; // Размер временного буфера в байтах

    CK_OBJECT_HANDLE_PTR decKeys; // Массив хэндлов ключей для расшифрования
    CK_ULONG decKeysCount;        // Количество хэндлов объектов в массиве

    CK_BYTE_PTR decrypted; // Указатель на временный буфер для расшифрованных данных
    CK_ULONG decryptedSize; // Размер временного буфера в байтах

    CK_BYTE paddingSize; // Размер дополнения данных
    CK_ULONG blockSize;  // Размер блока данных

    CK_BYTE encryptedRound[GOST28147_89_BLOCK_SIZE]; // Результат раунда в процессе зашифрования
    CK_BYTE decryptedRound[GOST28147_89_BLOCK_SIZE]; // Результат раунда в процессе расшифрования
    CK_BYTE xorBlock[GOST28147_89_BLOCK_SIZE]; // Вспомогательный буфер для операции XOR
    CK_BYTE initVector[GOST28147_89_BLOCK_SIZE]; // Буфер, содержащий вектор инициализации

    CK_BYTE keyBlobForMeshing[GOST_28147_KEY_SIZE]; // Буфер для хранения ключа при Key Meshing

    // Шаблон для установки нового значения ключа при Key Meshing
    CK_ATTRIBUTE keyValueAttr;

    CK_ULONG keySize; // Размер секретного ключа

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
    int r;    // Код возврата для функций, возвращающих int

    CK_ULONG i; // Вспомогательная переменная-счетчик в циклах

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Заполнить шаблон                                                       *
     *************************************************************************/
    keyValueAttr.type = CKA_VALUE;
    keyValueAttr.pValue = keyBlobForMeshing;
    keyValueAttr.ulValueLen = sizeof(keyBlobForMeshing);

    /*************************************************************************
     * Заполнить data                                                         *
     *************************************************************************/
    memset(data, 0xff, sizeof(data));

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
    printf(" Slots available: %d\n", (int)slotCount);

    /*************************************************************************
     * Открыть сессию в первом доступном слоте                                *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Выполнить аутентификацию Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Initialization has been completed successfully.\n");

    /*************************************************************************
     * Выполнить дополнение данных по ISO 10126                               *
     *************************************************************************/
    paddingSize = GOST28147_89_BLOCK_SIZE - sizeof(data) % GOST28147_89_BLOCK_SIZE;

    dataWithPaddingSize = sizeof(data) + paddingSize;
    dataWithPadding = (CK_BYTE_PTR)malloc(dataWithPaddingSize * sizeof(CK_BYTE));
    CHECK("Memory allocation for data with padding", dataWithPadding != NULL_PTR, logout);

    memcpy(dataWithPadding, data, sizeof(data));

    printf("Fill in padding...\n");
    r = functionList->C_GenerateRandom(session, dataWithPadding + sizeof(data), paddingSize - 1);
    CHECK_AND_LOG(" C_GenerateRandom", rv == CKR_OK, rvToStr(rv), free_dataWithPadding);

    dataWithPadding[dataWithPaddingSize - 1] = paddingSize;

    /*************************************************************************
     * Зашифровать данные по алгоритму ГОСТ 28147-89                          *
     *************************************************************************/
    printf("\nEncrypting...\n");

    /*************************************************************************
     * Импорт секретного ключа                                                *
     *                                                                        *
     * Реализация режима CBC c учетом RFC4357 на основе режима ECB            *
     * возможна только для импортированных ключей                             *
     *************************************************************************/
    printf(" Import secret key...\n");
    r = functionList->C_CreateObject(session, importKeyTemplate, arraysize(importKeyTemplate), &encKey);
    CHECK_AND_LOG("  C_CreateObject", rv == CKR_OK, rvToStr(rv), free_dataWithPadding);

    memcpy(keyBlobForMeshing, keyValue, GOST_28147_KEY_SIZE);
    keySize = GOST_28147_KEY_SIZE;

    encryptedSize = dataWithPaddingSize;

    encrypted = (CK_BYTE_PTR)malloc(encryptedSize * sizeof(CK_BYTE));
    CHECK(" Memory allocation for encrypted data", encrypted != NULL_PTR, free_dataWithPadding);

    printf(" Fill in IV...\n");
    r = functionList->C_GenerateRandom(session, initVector, GOST28147_89_BLOCK_SIZE);
    CHECK_AND_LOG("  C_GenerateRandom", rv == CKR_OK, rvToStr(rv), free_dataWithPadding);

    memcpy(encryptedRound, initVector, GOST28147_89_BLOCK_SIZE);
    blockSize = GOST28147_89_BLOCK_SIZE;

    for (i = 0; i < dataWithPaddingSize / GOST28147_89_BLOCK_SIZE; ++i) {
        /*************************************************************************
         * Усложнение ключа по RFC 4357 через каждые 1024 байт открытого текста   *
         * (Совместимость с режимом КриптоПро CSP - CRYPT_MODE_CBCRFC4357)        *
         *************************************************************************/
        if (i % 128 == 0 && i != 0) {
            printf(" Key Meshing...\n  Key Meshing(set key)\n");
            rv = functionList->C_DecryptInit(session, &gost28147EncDecEcbMech, encKey);
            CHECK_AND_LOG("   C_DecryptInit", rv == CKR_OK, rvToStr(rv), free_encrypted);
            rv = functionList->C_Decrypt(session, constForKeyMeshing, GOST_28147_KEY_SIZE, keyBlobForMeshing, &keySize);
            CHECK_AND_LOG("   C_Decrypt", rv == CKR_OK, rvToStr(rv), free_encrypted);
            rv = functionList->C_SetAttributeValue(session, encKey, &keyValueAttr, 1);
            CHECK_AND_LOG("   C_SetAttributeValue", rv == CKR_OK, rvToStr(rv), free_encrypted);

            /*************************************************************************
             * Для алгоритма ГОСТ 28147 механизм усложнения ключа предусматривает     *
             * преобразование синхропосылки через каждый килобайт данных.             *
             *                                                                        *
             * Для совместимости с КриптоПро CSP при использовании режима             *
             * CRYPT_MODE_CBC закомментировать следующие 5 строк кода                 *
             * (выбранный режим не осуществляет это преобразование)                   *
             *************************************************************************/
            printf("  Key Meshing(set IV)\n");
            rv = functionList->C_EncryptInit(session, &gost28147EncDecEcbMech, encKey);
            CHECK_AND_LOG("   C_EncryptInit", rv == CKR_OK, rvToStr(rv), free_encrypted);
            rv = functionList->C_Encrypt(session, encryptedRound, GOST28147_89_BLOCK_SIZE, encryptedRound, &blockSize);
            CHECK_AND_LOG("   C_Encrypt", rv == CKR_OK, rvToStr(rv), free_encrypted);
        }

        Xor(encryptedRound, dataWithPadding + i * GOST28147_89_BLOCK_SIZE, GOST28147_89_BLOCK_SIZE, xorBlock);

        /*************************************************************************
         * Инициализировать операцию шифрования                                   *
         *************************************************************************/
        rv = functionList->C_EncryptInit(session, &gost28147EncDecEcbMech, encKey);
        CHECK_AND_LOG(" C_EncryptInit", rv == CKR_OK, rvToStr(rv), free_encrypted);
        /*************************************************************************
         * Зашифровать блок данных                                                *
         *************************************************************************/
        rv = functionList->C_Encrypt(session, xorBlock, GOST28147_89_BLOCK_SIZE, encryptedRound, &blockSize);
        CHECK_AND_LOG(" C_Encrypt", rv == CKR_OK, rvToStr(rv), free_encrypted);

        memcpy(encrypted + i * GOST28147_89_BLOCK_SIZE, encryptedRound, GOST28147_89_BLOCK_SIZE);
    }
    /*************************************************************************
     * Установить секретный ключ в начальное состояние                        *
     *************************************************************************/
    printf(" Set default key value...\n");
    memcpy(keyBlobForMeshing, keyValue, GOST_28147_KEY_SIZE);
    rv = functionList->C_SetAttributeValue(session, encKey, &keyValueAttr, 1);
    CHECK_AND_LOG("  C_SetAttributeValue", rv == CKR_OK, rvToStr(rv), free_encrypted);

    /*************************************************************************
     * Распечатать буфер, содержащий зашифрованные данные                     *
     *************************************************************************/
    printf(" Encrypted buffer is:\n");
    printHex(encrypted, encryptedSize);

    printf("Encryption has been completed successfully.\n");

    /*************************************************************************
     * Расшифровать данные по алгоритму ГОСТ 28147-89                         *
     *************************************************************************/
    printf("\nDecrypting data...\n");

    /*************************************************************************
     * Получить массив хэндлов секретных ключей                               *
     *************************************************************************/
    printf(" Getting secret key...\n");
    r = findObjects(functionList, session, secKeyTemplate, arraysize(secKeyTemplate), &decKeys, &decKeysCount);
    CHECK(" findObjects", r == 0, free_encrypted);

    CHECK_AND_LOG(" Checking number of keys found", decKeysCount > 0, "No objects found\n", free_encrypted);

    decryptedSize = encryptedSize;

    decrypted = (CK_BYTE_PTR)malloc(decryptedSize * sizeof(CK_BYTE));
    CHECK(" Memory allocation for decrypted data", decrypted != NULL_PTR, free_decKeys);

    memcpy(encryptedRound, initVector, GOST28147_89_BLOCK_SIZE);

    for (i = 0; i < encryptedSize / GOST28147_89_BLOCK_SIZE; ++i) {
        if (i % 128 == 0 && i != 0) {
            printf(" Key Meshing...\n  Key Meshing(set key)\n");
            rv = functionList->C_DecryptInit(session, &gost28147EncDecEcbMech, encKey);
            CHECK_AND_LOG("   C_DecryptInit", rv == CKR_OK, rvToStr(rv), free_encrypted);
            rv = functionList->C_Decrypt(session, constForKeyMeshing, GOST_28147_KEY_SIZE, keyBlobForMeshing, &keySize);
            CHECK_AND_LOG("   C_Decrypt", rv == CKR_OK, rvToStr(rv), free_encrypted);
            rv = functionList->C_SetAttributeValue(session, encKey, &keyValueAttr, 1);
            CHECK_AND_LOG("   C_SetAttributeValue", rv == CKR_OK, rvToStr(rv), free_encrypted);

            /*************************************************************************
             * Для алгоритма ГОСТ 28147 механизм усложнения ключа предусматривает     *
             * преобразование синхропосылки через каждый килобайт данных.             *
             *                                                                        *
             * Для совместимости с КриптоПро CSP при использовании режима             *
             * CRYPT_MODE_CBC закомментировать следующие 5 строк кода                 *
             * (выбранный режим не осуществляет это преобразование)                   *
             *************************************************************************/
            printf("  Key Meshing(set IV)\n");
            rv = functionList->C_EncryptInit(session, &gost28147EncDecEcbMech, encKey);
            CHECK_AND_LOG("   C_EncryptInit", rv == CKR_OK, rvToStr(rv), free_encrypted);
            rv = functionList->C_Encrypt(session, encryptedRound, GOST28147_89_BLOCK_SIZE, encryptedRound, &blockSize);
            CHECK_AND_LOG("   C_Encrypt", rv == CKR_OK, rvToStr(rv), free_encrypted);
        }

        /*************************************************************************
         * Инициализировать операцию расшифрования                                *
         *************************************************************************/
        rv = functionList->C_DecryptInit(session, &gost28147EncDecEcbMech, decKeys[0]);
        CHECK_AND_LOG(" C_DecryptInit", rv == CKR_OK, rvToStr(rv), free_decrypted);
        /*************************************************************************
         * Расшифровать блок данных                                               *
         *************************************************************************/
        rv = functionList->C_Decrypt(session, encrypted + i * GOST28147_89_BLOCK_SIZE, GOST28147_89_BLOCK_SIZE,
                                     xorBlock, &blockSize);
        CHECK_AND_LOG(" C_Decrypt", rv == CKR_OK, rvToStr(rv), free_decrypted);

        Xor(encryptedRound, xorBlock, GOST28147_89_BLOCK_SIZE, decryptedRound);
        memcpy(decrypted + i * GOST28147_89_BLOCK_SIZE, decryptedRound, GOST28147_89_BLOCK_SIZE);
        memcpy(encryptedRound, encrypted + i * GOST28147_89_BLOCK_SIZE, GOST28147_89_BLOCK_SIZE);
    }

    /*************************************************************************
     * Установить секретный ключ в начальное состояние                        *
     *************************************************************************/
    printf(" Set default key value...\n");
    memcpy(keyBlobForMeshing, keyValue, GOST_28147_KEY_SIZE);
    rv = functionList->C_SetAttributeValue(session, encKey, &keyValueAttr, 1);
    CHECK_AND_LOG("  C_SetAttributeValue", rv == CKR_OK, rvToStr(rv), free_encrypted);

    /*************************************************************************
     * Удалить паддинг                                                        *
     *************************************************************************/
    decryptedSize = decryptedSize - decrypted[decryptedSize - 1];

    /*************************************************************************
     * Распечатать буфер, содержащий расшифрованные данные                    *
     *************************************************************************/
    printf(" Decrypted buffer is:\n");
    printHex(decrypted, decryptedSize);

    printf("Decryption has been completed successfully.\n");

    /*************************************************************************
     * Сравнить исходные данные с расшифрованными                             *
     *************************************************************************/
    r = (decryptedSize == sizeof(data)) && (memcmp(data, decrypted, decryptedSize) == 0);
    CHECK("Compare decrypted and plain texts", r, free_decrypted);

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    /*************************************************************************
     * Выполнить действия для завершения работы с библиотекой PKCS#11         *
     *************************************************************************/
    printf("\nFinalizing... \n");

    /*************************************************************************
     * Очистить память, выделенную под расшифрованные и зашифрованные данные, *
     * объекты ключей                                                         *
     *************************************************************************/
free_decrypted:
    free(decrypted);

free_decKeys:
    free(decKeys);

free_encrypted:
    free(encrypted);

free_dataWithPadding:
    free(dataWithPadding);

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, выделенную под слоты                                  *
     *************************************************************************/
free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}
