/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
 *------------------------------------------------------------------------*
 * Использование команд выработки ключа экспорта по алгоритму KEG         *
 *  - установление соединения с Рутокен в первом доступном слоте;         *
 *  - выполнение аутентификации Пользователя;                             *
 *  - генерация UKM;                                                      *
 *  - выработка ключа экспорта;                                           *
 *  - сброс прав доступа Пользователя на Рутокен                          *
 *  - закрытие соединения с Рутокен.                                      *
 *------------------------------------------------------------------------*
 * Пример использует объекты, созданные в памяти Рутокен примерами        *
 * CreateGOST34.10-2012-256 (256 битный ключ)                             *
 * CreateGOST34.10-2012-512 (512 битный ключ)                             *
 *------------------------------------------------------------------------*
 * Для выработки ключа экспорта по алгоритму KEG, используя 256 битный    *
 * ключ (из примера CreateGOST34.10-2012-256) надо использовать:          *
 *  - шаблон для поиска закрытого ключа отправителя (256 битный ключ);    *
 *  - буфер, содержащий значение шаблонного открытого ключа               *
 *    получателя (для 256 битного закрытого ключа);                       *
 *  - буфер, содержащий UKM при выработке из 256 битного ключа.           *
 *                                                                        *
 * Для выработки ключа экспорта по алгоритму KEG, используя 512 битный    *
 * ключ (из примера CreateGOST34.10-2012-512) надо использовать:          *
 *  - шаблон для поиска закрытого ключа отправителя (512 битный ключ);    *
 *  - буфер, содержащий значение шаблонного открытого ключа               *
 *    получателя (для 512 битного закрытого ключа);                       *
 *  - буфер, содержащий UKM при выработке из 512 битного ключа.           *
 *                                                                        *
 * Для выработки ключа экспорта по алгоритму KEG для дальнейшого его      *
 * использования в симметричном алгоритме блочного шифрования Кузнечик    *
 * надо использовать:                                                     *
 *  - шаблон для создания ключа экспорта для алгоритма Кузнечик.          *
 *                                                                        *
 * Для выработки ключа экспорта по алгоритму KEG для дальнейшого его      *
 * использования в симметричном алгоритме блочного шифрования Магма       *
 * надо использовать:                                                     *
 *  - шаблон для создания ключа экспорта для алгоритма Магма.             *
 *************************************************************************/

#include <Common.h>

/*************************************************************************
 * Шаблон для поиска закрытого ключа отправителя (256 битный ключ)        *
 *************************************************************************/
CK_ATTRIBUTE privateKeyTemplate[] = {
    { CKA_ID, keyPairIdGost2012_256_1, sizeof(keyPairIdGost2012_256_1) - 1 }, // ID пары
    { CKA_CLASS, &privateKeyObject, sizeof(privateKeyObject) },               // Класс - закрытый ключ
    { CKA_KEY_TYPE, &keyTypeGostR3410_2012_256,
      sizeof(keyTypeGostR3410_2012_256) } // Тип ключа - ГОСТ Р 34.10-2012(256)
};
/*************************************************************************
 * Шаблон для поиска закрытого ключа отправителя (512 битный ключ)        *
 *************************************************************************/
/*
   CK_ATTRIBUTE privateKeyTemplate[] =
   {
    { CKA_ID, keyPairIdGost2012_512_1, sizeof(keyPairIdGost2012_512_1) - 1 },        // ID пары
    { CKA_CLASS, &privateKeyObject, sizeof(privateKeyObject) },                      // Класс - закрытый ключ
    { CKA_KEY_TYPE, &keyTypeGostR3410_2012_512, sizeof(keyTypeGostR3410_2012_512) }  // Тип ключа - ГОСТ
   Р 34.10-2012(512)
   };
 */
/*****************************************************************************
 * Шаблон для создания ключа экспорта по алгоритму KEG для алгоритма Кузнечик *
 *****************************************************************************/
CK_ATTRIBUTE kegKeyTemplate[] = {
    { CKA_LABEL, &secretKeyKuznechikLabel, sizeof(secretKeyKuznechikLabel) - 1 }, // Метка ключа
    { CKA_CLASS, &secretKeyObject, sizeof(secretKeyObject) },              // Класс - секретный ключ
    { CKA_KEY_TYPE, &keyTypeKuznechikTwin, sizeof(keyTypeKuznechikTwin) }, // Тип ключа
    { CKA_TOKEN, &attributeFalse, sizeof(attributeFalse) }, // Ключ является объектом сессии
    { CKA_MODIFIABLE, &attributeTrue, sizeof(attributeTrue) }, // Ключ может быть изменен после создания
    { CKA_PRIVATE, &attributeTrue, sizeof(attributeTrue) }, // Ключ доступен только после аутентификации на токене
    { CKA_EXTRACTABLE, &attributeTrue, sizeof(attributeTrue) }, // Ключ может быть извлечен в зашифрованном виде
    { CKA_SENSITIVE, &attributeFalse, sizeof(attributeFalse) } // Ключ может быть извлечен в открытом виде
};
/*****************************************************************************
 * Шаблон для создания ключа экспорта по алгоритму KEG для алгоритма Магма    *
 *****************************************************************************/
/*
   CK_ATTRIBUTE kegKeyTemplate[] =
   {
    { CKA_LABEL, &secretKeyMagmaLabel, sizeof(secretKeyMagmaLabel) - 1 },          // Метка ключа
    { CKA_CLASS, &secretKeyObject, sizeof(secretKeyObject) },                      // Класс - секретный ключ
    { CKA_KEY_TYPE, &keyTypeMagmaTwin, sizeof(keyTypeMagmaTwin) },                 // Тип ключа
    { CKA_TOKEN, &attributeFalse, sizeof(attributeFalse) },                        // Ключ является объектом сессии
    { CKA_MODIFIABLE, &attributeTrue, sizeof(attributeTrue) },                     // Ключ может быть изменен после
   создания { CKA_PRIVATE, &attributeTrue, sizeof(attributeTrue) },                        // Ключ доступен только после
   аутентификации на токене { CKA_EXTRACTABLE, &attributeTrue, sizeof(attributeTrue) },                    // Ключ может
   быть извлечен в зашифрованном виде { CKA_SENSITIVE, &attributeFalse, sizeof(attributeFalse) }                     //
   Ключ может быть извлечен в открытом виде
   };
 */

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_SLOT_ID_PTR slots; // Массив идентификаторов слотов
    CK_ULONG slotCount;   // Количество идентификаторов слотов в массиве

    CK_MECHANISM_TYPE_PTR mechanisms; // Массив поддерживаемых механизмов
    CK_ULONG mechanismCount;          // Количество поддерживаемых механизмов

    CK_OBJECT_HANDLE kegKey; // Хэндл выработанного ключа экспорта

    CK_OBJECT_HANDLE_PTR privateKeys; // Массив хэндлов закрытых ключей
    CK_ULONG privateKeysCount;        // Количество хэндлов объектов в массиве

    CK_BYTE kegKeyValue[KEG_KEY_SIZE]; // Буфер, содержащий тело ключа экспорта

    CK_ATTRIBUTE valueAttr = { CKA_VALUE, NULL_PTR,
                               0 }; // Структура данных типа CK_ATTRIBUTE для хранения значения атрибута CKA_VALUE

    // Буфер, содержащий значение шаблонного открытого ключа получателя (для 256 битного закрытого ключа)
    CK_BYTE publicKeyValue[GOST_3410_12_256_KEY_SIZE] = {
        0x76, 0x25, 0x13, 0x0F, 0x19, 0x17, 0x3D, 0x3B, 0x24, 0xCC, 0xA7, 0xC7, 0x72, 0xB3, 0x5D, 0x83,
        0xB0, 0xBB, 0x42, 0xC9, 0x66, 0xD5, 0xC1, 0x5A, 0x0A, 0x9F, 0xD4, 0x24, 0xF0, 0x46, 0xB2, 0xCD,
        0x85, 0xDD, 0xC5, 0x73, 0xAE, 0x72, 0x8D, 0x6F, 0xC8, 0x9C, 0xE2, 0x5B, 0x89, 0x05, 0xE8, 0x9D,
        0x75, 0x93, 0xBF, 0xE9, 0x38, 0xC3, 0x43, 0x27, 0x09, 0x59, 0x7E, 0x7D, 0x51, 0xA8, 0x35, 0x53
    };

    // Буфер, содержащий UKM при выработке из 256 битного ключа
    CK_BYTE ukm[UKM_KEG_256_LENGTH];

    /*
       // Буфер, содержащий значение шаблонного открытого ключа получателя (для 512 битного закрытого ключа)
       CK_BYTE publicKeyValue[GOST_3410_12_512_KEY_SIZE] =
       {
        0x53, 0x71, 0xbf, 0x37, 0x50, 0xdf, 0xd9, 0xcd, 0xe5, 0xf2, 0xb0, 0xc2, 0xb4, 0x0c, 0x28, 0x93,
        0xc6, 0xd3, 0x67, 0x97, 0x19, 0x09, 0x8c, 0xab, 0x0e, 0x34, 0x5b, 0x9c, 0xea, 0xfe, 0xae, 0x06,
        0x77, 0x44, 0x9c, 0xab, 0x05, 0xa3, 0x16, 0x7e, 0x71, 0x80, 0x50, 0x96, 0xfe, 0xe9, 0xab, 0x35,
        0xcb, 0x2b, 0x40, 0x3c, 0x65, 0x08, 0x2a, 0x20, 0xf4, 0xf2, 0x8b, 0x73, 0x91, 0x3e, 0xc1, 0x8f,
        0xdd, 0xac, 0x25, 0x9c, 0x17, 0x6f, 0x9e, 0xb0, 0x42, 0x23, 0xfb, 0x25, 0x5a, 0xa3, 0x08, 0xf4,
        0x4c, 0xd3, 0x25, 0x5b, 0x97, 0xdc, 0xda, 0xcc, 0x11, 0x66, 0x7b, 0x62, 0x21, 0x59, 0x71, 0x0c,
        0x4c, 0x17, 0xec, 0xe8, 0xa0, 0x83, 0x1a, 0xc1, 0x4e, 0xab, 0xff, 0xfe, 0x76, 0x7c, 0x8b, 0x58,
        0xbd, 0xda, 0xa7, 0xcb, 0x44, 0xdc, 0x18, 0x7a, 0x8c, 0xda, 0x6f, 0x80, 0xe3, 0xa4, 0xe8, 0x17
       };

       // Буфер, содержащий UKM при выработке из 512 битного ключа
       CK_BYTE ukm[UKM_KEG_512_LENGTH];
     */

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
    int r;    // Код возврата для функций, возвращающих int

    CK_ULONG i; // Вспомогательная переменная-счетчик в циклах

    int isGostKegDeriveSupported = 0; // Флаги для проверки поддержки токеном CKM_GOST_KEG

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
    printf(" Slots available: %d\n", (int)slotCount);

    /*************************************************************************
     * Получить список поддерживаемых токеном механизмов                      *
     *************************************************************************/
    rv = functionList->C_GetMechanismList(slots[0], NULL_PTR, &mechanismCount);
    CHECK_AND_LOG(" C_GetMechanismList (number of mechanisms)", rv == CKR_OK, rvToStr(rv), free_slots);

    CHECK_AND_LOG(" Checking mechanisms available", mechanismCount > 0, " No mechanisms available", free_slots);

    mechanisms = (CK_MECHANISM_TYPE_PTR)malloc(mechanismCount * sizeof(CK_MECHANISM_TYPE));
    CHECK(" Memory allocation for mechanisms", mechanisms != NULL_PTR, free_slots);

    rv = functionList->C_GetMechanismList(slots[0], mechanisms, &mechanismCount);
    CHECK_AND_LOG(" C_GetMechanismList", rv == CKR_OK, rvToStr(rv), free_mechanisms);

    /*************************************************************************
     * Определение поддерживаемых токеном механизмов                          *
     *************************************************************************/
    for (i = 0; i < mechanismCount; ++i) {
        if (CKM_GOST_KEG == mechanisms[i]) {
            isGostKegDeriveSupported = 1;
            break;
        }
    }

    CHECK_AND_LOG(" Checking CKM_GOST_KEG support", isGostKegDeriveSupported, "CKM_GOST_KEG isn`t supported!",
                  free_mechanisms);

    /*************************************************************************
     * Открыть RW сессию в первом доступном слоте                             *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_mechanisms);

    /*************************************************************************
     * Выполнить аутентификацию Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Initialization has been completed successfully.\n");

    /*************************************************************************
     * Выработать UKM                                                         *
     *************************************************************************/
    rv = functionList->C_GenerateRandom(session, ukm, arraysize(ukm));
    CHECK_AND_LOG(" C_GenerateRandom", rv == CKR_OK, rvToStr(rv), logout);

    /*************************************************************************
     * Получить массив хэндлов закрытых ключей                                *
     *************************************************************************/
    printf(" Getting private key...\n");

    r = findObjects(functionList, session, privateKeyTemplate, arraysize(privateKeyTemplate), &privateKeys,
                    &privateKeysCount);
    CHECK(" findObjects", r == 0, logout);

    CHECK_AND_LOG(" Checking number of keys found", privateKeysCount > 0, " No objects found", logout);

    /*************************************************************************
     * Поместить в структуру типа CK_MECHANISM параметры, необходимые         *
     * для выработки ключа экспорта по алгоритму KEG                          *
     *************************************************************************/
    CK_ECDH1_DERIVE_PARAMS params = { CKD_NULL, arraysize(ukm), ukm, arraysize(publicKeyValue), publicKeyValue };
    gostKegDerivationMech.pParameter = &params;
    gostKegDerivationMech.ulParameterLen = sizeof(params);

    /*************************************************************************
     * Выработать ключ экспорта по алгоритму KEG                              *
     *************************************************************************/
    rv = functionList->C_DeriveKey(session, &gostKegDerivationMech, privateKeys[0], kegKeyTemplate,
                                   arraysize(kegKeyTemplate), &kegKey);

    CHECK_AND_LOG(" C_DeriveKey", rv == CKR_OK, rvToStr(rv), free_private_keys);

    /*************************************************************************
     * Получить и распечатать значение выработанного по алгоритму KEG ключа   *
     *************************************************************************/
    valueAttr.pValue = kegKeyValue;
    valueAttr.ulValueLen = arraysize(kegKeyValue);

    rv = functionList->C_GetAttributeValue(session, kegKey, &valueAttr, 1);
    CHECK_AND_LOG(" C_GetAttributeValue", rv == CKR_OK, rvToStr(rv), destroy_derived_key);

    printf(" Derived key value:\n");
    printHex(kegKeyValue, arraysize(kegKeyValue));

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    printf("Key has been derived successfully.\n");

    /*************************************************************************
     * Выполнить действия для завершения работы с библиотекой PKCS#11         *
     *************************************************************************/
    printf("\nFinalizing... \n");

    /*************************************************************************
     * Уничтожить выработанный по алгоритму KEG ключ                          *
     *************************************************************************/
destroy_derived_key:
    rv = functionList->C_DestroyObject(session, kegKey);
    CHECK_RELEASE_AND_LOG(" C_DestroyObject", rv == CKR_OK, rvToStr(rv), errorCode);
    kegKey = CK_INVALID_HANDLE;

    /*************************************************************************
     * Очистить память, выделенную под массив хэндлов закрытых ключей         *
     *************************************************************************/
free_private_keys:
    free(privateKeys);

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, выделенную под слоты и механизмы                      *
     *************************************************************************/
free_mechanisms:
    free(mechanisms);

free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}
