/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
 *------------------------------------------------------------------------*
 * Использование команд выработки HMAC по алгоритму ГОСТ Р 34.11-94       *
 * на ключе ГОСТ 28147-89:                                                *
 *  - установление соединения с Рутокен в первом доступном слоте;         *
 *  - выполнение аутентификации Пользователя;                             *
 *  - создание демонстрационного секретного ключа ГОСТ 28147-89           *
 *    в оперативной памяти;                                               *
 *  - выработка HMAC по алгоритму ГОСТ Р 34.11-94 для сообщения           *
 *    на демонстрационном ключе;                                          *
 *  - проверка выработки HMAC на демонстрационном ключе;                  *
 *  - удаление созданного ключа,сброс прав доступа Пользователя           *
 *    на Рутокен и закрытие соединения с Рутокен.                         *
 *************************************************************************/

#include <Common.h>

/*************************************************************************
 * Данные для выработки кода аутентификации сообщения на основе HMAC      *
 *************************************************************************/
CK_BYTE data[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                   0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07 };

/*************************************************************************
 * Шаблон для создания симметричного ключа ГОСТ 28147-89                  *
 * в оперативной памяти                                                   *
 *************************************************************************/
CK_ATTRIBUTE secretKeyTemplate[] = {
    { CKA_CLASS, &secretKeyObject, sizeof(secretKeyObject) },      // Класс - секретный ключ
    { CKA_LABEL, &secretKeyLabel, sizeof(secretKeyLabel) - 1 },    // Метка ключа
    { CKA_ID, &secretKeyId, sizeof(secretKeyId) - 1 },             // Идентификатор ключа
    { CKA_KEY_TYPE, &keyTypeGost28147, sizeof(keyTypeGost28147) }, // Тип ключа - ГОСТ 28147-89
    { CKA_ENCRYPT, &attributeTrue, sizeof(attributeTrue) }, // Ключ предназначен для зашифрования
    { CKA_DECRYPT, &attributeTrue, sizeof(attributeTrue) }, // Ключ предназначен для расшифрования
    { CKA_TOKEN, &attributeFalse, sizeof(attributeFalse) }, // Ключ не является объектом токена
    { CKA_PRIVATE, &attributeTrue, sizeof(attributeTrue) }, // Ключ недоступен без аутентификации на токене
    { CKA_GOST28147_PARAMS, parametersGost28147, sizeof(parametersGost28147) } // Параметры алгоритма из стандарта
};

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_SLOT_ID_PTR slots; // Массив идентификаторов слотов
    CK_ULONG slotCount;   // Количество идентификаторов слотов в массиве

    CK_MECHANISM_TYPE_PTR mechanisms; // Массив поддерживаемых механизмов
    CK_ULONG mechanismCount;          // Количество поддерживаемых механизмов

    CK_OBJECT_HANDLE signverKey; // Хэндл ключа для HMAC

    CK_BYTE_PTR signature; // Указатель на буфер, содержащий результат HMAC от данных
    CK_ULONG signatureSize; // Размер буфера, содержащего результат HMAC от данных, в байтах

    CK_ULONG i; // Вспомогательная переменная-счетчик в циклах

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11

    int isGost3411HMACSupported = 0; // Флаги для проверки поддержки токеном CKM_GOSTR3411_НMAC
    int isGost28147_89Supported = 0; // Флаг для проверки поддержки токеном CKM_GOST28147_KEY_GEN

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
    printf(" Slots available: %d\n", (int)slotCount);

    /*************************************************************************
     * Получить список поддерживаемых токеном механизмов                      *
     *************************************************************************/
    rv = functionList->C_GetMechanismList(slots[0], NULL_PTR, &mechanismCount);
    CHECK_AND_LOG(" C_GetMechanismList (number of mechanisms)", rv == CKR_OK, rvToStr(rv), free_slots);

    CHECK_AND_LOG(" Checking mechanisms available", mechanismCount > 0, " No mechanisms available", free_slots);

    mechanisms = (CK_MECHANISM_TYPE_PTR)malloc(mechanismCount * sizeof(CK_MECHANISM_TYPE));
    CHECK(" Memory allocation for mechanisms", mechanisms != NULL_PTR, free_slots);

    rv = functionList->C_GetMechanismList(slots[0], mechanisms, &mechanismCount);
    CHECK_AND_LOG(" C_GetMechanismList", rv == CKR_OK, rvToStr(rv), free_mechanisms);

    /*************************************************************************
     * Определение поддерживаемых токеном механизмов                          *
     *************************************************************************/
    for (i = 0; i < mechanismCount; ++i) {
        if (mechanisms[i] == CKM_GOSTR3411_HMAC) {
            isGost3411HMACSupported = 1;
            continue;
        }
        if (mechanisms[i] == CKM_GOST28147_KEY_GEN) {
            isGost28147_89Supported = 1;
            continue;
        }
    }

    CHECK_AND_LOG(" Checking CKM_GOSTR3411_HMAC support", isGost3411HMACSupported,
                  "CKM_GOSTR3411_HMAC isn`t supported!\n", free_mechanisms);
    CHECK_AND_LOG(" Checking CKM_GOST28147_KEY_GEN support", isGost28147_89Supported,
                  "CKM_GOST28147_KEY_GEN isn`t supported!\n", free_mechanisms);

    /*************************************************************************
     * Открыть сессию в первом доступном слоте                                *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_mechanisms);

    /*************************************************************************
     * Выполнить аутентификацию Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Initialization has been completed successfully.\n");

    /*************************************************************************
     * Сгенерировать секретный ключ ГОСТ 28147-89                             *
     *************************************************************************/
    printf("\nGenerating GOST 28147-89 secret key...\n");

    rv = functionList->C_GenerateKey(session, &gost28147KeyGenMech, secretKeyTemplate, arraysize(secretKeyTemplate),
                                     &signverKey);
    CHECK_AND_LOG(" C_GenerateKey", rv == CKR_OK, rvToStr(rv), logout);

    /*************************************************************************
     * Сформировать HMAC по алгоритму на основе ГОСТ Р 34.11-94               *
     *************************************************************************/
    printf("\nSigning data...\n");

    /*************************************************************************
     * Инициализировать операцию HMAC                                         *
     *************************************************************************/
    rv = functionList->C_SignInit(session, &gostR3411HmacMech, signverKey);
    CHECK_AND_LOG("  C_SignInit", rv == CKR_OK, rvToStr(rv), delete_signverkey);

    /*************************************************************************
     * Определить размер HMAC                                                 *
     *************************************************************************/
    rv = functionList->C_Sign(session, data, sizeof(data), NULL_PTR, &signatureSize);
    CHECK_AND_LOG("  C_Sign (get size)", rv == CKR_OK, rvToStr(rv), delete_signverkey);

    /*************************************************************************
     * Выработать HMAC                                                        *
     *************************************************************************/
    signature = (CK_BYTE*)malloc(signatureSize * sizeof(CK_BYTE));
    CHECK("  Memory allocation for HMAC", signature != NULL, delete_signverkey);

    rv = functionList->C_Sign(session, data, sizeof(data), signature, &signatureSize);
    CHECK_AND_LOG("  C_Sign (signing)", rv == CKR_OK, rvToStr(rv), free_signature);

    /*************************************************************************
     * Распечатать буфер, содержащий HMAC                                     *
     *************************************************************************/
    printf("  HMAC buffer is: \n");
    printHex(signature, signatureSize);
    printf("HMAC has been computed successfully.\n");

    /*************************************************************************
     * Выполнить проверку HMAC                                                *
     *************************************************************************/
    printf("\nVerifying HMAC...\n");

    /*************************************************************************
     * Инициализировать операцию проверки HMAC                                *
     *************************************************************************/
    rv = functionList->C_VerifyInit(session, &gostR3411HmacMech, signverKey);
    CHECK_AND_LOG(" C_VerifyInit", rv == CKR_OK, rvToStr(rv), free_signature);

    /*************************************************************************
     * Проверить HMAC                                                         *
     *************************************************************************/
    rv = functionList->C_Verify(session, data, sizeof(data), signature, signatureSize);
    CHECK_AND_LOG("  C_Verify", rv == CKR_OK, rvToStr(rv), free_signature);

    printf("Verifying has been completed successfully.\n");

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    /*************************************************************************
     * Выполнить действия для завершения работы с библиотекой PKCS#11         *
     *************************************************************************/
    printf("\nFinalizing... \n");

    /*************************************************************************
     * Очистить память, выделенную под вычисленный HMAC                       *
     *************************************************************************/
free_signature:
    free(signature);

    /*************************************************************************
     * Удалить созданный ключ                                                 *
     *************************************************************************/
delete_signverkey:
    rv = functionList->C_DestroyObject(session, signverKey);
    CHECK_RELEASE_AND_LOG(" C_DestroyObject", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, выделенную под механизмы                              *
     *************************************************************************/
free_mechanisms:
    free(mechanisms);

    /*************************************************************************
     * Очистить память, выделенную под слоты                                  *
     *************************************************************************/
free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}
