/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
 *------------------------------------------------------------------------*
 * Использование команд вычисления/проверки ЭЦП на ключах RSA:            *
 *  - установление соединения с Рутокен в первом доступном слоте;         *
 *  - выполнение аутентификации Пользователя на Рутокен;                  *
 *  - подпись сообщения на демонстрационном ключе;                        *
 *  - проверка подписи на демонстрационном ключе;                         *
 *  - сброс прав доступа Пользователя на Рутокен и закрытие соединения    *
 *    с Рутокен.                                                          *
 *------------------------------------------------------------------------*
 * Пример использует объекты, созданные в памяти Рутокен примером         *
 * CreateRSA.                                                             *
 *************************************************************************/

#include <Common.h>

/*************************************************************************
 * Данные для подписи                                                     *
 *************************************************************************/
CK_BYTE data[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                   0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07 };

/*************************************************************************
 * Шаблон для поиска открытого ключа RSA                                  *
 *************************************************************************/
CK_ATTRIBUTE publicKeyTemplate[] = {
    { CKA_ID, &keyPairIdRsa, sizeof(keyPairIdRsa) - 1 },     // ID пары
    { CKA_CLASS, &publicKeyObject, sizeof(publicKeyObject) } // Класс - открытый ключ
};

/*************************************************************************
 * Шаблон для поиска закрытого ключа RSA                                  *
 *************************************************************************/
CK_ATTRIBUTE privateKeyTemplate[] = {
    { CKA_ID, &keyPairIdRsa, sizeof(keyPairIdRsa) - 1 },       // ID пары
    { CKA_CLASS, &privateKeyObject, sizeof(privateKeyObject) } // Класс - закрытый ключ
};

/*************************************************************************
 * ASN.1 тег типа OCTET STRING [X.690-02, Section 8.21.5.4 (EXAMPLE)]     *
 *************************************************************************/
#define ASN1_OCTET_STRING_TAG 0x04

/*************************************************************************
 * ASN.1 тег типа NULL [X.690-02, Section 8.8.2 (EXAMPLE)]                *
 *************************************************************************/
#define ASN1_NULL_TAG 0x05

/*************************************************************************
 * ASN.1 тег типа OID [X.690-02, Section 8.19.5 (EXAMPLE)]                *
 *************************************************************************/
#define ASN1_OID_TAG 0x06

/*************************************************************************
 * ASN.1 тег типа SEQUENCE [X.690-02, Section 8.9.3 (EXAMPLE)]            *
 *************************************************************************/
#define ASN1_SEQUENCE_TAG 0x30

/*************************************************************************
 * Шаблон объекта DigestInfo для алгоритма SHA-1                          *
 *************************************************************************/
CK_BYTE sha1DigestInfoTemplate[] = { ASN1_SEQUENCE_TAG, 33,
                                     // Поле digestAlgorithm [X.509-88, Section 7.2]
                                     ASN1_SEQUENCE_TAG, 9,
                                     // Поле algorithm. Заполняется по правилам [X.690-02, Section 8.19]
                                     // Для SHA-1: 1.3.14.3.2.26 [RFC3447, Appendix B.1]
                                     ASN1_OID_TAG, 5, 0x2B, 0x0E, 0x03, 0x02, 0x1A,
                                     // Поле parameters [RFC3447, Appendix B.1]
                                     ASN1_NULL_TAG, 0,
                                     // Поле digest (необходимо заполнить хэш-кодом)
                                     ASN1_OCTET_STRING_TAG, 20 };

/*************************************************************************
 * Шаблон объекта DigestInfo для алгоритма SHA-224                        *
 *************************************************************************/
CK_BYTE sha224DigestInfoTemplate[] = { ASN1_SEQUENCE_TAG, 45,
                                       // Поле digestAlgorithm [X.509-88, Section 7.2]
                                       ASN1_SEQUENCE_TAG, 13,
                                       // Поле algorithm. Заполняется по правилам [X.690-02, Section 8.19]
                                       // Для SHA-384: 2.16.840.1.101.3.4.2.4 [RFC3874, Section 4]
                                       ASN1_OID_TAG, 9, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x04,
                                       // Поле parameters [RFC3447, Appendix B.1]
                                       ASN1_NULL_TAG, 0,
                                       // Поле digest (необходимо заполнить хэш-кодом)
                                       ASN1_OCTET_STRING_TAG, 28 };

/*************************************************************************
 * Шаблон объекта DigestInfo для алгоритма SHA-256                        *
 *************************************************************************/
CK_BYTE sha256DigestInfoTemplate[] = { ASN1_SEQUENCE_TAG, 49,
                                       // Поле digestAlgorithm [X.509-88, Section 7.2]
                                       ASN1_SEQUENCE_TAG, 13,
                                       // Поле algorithm. Заполняется по правилам [X.690-02, Section 8.19]
                                       // Для SHA-256: 2.16.840.1.101.3.4.2.1 [RFC3447, Appendix B.1]
                                       ASN1_OID_TAG, 9, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01,
                                       // Поле parameters [RFC3447, Appendix B.1]
                                       ASN1_NULL_TAG, 0,
                                       // Поле digest (необходимо заполнить хэш-кодом)
                                       ASN1_OCTET_STRING_TAG, 32 };

/*************************************************************************
 * Шаблон объекта DigestInfo для алгоритма SHA-384                        *
 *************************************************************************/
CK_BYTE sha384DigestInfoTemplate[] = { ASN1_SEQUENCE_TAG, 65,
                                       // Поле digestAlgorithm [X.509-88, Section 7.2]
                                       ASN1_SEQUENCE_TAG, 13,
                                       // Поле algorithm. Заполняется по правилам [X.690-02, Section 8.19]
                                       // Для SHA-384: 2.16.840.1.101.3.4.2.2 [RFC3447, Appendix B.1]
                                       ASN1_OID_TAG, 9, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x02,
                                       // Поле parameters [RFC3447, Appendix B.1]
                                       ASN1_NULL_TAG, 0,
                                       // Поле digest (необходимо заполнить хэш-кодом)
                                       ASN1_OCTET_STRING_TAG, 48 };

/*************************************************************************
 * Шаблон объекта DigestInfo для алгоритма SHA-512                        *
 *************************************************************************/
CK_BYTE sha512DigestInfoTemplate[] = { ASN1_SEQUENCE_TAG, 81,
                                       // Поле digestAlgorithm [X.509-88, Section 7.2]
                                       ASN1_SEQUENCE_TAG, 13,
                                       // Поле algorithm. Заполняется по правилам [X.690-02, Section 8.19]
                                       // Для SHA-384: 2.16.840.1.101.3.4.2.3 [RFC3447, Appendix B.1]
                                       ASN1_OID_TAG, 9, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x03,
                                       // Поле parameters [RFC3447, Appendix B.1]
                                       ASN1_NULL_TAG, 0,
                                       // Поле digest (необходимо заполнить хэш-кодом)
                                       ASN1_OCTET_STRING_TAG, 64 };

/*************************************************************************
 * Функция, кодирующая хэш-код в ASN.1 объект типа                        *
 * DigestInfo [RFC2313, Section 10.1.2].									 *
 *------------------------------------------------------------------------*
 * - hashAlgorithm     - код механизма, использованного при хэшировании.	 *
 *                       Допустимые значения: CMK_SHA_1, CMK_SHA224,	     *
 *                       CKM_SHA256, CKM_SHA384, CKM_SHA512;              *
 * - hash              - хэш-код, полученный указанным механизмом;        *
 * - digestInfoSize    - место в памяти, куда будет помещен размер        *
 *						объекта DigestInfo, в байтах;					 *
 * - digestInfoPtr     - место в памяти, куда будет помещен указатель на  *
 *                       объект DigestInfo.                               *
 *************************************************************************/
void EncodeToDigestInfo(CK_ULONG hashAlgorithm, CK_BYTE_PTR hash, CK_ULONG_PTR digestInfoSize,
                        CK_BYTE_PTR_PTR digestInfoPtr) {
    CK_BYTE_PTR digestInfoTemplate;
    CK_ULONG digestInfoTemplateSize;
    CK_ULONG hashSize;

    switch (hashAlgorithm) {
    case CKM_SHA_1:
        digestInfoTemplate = sha1DigestInfoTemplate;
        digestInfoTemplateSize = sizeof(sha1DigestInfoTemplate);
        break;
    case CKM_SHA224:
        digestInfoTemplate = sha224DigestInfoTemplate;
        digestInfoTemplateSize = sizeof(sha224DigestInfoTemplate);
        break;
    case CKM_SHA256:
        digestInfoTemplate = sha256DigestInfoTemplate;
        digestInfoTemplateSize = sizeof(sha256DigestInfoTemplate);
        break;
    case CKM_SHA384:
        digestInfoTemplate = sha384DigestInfoTemplate;
        digestInfoTemplateSize = sizeof(sha384DigestInfoTemplate);
        break;
    case CKM_SHA512:
        digestInfoTemplate = sha512DigestInfoTemplate;
        digestInfoTemplateSize = sizeof(sha512DigestInfoTemplate);
        break;
    default:
        *digestInfoSize = 0;
        return;
    }
    hashSize = digestInfoTemplate[digestInfoTemplateSize - 1];

    *digestInfoSize = digestInfoTemplateSize + hashSize;
    *digestInfoPtr = (CK_BYTE_PTR)malloc(*digestInfoSize * sizeof(CK_BYTE));
    if (!*digestInfoPtr) {
        *digestInfoSize = 0;
        return;
    }
    memcpy(*digestInfoPtr, digestInfoTemplate, digestInfoTemplateSize);
    memcpy(*digestInfoPtr + digestInfoTemplateSize, hash, hashSize);
}

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_SLOT_ID_PTR slots; // Массив идентификаторов слотов
    CK_ULONG slotCount;   // Количество идентификаторов слотов в массиве

    CK_MECHANISM_TYPE_PTR mechanisms; // Массив поддерживаемых механизмов
    CK_ULONG mechanismCount;          // Количество поддерживаемых механизмов

    CK_OBJECT_HANDLE_PTR objects1; // Массив хэндлов объектов, соответствующих критериям поиска
    CK_ULONG objectCount1; // Количество хэндлов объектов в массиве

    CK_OBJECT_HANDLE_PTR objects2; // Массив хэндлов объектов, соответствующих критериям поиска
    CK_ULONG objectCount2; // Количество хэндлов объектов в массиве

    CK_BYTE_PTR signature; // Указатель на буфер, содержащий цифровую подпись для данных
    CK_ULONG signatureSize; // Размер буфера, содержащего цифровую подпись для данных, в байтах

    CK_BYTE_PTR hash1; // Указатель на временный буфер для хэш-кода от данных
    CK_BYTE_PTR hash2; // Указатель на временный буфер для хэш-кода от данных

    CK_ULONG hashSize1; // Размер временного буфера для хэш-кода от данных, в байтах
    CK_ULONG hashSize2; // Размер временного буфера для хэш-кода от данных, в байтах

    CK_BYTE_PTR digestInfo1; // Указатель на временный буфер для объекта DigestInfo от хэш-кода
    CK_BYTE_PTR digestInfo2; // Указатель на временный буфер для объекта DigestInfo от хэш-кода

    CK_ULONG digestInfoSize1; // Размер временного буфера для объекта DigestInfo от хэш-кода, в байтах
    CK_ULONG digestInfoSize2; // Размер временного буфера для объекта DigestInfo от хэш-кода, в байтах

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
    int r;    // Код возврата для функций, возвращающих int

    CK_ULONG i; // Вспомогательная переменная-счетчик в циклах

    int isRsaSupported = 0;    // Флаги для проверки поддержки токеном CKM_RSA_PKCS
    int isSha256Supported = 0; // Флаги для проверки поддержки токеном CKM_SHA256

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
    printf(" Slots available: %d\n", (int)slotCount);

    /*************************************************************************
     * Получить список поддерживаемых токеном механизмов                      *
     *************************************************************************/
    rv = functionList->C_GetMechanismList(slots[0], NULL_PTR, &mechanismCount);
    CHECK_AND_LOG(" C_GetMechanismList (number of mechanisms)", rv == CKR_OK, rvToStr(rv), free_slots);

    CHECK_AND_LOG(" Checking mechanisms available", mechanismCount > 0, " No mechanisms available", free_slots);

    mechanisms = (CK_MECHANISM_TYPE_PTR)malloc(mechanismCount * sizeof(CK_MECHANISM_TYPE));
    CHECK(" Memory allocation for mechanisms", mechanisms != NULL_PTR, free_slots);

    rv = functionList->C_GetMechanismList(slots[0], mechanisms, &mechanismCount);
    CHECK_AND_LOG(" C_GetMechanismList", rv == CKR_OK, rvToStr(rv), free_mechanisms);

    /*************************************************************************
     * Определение поддерживаемых токеном механизмов                          *
     *************************************************************************/
    for (i = 0; i < mechanismCount; ++i) {
        if (mechanisms[i] == CKM_RSA_PKCS) {
            isRsaSupported = 1;
            break;
        }
    }

    for (i = 0; i < mechanismCount; ++i) {
        if (mechanisms[i] == CKM_SHA256) {
            isSha256Supported = 1;
            break;
        }
    }

    CHECK_AND_LOG(" Checking CKM_RSA_PKCS support", isRsaSupported, "CKM_RSA_PKCS isn`t supported!\n", free_mechanisms);
    CHECK_AND_LOG(" Checking CKM_SHA256 support", isSha256Supported, "CKM_SHA256 isn`t supported!\n", free_mechanisms);

    /*************************************************************************
     * Открыть сессию в первом доступном слоте                                *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_mechanisms);

    /*************************************************************************
     * Выполнить аутентификацию Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Initialization has been completed successfully.\n");

    /*************************************************************************
     * Сформировать цифровую подпись данных по алгоритму RSA                  *
     *************************************************************************/
    printf("\nSigning data...\n");

    /*************************************************************************
     * Получить массив хэндлов закрытых ключей                                *
     *************************************************************************/
    printf(" Getting signing key...\n");

    r = findObjects(functionList, session, privateKeyTemplate, arraysize(privateKeyTemplate), &objects1, &objectCount1);
    CHECK(" findObjects", r == 0, logout);

    CHECK_AND_LOG(" Checking number of keys found", objectCount1 > 0, "No objects found\n", logout);

    /*************************************************************************
     * Вычислить хэш-код данных                                               *
     *************************************************************************/
    printf(" Hashing data...\n");

    /*************************************************************************
     * Инициализировать хэш-функцию                                           *
     *************************************************************************/
    rv = functionList->C_DigestInit(session, &sha256Mech);
    CHECK_AND_LOG("  C_DigestInit", rv == CKR_OK, rvToStr(rv), free_objects1);

    /*************************************************************************
     * Определить размер хэш-кода                                             *
     *************************************************************************/
    rv = functionList->C_Digest(session, data, sizeof(data), NULL_PTR, &hashSize1);
    CHECK_AND_LOG("  C_Digest(get size)", rv == CKR_OK, rvToStr(rv), free_objects1);

    /*************************************************************************
     * Вычислить хэш-код данных                                               *
     *************************************************************************/

    hash1 = (CK_BYTE*)malloc(hashSize1 * sizeof(CK_BYTE));
    CHECK("  Memory allocation for hash", hash1 != NULL, free_objects1);

    rv = functionList->C_Digest(session, data, sizeof(data), hash1, &hashSize1);
    CHECK_AND_LOG("  C_Digest (get hash)", rv == CKR_OK, rvToStr(rv), free_hash1);

    /*************************************************************************
     * Распечатать буфер, содержащий хэш-код                                  *
     *************************************************************************/
    printf("  Hashed buffer is: \n");
    printHex(hash1, hashSize1);
    printf(" Hashing has been completed.\n");

    /*************************************************************************
     * Закодировать хэш-код в объект DigestInfo                               *
     *************************************************************************/
    printf(" Encoding hash...\n");
    EncodeToDigestInfo(sha256Mech.mechanism, hash1, &digestInfoSize1, &digestInfo1);
    CHECK("  DigestInfo encoding", digestInfo1 != NULL, free_hash1);

    /*************************************************************************
     * Распечатать буфер, содержащий объект DigestInfo                        *
     *************************************************************************/
    printf("  Encoded hash is: \n");
    printHex(digestInfo1, digestInfoSize1);
    printf(" Encoding has been completed.\n");

    /*************************************************************************
     * Сформировать цифровую подпись данных по алгоритму RSA                  *
     *************************************************************************/
    printf(" Signing data...\n");
    /*************************************************************************
     * Инициализировать операцию подписи данных                               *
     *************************************************************************/
    rv = functionList->C_SignInit(session, &rsaSigVerMech, objects1[0]);
    CHECK_AND_LOG("  C_SignInit", rv == CKR_OK, rvToStr(rv), free_digestInfo1);

    /*************************************************************************
     * Определить размер данных подписи                                       *
     *************************************************************************/
    rv = functionList->C_Sign(session, digestInfo1, digestInfoSize1, NULL_PTR, &signatureSize);
    CHECK_AND_LOG("  C_Sign (get size)", rv == CKR_OK, rvToStr(rv), free_digestInfo1);

    /*************************************************************************
     * Подписать данные                                                      *
     ************************************************************************/
    signature = (CK_BYTE*)malloc(signatureSize * sizeof(CK_BYTE));
    CHECK("  Memory allocation for signature", signature != NULL, free_digestInfo1);

    rv = functionList->C_Sign(session, digestInfo1, digestInfoSize1, signature, &signatureSize);
    CHECK_AND_LOG("  C_Sign (signing)", rv == CKR_OK, rvToStr(rv), free_signature);

    /*************************************************************************
     * Распечатать буфер, содержащий подпись                                  *
     *************************************************************************/
    printf("  Signature buffer is: \n");
    printHex(signature, signatureSize);
    printf("Data has been signed successfully.\n");

    /*************************************************************************
     * Выполнить проверку подписи по алгоритму RSA                            *
     *************************************************************************/
    printf("\nVerifying signature...\n");

    /*************************************************************************
     * Получить массив хэндлов открытых ключей                                *
     *************************************************************************/
    printf(" Getting key to verify...\n");
    r = findObjects(functionList, session, publicKeyTemplate, arraysize(publicKeyTemplate), &objects2, &objectCount2);
    CHECK(" findObjects", r == 0, free_signature);

    CHECK_AND_LOG(" Checking number of keys found", objectCount2 > 0, "No objects found\n", free_signature);

    /*************************************************************************
     * Вычислить хэш-код от данных                                            *
     *************************************************************************/
    printf(" Hashing data...\n");

    /*************************************************************************
     * Инициализировать хэш-функцию                                           *
     *************************************************************************/
    rv = functionList->C_DigestInit(session, &sha256Mech);
    CHECK_AND_LOG("  C_DigestInit", rv == CKR_OK, rvToStr(rv), free_objects2);

    /*************************************************************************
     * Определить размер хэш-кода                                             *
     *************************************************************************/
    rv = functionList->C_Digest(session, data, sizeof(data), NULL_PTR, &hashSize2);
    CHECK_AND_LOG("  C_Digest (get size)", rv == CKR_OK, rvToStr(rv), free_objects2);

    /*************************************************************************
     * Вычислить хэш-код данных                                               *
     *************************************************************************/
    hash2 = (CK_BYTE*)malloc(hashSize2 * sizeof(CK_BYTE));
    CHECK("  Memory allocation for hash", hash2 != NULL, free_objects2);

    rv = functionList->C_Digest(session, data, sizeof(data), hash2, &hashSize2);
    CHECK_AND_LOG("  C_Digest (get hash)", rv == CKR_OK, rvToStr(rv), free_hash2);

    /*************************************************************************
     * Распечатать буфер, содержащий хэш-код                                  *
     *************************************************************************/
    printf("  Hashed buffer is: \n");
    printHex(hash2, hashSize2);
    printf(" Hashing has been completed.\n");

    /*************************************************************************
     * Закодировать хэш-код в объект DigestInfo                               *
     *************************************************************************/
    printf(" Encoding hash...\n");
    EncodeToDigestInfo(sha256Mech.mechanism, hash2, &digestInfoSize2, &digestInfo2);
    CHECK("  DigestInfo encoding", digestInfo2 != NULL, free_hash2);

    /*************************************************************************
     * Распечатать буфер, содержащий объект DigestInfo                        *
     *************************************************************************/
    printf("  Encoded hash is: \n");
    printHex(digestInfo2, digestInfoSize2);
    printf(" Encoding has been completed.\n");

    /*************************************************************************
     * Проверка подписи                                                       *
     *************************************************************************/
    printf(" Verifying data...\n");
    /*************************************************************************
     * Инициализировать операцию проверки подписи                             *
     *************************************************************************/
    rv = functionList->C_VerifyInit(session, &rsaSigVerMech, objects2[0]);
    CHECK_AND_LOG("  C_VerifyInit", rv == CKR_OK, rvToStr(rv), free_digestInfo2);

    /*************************************************************************
     * Проверить подпись для данных                                           *
     *************************************************************************/
    rv = functionList->C_Verify(session, digestInfo2, digestInfoSize2, signature, signatureSize);
    CHECK_AND_LOG("  C_Verify", rv == CKR_OK, rvToStr(rv), free_digestInfo2);

    printf("Verifying has been completed successfully.\n");

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    /*************************************************************************
     * Выполнить действия для завершения работы с библиотекой PKCS#11         *
     *************************************************************************/
    printf("\nFinalizing... \n");

    /*************************************************************************
     * Очистить память, выделенную под хэш-коды, подпись и объекты            *
     *************************************************************************/
free_digestInfo2:
    free(digestInfo2);

free_hash2:
    free(hash2);

free_objects2:
    free(objects2);

free_signature:
    free(signature);

free_digestInfo1:
    free(digestInfo1);

free_hash1:
    free(hash1);

free_objects1:
    free(objects1);

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, выделенную под механизмы и слоты                      *
     *************************************************************************/
free_mechanisms:
    free(mechanisms);

free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}
