/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Данный пример демонстрирует расшифрование упакованного CMS сообщения с *
 * помощью pkicore. Для успешного выполнения программы на токене должен   *
 * находиться сертификат с ключевой парой, для которого было зашифровано  *
 * сообщение. CMS сообщение читается из файла enveloped_data. Для         *
 * создания CMS сообщения можно воспользоваться примером CmsEnvelop.      *
 * ImportCert содержит пример записи сертификата.                         *
 *************************************************************************/

#include <common.h>

using namespace std;
using namespace rutoken::pkicore;

int main() {
    try {
        cout << boolalpha;

        /**********************************************************************
         * Инициализируем pkicore, передав путь до директории с библиотекой    *
         * rtPKCS11ECP.                                                        *
         **********************************************************************/
        rutoken::pkicore::initialize(".");
        SCOPE_EXIT() {
            /**********************************************************************
             * Завершаем работу с pkicore при выходе из текущего блока.            *
             **********************************************************************/
            rutoken::pkicore::deinitialize();
        };

        /**********************************************************************
         * Получаем список подключенных устройств и продолжаем работу с первым *
         * доступным устройством.                                              *
         **********************************************************************/
        auto devices = Pkcs11Device::enumerate();
        if (devices.empty()) {
            throw runtime_error("There must be at least one device connected");
        }

        auto device = move(devices.front());

        /**********************************************************************
         * Получаем список сертификатов на устройстве и продолжаем работу с    *
         * первым найденным.                                                   *
         **********************************************************************/
        auto certs = device.enumerateCerts();
        if (certs.empty()) {
            throw runtime_error("There must be at least one user certificate on device");
        }

        auto cert = move(certs.front());

        /**********************************************************************
         * Аутентифицируемся на устройстве.                                    *
         **********************************************************************/
        device.login("12345678");
        SCOPE_EXIT(&device) {
            /**********************************************************************
             * Сбрасываем права доступа при выходе из текущего блока.              *
             **********************************************************************/
            device.logout();
        };

        /**********************************************************************
         * Читаем файл с зашифрованным сообщением.                             *
         **********************************************************************/
        const auto inputFileName = "enveloped_data";

        cout << "Reading enveloped message from " << inputFileName << endl;

        auto envelopedData = cms::EnvelopedData::parse(readFile(inputFileName));

        /**********************************************************************
         * Задаем параметры и расшифровываем сообщение.                        *
         **********************************************************************/
        cms::EnvelopedData::DecryptParams params(cert);
        auto cmsMessage = envelopedData.decrypt(params);
        if (cmsMessage.contentType() != cms::ContentType::data) {
            throw runtime_error("Unsupported content type of enveloped data");
        }

        /**********************************************************************
         * Выводим расшифрованное сообщение.                                   *
         **********************************************************************/
        auto data = cms::Data::cast(move(cmsMessage)).data();
        string message(data.begin(), data.end());

        cout << "Decrypted message: " << message << endl;
    } catch (const exception& e) {
        cerr << e.what() << endl;
        return 1;
    }

    return 0;
}
