/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2025, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Данный пример демонстрирует создание упакованного CMS сообщения,       *
 * предназначенного для двух получателей, с помощью pkicore. Сертификаты  *
 * получателей в DER формате читаются из файлов cert1.cer и cert2.cer.    *
 * Упакованное сообщение записывается в файл enveloped_data.              *
 *************************************************************************/

#include <common.h>

using namespace std;
using namespace rutoken::pkicore;

int main() {
    try {
        cout << boolalpha;

        /**********************************************************************
         * Инициализируем pkicore, передав путь до директории с библиотекой    *
         * rtPKCS11ECP.                                                        *
         **********************************************************************/
        rutoken::pkicore::initialize(".");
        SCOPE_EXIT() {
            /**********************************************************************
             * Завершаем работу с pkicore при выходе из текущего блока.            *
             **********************************************************************/
            rutoken::pkicore::deinitialize();
        };

        /**********************************************************************
         * Получаем список подключенных устройств и продолжаем работу с первым *
         * доступным устройством.                                              *
         **********************************************************************/
        auto devices = Pkcs11Device::enumerate();
        if (devices.empty()) {
            throw runtime_error("There must be at least one device connected");
        }

        auto device = move(devices.front());

        /**********************************************************************
         * Аутентифицируемся на устройстве.                                    *
         **********************************************************************/
        device.login("12345678");
        SCOPE_EXIT(&device) {
            /**********************************************************************
             * Сбрасываем права доступа при выходе из текущего блока.              *
             **********************************************************************/
            device.logout();
        };

        /**********************************************************************
         * Читаем файлы с сертификатами получателей сообщения в DER формате.   *
         **********************************************************************/
        const auto certFileName1 = "cert1.cer";
        const auto certFileName2 = "cert2.cer";

        cout << "Reading certificate from " << certFileName1 << endl;

        auto certData = readFile(certFileName1);
        ExternalCert cert1(certData.data(), certData.size());

        cout << "Reading certificate from " << certFileName2 << endl;

        certData = readFile(certFileName2);
        ExternalCert cert2(certData.data(), certData.size());

        /**********************************************************************
         * Задаем параметры и шифруем сообщение.                               *
         **********************************************************************/
        string message = "This is a message";
        cout << "Message for enveloping: " << message << endl;

        cms::Data data(vector<uint8_t>(message.begin(), message.end()));
        cms::EnvelopParams params(device);
        params.addRecipient(cert1).addRecipient(cert2);

        auto envelopedData = cms::envelop(data, params);

        /**********************************************************************
         * Пишем зашифрованное сообщение в файл.                               *
         **********************************************************************/
        const auto outFileName = "enveloped_data";

        cout << "Writing enveloped message to " << outFileName << endl;

        auto outBuf = envelopedData.toBer();
        writeFile(outFileName, outBuf.data(), outBuf.size());
    } catch (const exception& e) {
        cerr << e.what() << endl;
        return 1;
    }

    return 0;
}
