#!/usr/bin/env bash

set -eu

# Для запуска скрипта необходимо выставить следующие параметры:
#vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
# Путь до дистрибутива продукта (пакет rutokenlogon)
RUTOKENLOGON_PACKAGE='path/to/rutokenlogon_0.1.0-astra1_arm64.deb'
# Путь, по которому будет сохранен пользовательский дистрибутив
USER_DISTRIBUTIVE_OUPUT_DIR=""
# Тип домена. Возможные значения: ipa, samba, aldpro и ad
DOMAIN_TYPE=""
# Путь до корневого сертификата или цепочки корневых сертификатов
PATH_TO_CA=""
#^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

echo "Checking the environment and params..."

[[ $# -ne 0 ]] && echo "No arguments required" && exit 1

[[ -z "$RUTOKENLOGON_PACKAGE" ]] &&
echo "RUTOKENLOGON_PACKAGE param must be set" && exit 1
[[ -z "$DOMAIN_TYPE" ]] &&
echo "DOMAIN_TYPE param must be set" && exit 1

[[ "$DOMAIN_TYPE" != "ipa" ]] &&
[[ "$DOMAIN_TYPE" != "samba" ]] &&
[[ "$DOMAIN_TYPE" != "aldpro" ]] &&
[[ "$DOMAIN_TYPE" != "ad" ]] &&
echo "Invalid value of the DOMAIN_TYPE param: $DOMAIN_TYPE" && exit 1

[[ -z "$PATH_TO_CA" ]] &&
[[ "$DOMAIN_TYPE" != "ipa" ]] &&
[[ "$DOMAIN_TYPE" != "aldpro" ]] &&
echo "Path to CA certificate must be specified for this domain type: $DOMAIN_TYPE" && exit 1

[[ -n "$PATH_TO_CA" ]] &&
echo "Checking the CA certificate" &&
openssl x509 -in "$PATH_TO_CA" -inform PEM > /dev/null &&
[[ ! -f "$PATH_TO_CA" ]] && echo "CA path is not valid" && exit 1

[[ ! -f "$RUTOKENLOGON_PACKAGE" ]] && echo "Invalid rutokenlogon package path" && exit 1

[[ "$RUTOKENLOGON_PACKAGE" != *.deb ]] &&
[[ "$RUTOKENLOGON_PACKAGE" != *.rpm ]] &&
echo "Invalid rutokenlogon package" && exit 1

[[ -z "$USER_DISTRIBUTIVE_OUPUT_DIR" ]] && USER_DISTRIBUTIVE_OUPUT_DIR=$(pwd)

DISTRIBUTIVE_FOLDER="rutokenlogon_distributive"
DISTRIBUTIVE_PATH="${USER_DISTRIBUTIVE_OUPUT_DIR}/${DISTRIBUTIVE_FOLDER}"

echo "Creating an temporary folder"
mkdir -p "$DISTRIBUTIVE_PATH"
trap "rm -rf ${USER_DISTRIBUTIVE_OUPUT_DIR}/${DISTRIBUTIVE_FOLDER}" EXIT

[[ ! -z "$PATH_TO_CA" ]] &&
echo "Copying the CA certificate" &&
CA_NAME=$(basename "$PATH_TO_CA") &&
cp "$PATH_TO_CA" "${DISTRIBUTIVE_PATH}/${CA_NAME}"

echo "Creating a user script"
USER_SCRIPT_NAME="install_rutokenlogon.sh"
cat > "${DISTRIBUTIVE_PATH}/${USER_SCRIPT_NAME}" << EOF
#!/usr/bin/env bash
set -eu

SCRIPT_DIR=\$(dirname \$BASH_SOURCE)
PACKAGE_NAME="$(basename "$RUTOKENLOGON_PACKAGE")"
RUTOKENLOGON_PACKAGE="${SCRIPT_DIR}/${PACKAGE_NAME}"

if [[ "\$PACKAGE_NAME" == *.deb ]]; then
    apt-get update
    apt-get install -y krb5-pkinit "\$RUTOKENLOGON_PACKAGE"
elif [[ "\$PACKAGE_NAME" == *alt* ]]; then
    apt-get update
    apt-get install -y "\$RUTOKENLOGON_PACKAGE"
else
    dnf install -y krb5-pkinit "\$RUTOKENLOGON_PACKAGE"
fi

rtlogon-cli configure --domain "${DOMAIN_TYPE}" ${CA_NAME:+--ca-cert "\$SCRIPT_DIR/$CA_NAME"}
EOF

chmod +x "${DISTRIBUTIVE_PATH}/${USER_SCRIPT_NAME}"

echo "Copying found rutokenlogon package"
cp "$RUTOKENLOGON_PACKAGE" "$DISTRIBUTIVE_PATH"

pushd "$USER_DISTRIBUTIVE_OUPUT_DIR" > /dev/null
echo "Compressing a temporary folder into an archive"
zip -r "${DISTRIBUTIVE_FOLDER}_deployment.zip" "${DISTRIBUTIVE_FOLDER}"/* > /dev/null
popd > /dev/null

echo "Rtlogon package sucessfully created: ${USER_DISTRIBUTIVE_OUPUT_DIR}/${DISTRIBUTIVE_FOLDER}_deployment.zip"
