<?php
/**
 * This is the contract for implementing CurveFp (EC prime finite-field).
 *
 * @author Matej Danter
 */
interface CurveFpInterface {
        //constructor that sets up the instance variables
        public function  __construct($prime, $a, $b);

        public function contains($x,$y);

        public function getA();

        public function getB();

        public function getPrime();

        public static function cmp(CurveFp $cp1, CurveFp $cp2);

}

/**
 * This is the contract for implementing Point, which encapsulates entities
 * and operations over the points on the Elliptic Curve.
 *
 * @author Matej Danter
 */
interface PointInterface {
    public function __construct(CurveFp $curve, $x, $y, $order = null);

    public static function cmp($p1, $p2);

    public static function add($p1, $p2);

    public static function mul($x2, Point $p1);

    public static function leftmost_bit($x);

    public static function rmul(Point $p1, $m);

    public function __toString();

    public static function double(Point $p1);

    public function getX();

    public function getY();

    public function getCurve();

    public function getOrder();

}

/**
 * This is the contract for the PublicKey portion of ECDSA.
 *
 * @author Matej Danter
 */
interface PublicKeyInterface {
    
    public function __construct(Point $generator, Point $point);

    public function GOST_verifies($hash, Signature $signature);

    public function getCurve();

    public function getGenerator();

    public function getPoint();

}

/**
 * This is the contract for describing a signature used in ECDSA.
 *
 * @author Matej Danter
 */
interface SignatureInterface {
    public function __construct($r, $s);

    public function getR();

    public function getS();
}

/**
 * The gmp extension in PHP does not implement certain necessary operations
 * for elliptic curve encryption
 * This class implements all neccessary static methods
 *
 */
class gmp_Utils {

    public static function gmp_mod2($n, $d) {
        if (extension_loaded('gmp') && USE_EXT=='GMP') {
            $res = gmp_div_r($n, $d);
            if (gmp_cmp(0, $res) > 0) {
                $res = gmp_add($d, $res);
            }
            return gmp_strval($res);
        } else {
            throw new Exception("PLEASE INSTALL GMP");
        }
    }

    public static function gmp_random($n) {
        if (extension_loaded('gmp') && USE_EXT=='GMP') {
            $random = gmp_strval(gmp_random());
            $small_rand = rand();
            while (gmp_cmp($random, $n) > 0) {
                $random = gmp_div($random, $small_rand, GMP_ROUND_ZERO);
            }

            return gmp_strval($random);
        } else {
            throw new Exception("PLEASE INSTALL GMP");
        }
    }

    public static function gmp_hexdec($hex) {
        if (extension_loaded('gmp') && USE_EXT=='GMP') {
            $dec = gmp_strval(gmp_init($hex), 10);

            return $dec;
        } else {
            throw new Exception("PLEASE INSTALL GMP");
        }
    }

    public static function gmp_dechex($dec) {
        if (extension_loaded('gmp') && USE_EXT=='GMP') {
            $hex = gmp_strval(gmp_init($dec), 16);

            return $hex;
        } else {
            throw new Exception("PLEASE INSTALL GMP");
        }
    }

}

/**
 * This class is a representation of an EC over a field modulo a prime number
 *
 * Important objectives for this class are:
 *  - Does the curve contain a point?
 *  - Comparison of two curves.
 */
class CurveFp implements CurveFpInterface {

    //Elliptic curve over the field of integers modulo a prime
    protected $a = 0;
    protected $b = 0;
    protected $prime = 0;

    //constructor that sets up the instance variables
    public function __construct($prime, $a, $b) {
        $this->a = $a;
        $this->b = $b;
        $this->prime = $prime;
    }

    public function contains($x, $y) {
        $eq_zero = null;

        if (extension_loaded('gmp') && USE_EXT=='GMP') {

            $eq_zero = gmp_cmp(gmp_Utils::gmp_mod2(gmp_sub(gmp_pow($y, 2), gmp_add(gmp_add(gmp_pow($x, 3), gmp_mul($this->a, $x)), $this->b)), $this->prime), 0);


            if ($eq_zero == 0) {
                return true;
            } else {
                return false;
            }
        } else {
            throw new ErrorException("Please install GMP");
        }
    }

    public function getA() {
        return $this->a;
    }

    public function getB() {
        return $this->b;
    }

    public function getPrime() {
        return $this->prime;
    }

    public static function cmp(CurveFp $cp1, CurveFp $cp2) {
        $same = null;

        if (extension_loaded('gmp') && USE_EXT=='GMP') {

            if (gmp_cmp($cp1->a, $cp2->a) == 0 && gmp_cmp($cp1->b, $cp2->b) == 0 && gmp_cmp($cp1->prime, $cp2->prime) == 0) {
                return 0;
            } else {
                return 1;
            }

        } else {
            throw new ErrorException("Please install GMP");
        }
    }

}

/**
 * This class encapsulates the GOST recommended curves
 *  
 * @author Eugene Sukhov
 */
class GOSTcurve {

    public static function curve_GOST() {
        #  GOST Curve:
        if (extension_loaded('gmp') && USE_EXT == 'GMP') {
            $_p = gmp_Utils::gmp_hexdec('0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffd97'); 
            $_a = gmp_Utils::gmp_hexdec('0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffd94'); 
            $_b = '166';
            $curve_GOST = new CurveFp($_p, $_a, $_b);
        } else {
			return false;
        }
        return $curve_GOST;
    }	

    public static function generator_GOST() {
        #  GOST Curve:
        if (extension_loaded('gmp') && USE_EXT == 'GMP') {
            $_p = gmp_Utils::gmp_hexdec('0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffd97'); 
            $_a = gmp_Utils::gmp_hexdec('0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffd94'); 
            $_r = gmp_Utils::gmp_hexdec('0xffffffffffffffffffffffffffffffff6c611070995ad10045841b09b761b893');
	    $_b = '166';
            $_Gx ='1';
            $_Gy = gmp_Utils::gmp_hexdec('0x8d91e471e0989cda27df505a453f2b7635294f2ddf23e3b122acc99c9e9f1e14');
            $curve_GOST = new CurveFp($_p, $_a, $_b);
            $generator_GOST = new Point($curve_GOST, $_Gx, $_Gy, $_r);
        } else {
			return false;
        }
        return $generator_GOST;
    }

}

/*
 * This class is where the elliptic curve arithmetic takes place.
 *
 * The important methods are:
 *      - add: adds two points according to ec arithmetic
 *      - double: doubles a point on the ec field mod p
 *      - mul: uses double and add to achieve multiplication
 *
 * The rest of the methods are there for supporting the ones above.
 */

class Point implements PointInterface {

        public $curve;
        public $x;
        public $y;
        public $order;
        public static $infinity = 'infinity';

        public function __construct(CurveFp $curve, $x, $y, $order = null) {
            $this->curve = $curve;
            $this->x = $x;
            $this->y = $y;
            $this->order = $order;


            if (isset($this->curve) && ($this->curve instanceof CurveFp)) {
                if (!$this->curve->contains($this->x, $this->y)) {
                    throw new ErrorException("Curve" . print_r($this->curve, true) . " does not contain point ( " . $x . " , " . $y . " )");
                    
                }

                if ($this->order != null) {

                    if (self::cmp(self::mul($order, $this), self::$infinity) != 0) {
                        throw new ErrorException("SELF * ORDER MUST EQUAL INFINITY.");
                    }
                }
            }
        }

        public static function cmp($p1, $p2) {
            if (extension_loaded('gmp') && USE_EXT=='GMP') {
                if (!($p1 instanceof Point)) {
                    if (($p2 instanceof Point))
                        return 1;
                    if (!($p2 instanceof Point))
                        return 0;
                }

                if (!($p2 instanceof Point)) {
                    if (($p1 instanceof Point))
                        return 1;
                    if (!($p1 instanceof Point))
                        return 0;
                }

                if (gmp_cmp($p1->x, $p2->x) == 0 && gmp_cmp($p1->y, $p2->y) == 0 && CurveFp::cmp($p1->curve, $p2->curve)) {
                    return 0;
                } else {
                    return 1;
                }

            } else {
                throw new ErrorException("Please install GMP");
            }
        }

        public static function add($p1, $p2) {

            if (self::cmp($p2, self::$infinity) == 0 && ($p1 instanceof Point)) {
                return $p1;
            }
            if (self::cmp($p1, self::$infinity) == 0 && ($p2 instanceof Point)) {
                return $p2;
            }

            if (self::cmp($p1, self::$infinity) == 0 && self::cmp($p2, self::$infinity) == 0) {
                return self::$infinity;
            }

            if (extension_loaded('gmp') && USE_EXT=='GMP') {


                if (CurveFp::cmp($p1->curve, $p2->curve) == 0) {
                    if (gmp_Utils::gmp_mod2(gmp_cmp($p1->x, $p2->x), $p1->curve->getPrime()) == 0) {
                        if (gmp_Utils::gmp_mod2(gmp_add($p1->y, $p2->y), $p1->curve->getPrime()) == 0) {
                            return self::$infinity;
                        } else {
                            return self::double($p1);
                        }
                    }

                    $p = $p1->curve->getPrime();

//                    $l = gmp_strval(gmp_mul(gmp_sub($p2->y, $p1->y), NumberTheory::inverse_mod(gmp_sub($p2->x, $p1->x), $p)));
                    $l = gmp_strval(gmp_mul(gmp_sub($p2->y, $p1->y), gmp_strval(gmp_invert(gmp_sub($p2->x, $p1->x), $p))));


                    $x3 = gmp_strval(gmp_Utils::gmp_mod2(gmp_sub(gmp_sub(gmp_pow($l, 2), $p1->x), $p2->x), $p));


                    $y3 = gmp_strval(gmp_Utils::gmp_mod2(gmp_sub(gmp_mul($l, gmp_sub($p1->x, $x3)), $p1->y), $p));


                    $p3 = new Point($p1->curve, $x3, $y3);


                    return $p3;
                } else {
                    throw new ErrorException("The Elliptic Curves do not match.");
                }
            } else {
                throw new ErrorException("Please install GMP");
            }
        }

        public static function mul($x2, Point $p1) {
            if (extension_loaded('gmp') && USE_EXT=='GMP') {
                $e = $x2;

                if (self::cmp($p1, self::$infinity) == 0) {
                    return self::$infinity;
                }
                if ($p1->order != null) {
                    $e = gmp_strval(gmp_Utils::gmp_mod2($e, $p1->order));
                }
                if (gmp_cmp($e, 0) == 0) {

                    return self::$infinity;
                }

                $e = gmp_strval($e);

                if (gmp_cmp($e, 0) > 0) {

                    $e3 = gmp_mul(3, $e);

                    $negative_self = new Point($p1->curve, $p1->x, gmp_strval(gmp_sub(0, $p1->y)), $p1->order);
                    $i = gmp_div(self::leftmost_bit($e3), 2);

                    $result = $p1;

                    while (gmp_cmp($i, 1) > 0) {

                        $result = self::double($result);

                        if (gmp_cmp(gmp_and($e3, $i), 0) != 0 && gmp_cmp(gmp_and($e, $i), 0) == 0) {

                            $result = self::add($result, $p1);
                        }
                        if (gmp_cmp(gmp_and($e3, $i), 0) == 0 && gmp_cmp(gmp_and($e, $i), 0) != 0) {
                            $result = self::add($result, $negative_self);
                        }

                        $i = gmp_strval(gmp_div($i, 2));
                    }
                    return $result;
                }
            } else {
                throw new ErrorException("Please install GMP");
            }
        }

        public static function leftmost_bit($x) {
            if (extension_loaded('gmp') && USE_EXT=='GMP') {
                if (gmp_cmp($x, 0) > 0) {
                    $result = 1;
                    while (gmp_cmp($result, $x) < 0 || gmp_cmp($result, $x) == 0) {
                        $result = gmp_mul(2, $result);
                    }
                    return gmp_strval(gmp_div($result, 2));
                }
            } else {
                throw new ErrorException("Please install GMP");
            }
        }

        public static function rmul(Point $x1, $m) {
            return self::mul($m, $x1);
        }

        public function __toString() {
            if (!($this instanceof Point) && $this == self::$infinity)
                return self::$infinity;
            return "(" . $this->x . "," . $this->y . ")";
        }

        public static function double(Point $p1) {


            if (extension_loaded('gmp') && USE_EXT=='GMP') {

                $p = $p1->curve->getPrime();
                $a = $p1->curve->getA();

//                $inverse = NumberTheory::inverse_mod(gmp_strval(gmp_mul(2, $p1->y)), $p);
                $inverse = gmp_strval(gmp_invert(gmp_strval(gmp_mul(2, $p1->y)), $p));

                $three_x2 = gmp_mul(3, gmp_pow($p1->x, 2));

                $l = gmp_strval(gmp_Utils::gmp_mod2(gmp_mul(gmp_add($three_x2, $a), $inverse), $p));

                $x3 = gmp_strval(gmp_Utils::gmp_mod2(gmp_sub(gmp_pow($l, 2), gmp_mul(2, $p1->x)), $p));

                $y3 = gmp_strval(gmp_Utils::gmp_mod2(gmp_sub(gmp_mul($l, gmp_sub($p1->x, $x3)), $p1->y), $p));

                if (gmp_cmp(0, $y3) > 0)
                    $y3 = gmp_strval(gmp_add($p, $y3));

                $p3 = new Point($p1->curve, $x3, $y3);

                return $p3;
            } else {
                throw new ErrorException("Please install GMP");
            }
        }

        public function getX() {
            return $this->x;
        }

        public function getY() {
            return $this->y;
        }

        public function getCurve() {
            return $this->curve;
        }

        public function getOrder() {
            return $this->order;
        }

    }


/**
 * This class serves as public- private key exchange for signature verification
 */
class PublicKey implements PublicKeyInterface {

    protected $curve;
    protected $generator;
    protected $point;

    public function __construct(Point $generator, Point $point) {
        $this->curve = $generator->getCurve();
        $this->generator = $generator;
        $this->point = $point;

        $n = $generator->getOrder();

        if ($n == null) {
            throw new ErrorExcpetion("Generator Must have order.");
        }
        if (Point::cmp(Point::mul($n, $point), Point::$infinity) != 0) {
            throw new ErrorException("Generator Point order is bad.");
        }

        if (extension_loaded('gmp') && USE_EXT=='GMP') {
            if (gmp_cmp($point->getX(), 0) < 0 || gmp_cmp($n, $point->getX()) <= 0 || gmp_cmp($point->getY(), 0) < 0 || gmp_cmp($n, $point->getY()) <= 0) {
                throw new ErrorException("Generator Point has x and y out of range.");
            }
        } else {
            throw new ErrorException("Please install GMP");
        }
    }

    public function GOST_verifies($hash, Signature $signature) {
        if (extension_loaded('gmp') && USE_EXT=='GMP') {
            $G = $this->generator; //P
            $n = $this->generator->getOrder();//q
            $point = $this->point; //Q
            $r = $signature->getR();
            $s = $signature->getS();

            if (gmp_cmp($r, 1) < 0 || gmp_cmp($r, gmp_sub($n, 1)) > 0) {
                return false;
            }
            if (gmp_cmp($s, 1) < 0 || gmp_cmp($s, gmp_sub($n, 1)) > 0) {
        	return false;
            }
//step 3 GOST
	    $e = gmp_Utils::gmp_mod2($hash, $n);
	    if($e==0)
		    $e=1;
// step 4 GOST
	    $v = gmp_strval(gmp_invert($e,$n));
// step 5 GOST
	    $z1 = gmp_Utils::gmp_mod2(gmp_mul($s, $v), $n);
	    $z2 = gmp_Utils::gmp_mod2(gmp_mul(gmp_neg($r), $v), $n);
// step 6 GOST
            $C = Point::add(Point::mul($z1, $G), Point::mul($z2, $point));
            $R = gmp_Utils::gmp_mod2($C->getX(), $n);

		if (0){
				echo "n - ".$n."\n";
				echo "h - ".$hash."\n";
				echo "e - ".gmp_Utils::gmp_dechex($e)."\n";		
				echo "v - ".gmp_Utils::gmp_dechex($v)."\n";
				echo "r - ".$r."\n";
				echo "s - ".$s."\n";
				echo "z1 - ".gmp_Utils::gmp_dechex($z1)."\nz2 - ".gmp_Utils::gmp_dechex($z2)."\n";
				echo "Q - ".$point."\nG - ".$G."\n";
				echo "C - ".$C."\nR - ".$R."\n";	    
		}

            if (gmp_cmp($R, $r) == 0)
                return true;
            else {
                return false;
            }
        } else {
            throw new ErrorException("Please install GMP");
        }
    }
 

    public function getCurve() {
        return $this->curve;
    }

    public function getGenerator() {
        return $this->generator;
    }

    public function getPoint() {
        return $this->point;
    }

    public function getPublicKey() {
        print_r($this);
        return $this;
    }

}
	
/**
 * Plain Old PHP Object that stores the signature r,s for ECDSA
 *
 * @author Matej Danter
 */

class Signature implements SignatureInterface{

    protected $r;
    protected $s;

    public function  __construct($r, $s) {
        $this->r = $r;
        $this->s = $s;


    }


    public function getR(){
        return $this->r;
    }

    public function getS(){
        return $this->s;
    }
}
	

?>