//  Copyright (c) 2014 Aktiv Co. All rights reserved.

#import "Pkcs11.h"

#define SLOTS_MAX_COUNT 100
@implementation PKCS11

+(long)setActivationPassword:(NSString *)password forReader:(NSString *)reader
{
	CK_FUNCTION_LIST_PTR            pFunctionList       = NULL_PTR;  // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
	CK_FUNCTION_LIST_EXTENDED_PTR   pFunctionListEx     = NULL_PTR;  // Указатель на список функций расширения PKCS#11, хранящийся в структуре CK_FUNCTION_LIST_EXTENDED
    CK_RV                           rv                  = CKR_OK;    // Вспомогательная переменная для хранения кода возврата
    CK_ULONG                        ulSlotCount         = SLOTS_MAX_COUNT;
    CK_SLOT_ID                      slots[SLOTS_MAX_COUNT];
    
    @try {
        /**********************************************************************
         * Шаг 1: Получение структуры с указателями на функции                 *
         *        стандарта PKCS#11.                                           *
         **********************************************************************/
        printf("Getting function list");
        rv = C_GetFunctionList(&pFunctionList);
        if (rv != CKR_OK)
        {
            printf(" -> Failed\n");
			THROW_ERROR(rv);
        }
        printf(" -> OK\n");
        
        /**********************************************************************
         * Шаг 2: Получение структуры с указателями на функции расширения      *
         *        стандарта PKCS#11.                                           *
         **********************************************************************/
        printf("Getting extended function list");
        rv = C_EX_GetFunctionListExtended(&pFunctionListEx);
        if (rv != CKR_OK)
        {
            printf(" -> Failed\n");
            THROW_ERROR(rv);
        }
        printf(" -> OK\n");
        
        /**********************************************************************
         * Шаг 3: Инициализация библиотеки.                                    *
         **********************************************************************/
        printf("Initializing library");
        rv = pFunctionList->C_Initialize(NULL_PTR);
        if (rv != CKR_OK)
        {
            printf(" -> Failed\n");
            THROW_ERROR(rv);
        }
        printf(" -> OK\n");
        
        /**********************************************************************
         * Шаг 4: Получение списка слотов                                     *
         **********************************************************************/
        printf("Getting slots");
        rv = pFunctionList->C_GetSlotList(CK_TRUE, slots, &ulSlotCount);
        if (rv != CKR_OK)
        {
            printf(" -> Failed\n");
            THROW_ERROR(rv);
        }
        printf(" -> OK\n");
        
        for (size_t i = 0 ; i < ulSlotCount; ++i) {
            /**********************************************************************
			 * Шаг 5: Получение информации о токене                                *
			 **********************************************************************/
            printf("Getting tokeninfo\n");
            CK_TOKEN_INFO tokenInfo;
            rv = pFunctionList->C_GetTokenInfo(slots[i], &tokenInfo);
            if (rv != CKR_OK)
            {
                printf(" -> Failed\n");
                continue;
            }
            printf(" -> OK\n");
            
            /**********************************************************************
			 * Шаг 6: Проверка, что сессия не открывается                          *
			 **********************************************************************/
            printf("Try open session\n");
            CK_SESSION_HANDLE hSession;
            rv = pFunctionList->C_OpenSession( slots[i],
                                              CKF_SERIAL_SESSION | CKF_RW_SESSION,
                                              NULL_PTR,
                                              NULL_PTR,
                                              &hSession);
            if(rv == CKR_OK) {
                printf("Token is valid\n");
                rv = pFunctionList->C_CloseSession(hSession);
                continue;
            }
            printf(" -> OK\n");
            
            /**********************************************************************
			 * Шаг 7: Сравнение серийного номера                                  *
			 **********************************************************************/
            int j = 0;
            for(j = 0; j < sizeof(tokenInfo.serialNumber) && tokenInfo.serialNumber[j] != ' '; ++j);
            if(j < sizeof(tokenInfo.serialNumber))
                tokenInfo.serialNumber[j] = '\0';
            else
                continue; // rutoken serial is less than 16 charecters
            
            NSString* hexSerial = [NSString stringWithCString:tokenInfo.serialNumber encoding:NSUTF8StringEncoding];
            size_t serialInt;
            NSScanner *scanner = [NSScanner scannerWithString:hexSerial];
            [scanner scanHexInt:&serialInt];
            NSString* serial = [NSString stringWithFormat:@"%zd", serialInt];
            
            // checking if it is the same rutoken
            if([reader rangeOfString:serial].location != NSNotFound) {
				/**********************************************************************
				 * Шаг 8: Установка пароля активации                                  *
				 **********************************************************************/
                printf("Setting activation password");
                rv = pFunctionListEx->C_EX_SetActivationPassword(slots[i], (CK_UTF8CHAR_PTR)[password cStringUsingEncoding:NSUTF8StringEncoding]);
                if (rv != CKR_OK)
                {
                    printf(" -> Failed\n");
					[self logData: @"Failed to set activation password"];
                    continue;
                }
                printf(" -> OK\n");
            }
            
            /**********************************************************************
			 * Шаг 9: Проверка, что сессия открывается                            *
			 **********************************************************************/
            printf("Check session now can be opened\n");
            rv = pFunctionList->C_OpenSession( slots[i],
                                              CKF_SERIAL_SESSION | CKF_RW_SESSION,
                                              NULL_PTR,
                                              NULL_PTR,
                                              &hSession);
            if(rv != CKR_OK) {
                printf("Failed to open session\n");
                printf("Setting activation password seems to have failed\n");
				[self logData: @"Fail to open session"];
                continue;
            }
            printf(" -> OK\n");
            pFunctionList->C_CloseSession(hSession);
        }
    } @catch(ApplError* e) {};
    
    if (pFunctionList)
    {
        /**********************************************************************
         * Деинициализировать библиотеку                                  *
         **********************************************************************/
        int rvTemp = CKR_OK;
        printf(" Finalizing library");
        rvTemp = pFunctionList->C_Finalize(NULL_PTR);
        if (rvTemp != CKR_OK)
            printf(" -> Failed\n");
        else
            printf(" -> OK\n");
        pFunctionList = NULL_PTR;
    }
    return rv;
}
@end
