//  Copyright (c) 2014 Aktiv Co. All rights reserved.

#import "Pkcs11.h"


static NSString* removeTrailingSpaces(const char* string, size_t length) {
	size_t i;
	for (i = length; i != 0; --i) {
		if (' ' != string[i - 1]) break;
	}

	return [[NSString alloc] initWithBytes:string length:i encoding:NSUTF8StringEncoding];
}

@implementation DataLogger
- (void)logData:(NSString*)data {
    NSDictionary* dict = [NSDictionary dictionaryWithObject:data forKey:@"data"];
    [[NSNotificationCenter defaultCenter] postNotificationName:@"LogData" object:self
                                                      userInfo:dict];
}
@end

@implementation Pkcs11Error

- (id)initWithCode:(CK_ULONG)code {
	return [super initWithDomain:@"pkcs11" code:code userInfo:nil];
}

+ (id)errorWithCode:(CK_ULONG)code {
	return [[Pkcs11Error alloc] initWithCode:code];
}

@end

@implementation Token

- (id)initWithId:(CK_SLOT_ID)id functions:(CK_FUNCTION_LIST_PTR)functions extendedFunctions:(CK_FUNCTION_LIST_EXTENDED_PTR)extendedFunctions {
	self = [super init];

	if (self) {
		_functions = functions;
		_extendedFunctions = extendedFunctions;
		_id = id;
        @try {
            NSDictionary* dict = [self getTokenInfo];
            _serialNumber = [dict valueForKey:@"serialNumber"];
        } @catch(Pkcs11Error* e) {
            _serialNumber = @"Unknown";
        }
    }
    
	return self;
}

- (NSDictionary*) getTokenInfo {
    CK_TOKEN_INFO tokenInfo;
    
    CK_RV rv = _functions->C_GetTokenInfo(_id, &tokenInfo);
    if (CKR_OK != rv) THROW_ERROR(rv);
    
    NSString* serialNumber = removeTrailingSpaces((char*)tokenInfo.serialNumber, sizeof(tokenInfo.serialNumber));
    NSString* hardwareVersion =[NSString stringWithFormat:@"%u.%u.%u.%u",
                               tokenInfo.hardwareVersion.major, tokenInfo.hardwareVersion.minor,
                               tokenInfo.firmwareVersion.major, tokenInfo.firmwareVersion.minor];
    NSMutableDictionary* dict = [NSMutableDictionary dictionaryWithCapacity:4];
    [dict setObject: serialNumber forKey:@"serialNumber"];
    [dict setObject: removeTrailingSpaces((char*)tokenInfo.label, sizeof(tokenInfo.label)) forKey:@"label"];
    [dict setObject: removeTrailingSpaces((char*)tokenInfo.model, sizeof(tokenInfo.model)) forKey:@"model"];
    [dict setObject: hardwareVersion forKey:@"hardwareVersion"];
    
    return dict;
}

- (NSDictionary*) getTokenInfoEx {
    CK_TOKEN_INFO_EXTENDED tokenInfoEx;
    tokenInfoEx.ulSizeofThisStructure = sizeof(CK_TOKEN_INFO_EXTENDED);
    
    CK_RV rv = _extendedFunctions->C_EX_GetTokenInfoExtended(_id, &tokenInfoEx);
    if (CKR_OK != rv) THROW_ERROR(rv);
    
    NSString* microcode = [NSString stringWithFormat:@"%lu",tokenInfoEx.ulMicrocodeNumber];
    NSString* voltage = [NSString stringWithFormat:@"%lu",tokenInfoEx.ulBatteryVoltage];
    NSMutableDictionary* dict = [NSMutableDictionary dictionaryWithCapacity:2];
    [dict setObject:microcode forKey:@"microcode"];
    [dict setObject: voltage forKey:@"voltage"];
    
    return dict;
}

@end


@implementation Pkcs11

- (id)init {
	self = [super init];

	if (self) {
		CK_ULONG rv = C_GetFunctionList(&_functions);
		if (CKR_OK != rv) THROW_ERROR(rv);

		rv = C_EX_GetFunctionListExtended(&_extendedFunctions);
		if (CKR_OK != rv) THROW_ERROR(rv);
        
        rv = C_GetFunctionList(&_functions);
        if (CKR_OK != rv) THROW_ERROR(rv);
        
        rv = _functions->C_Initialize(NULL_PTR);
		if (CKR_OK != rv) THROW_ERROR(rv);

		_tokens = [NSMutableDictionary dictionary];
		_serialSlotMap = [NSMutableDictionary dictionary];
	}

	return self;
}

- (void)dealloc {
	_functions->C_Finalize(NULL_PTR);
}

- (void)updateTokenList {
    CK_ULONG slotCount;
    CK_RV rv = _functions->C_GetSlotList(CK_FALSE, NULL_PTR, &slotCount);
    if (CKR_OK != rv) THROW_ERROR(rv);
    
    NSMutableData* slotIdData = [NSMutableData dataWithLength:slotCount * sizeof(CK_SLOT_ID)];
    const CK_SLOT_ID* slotIds = [slotIdData bytes];
    rv = _functions->C_GetSlotList(CK_TRUE, [slotIdData mutableBytes], &slotCount);
    if (CKR_OK != rv) THROW_ERROR(rv);
    
    NSMutableArray* slotIdsArray = [NSMutableArray arrayWithCapacity:slotCount];
    
    for (size_t i = 0; i != slotCount; ++i) [slotIdsArray addObject:[NSNumber numberWithInteger: slotIds[i]]];
    
    for (NSNumber* slotId in [_tokens allKeys]) {
        if(![slotIdsArray containsObject:slotId]) {
            Token* token = [_tokens objectForKey:slotId];
            [_serialSlotMap removeObjectForKey:[token serialNumber]];
            [_tokens removeObjectForKey:slotId];
        }
    }
    
    for (NSNumber* slotId in slotIdsArray) {
        CK_ULONG id = [slotId unsignedLongValue];
        Token* token = [[Token alloc] initWithId:id functions:_functions extendedFunctions:_extendedFunctions];
        if ([[_tokens allKeys] containsObject:slotId] && ![[[_tokens objectForKey:slotId] serialNumber] isEqualToString:[token serialNumber]]) {
            [_serialSlotMap removeObjectForKey:[token serialNumber]];
            [_tokens removeObjectForKey:slotId];
        }
        if(![[_tokens allKeys] containsObject:slotId]) {
            [_tokens setObject:token forKey:slotId];
            [_serialSlotMap setObject:slotId forKey:[token serialNumber]];
        }
    }
}

- (BOOL)hasToken {
	return 0 != [_tokens count];
}

- (NSArray*)serialNumbers {
	NSMutableArray* r = [NSMutableArray arrayWithCapacity:[_tokens count]];
	for (Token* t in [_tokens allValues]) [r addObject:[t serialNumber]];
	return r;
}

- (Token*)tokenForSerialNumber:(NSString*)serialNumber {
	return [_tokens objectForKey:[_serialSlotMap objectForKey:serialNumber]];
}

@end
